% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idf.R
\name{Idf}
\alias{Idf}
\title{Read, Modify, and Run an EnergyPlus Model}
\description{
eplusr provides parsing EnergyPlus Input Data File (IDF) files and strings
in a hierarchical structure, which was extremely inspired by
\href{https://openstudio-sdk-documentation.s3.amazonaws.com/cpp/OpenStudio-2.5.1-doc/utilities/html/idf_page.html}{OpenStudio utilities library},
but with total different data structure under the hook.
}
\details{
eplusr uses \code{Idf} class to present the whole IDF file and use \link{IdfObject}
to present a single object in IDF. Both \code{Idf} and \link{IdfObject} contain member
functions for helping modify the data in IDF so it complies with the
underlying IDD (EnergyPlus Input Data Dictionary).

Under the hook, eplusr uses a SQL-like structure to store both IDF and IDD
data in different \link[data.table:data.table]{data.table::data.table}s. So to modify an EnergyPlus model
in eplusr is equal to change the data in those IDF tables accordingly, in the
context of specific IDD data. This means that a corresponding \link{Idd} object is
needed whenever creating an \code{Idf} object. eplusr provides several
\link[=use_idd]{helpers} to easily download IDD files and create \link{Idd} objects.

All IDF reading process starts with function \code{\link[=read_idf]{read_idf()}} which returns an
\code{Idf} object. \code{Idf} class provides lots of methods to programmatically query
and modify EnergyPlus models.

Internally, the powerful \href{https://cran.r-project.org/package=data.table}{data.table}
package is used to speed up the whole IDF parsing process and store the
results. Under the hook, eplusr uses a SQL-like structure to store both IDF
and IDD data in \link[data.table:data.table]{data.table::data.table} format. Every IDF will be parsed and
stored in three tables:
\itemize{
\item \code{object}: contains object IDs, names and comments.
\item \code{value}: contains field values
\item \code{reference}: contains cross-reference data of field values.
}
}
\examples{

## ------------------------------------------------
## Method `Idf$new`
## ------------------------------------------------

\dontrun{
# example model shipped with eplusr from EnergyPlus v8.8
path_idf <- system.file("extdata/1ZoneUncontrolled.idf", package = "eplusr") # v8.8

# If neither EnergyPlus v8.8 nor Idd v8.8 was found, error will
# occur. If Idd v8.8 is found, it will be used automatically.
idf <- Idf$new(path_idf)

# argument `idd` can be specified explicitly using `use_idd()`
idf <- Idf$new(path_idf, idd = use_idd(8.8))

# you can set `download` arugment to "auto" in `use_idd()` if you
# want to automatically download corresponding IDD file when
# necessary
idf <- Idf$new(path_idf, use_idd(8.8, download = "auto"))

# Besides use a path to an IDF file, you can also provide IDF in literal
# string format
string_idf <-
    "
    Version, 8.8;
    Building,
        Building;                !- Name
    "

Idf$new(string_idf, use_idd(8.8, download = "auto"))
}


## ------------------------------------------------
## Method `Idf$version`
## ------------------------------------------------

\dontrun{
# get version
idf$version()
}


## ------------------------------------------------
## Method `Idf$path`
## ------------------------------------------------

\dontrun{
# get path
idf$path()

# return `NULL` if Idf is not created from a file
Idf$new("Version, 8.8;\n")$path()
}


## ------------------------------------------------
## Method `Idf$group_name`
## ------------------------------------------------

\dontrun{
# get names of all groups Idf contains
idf$group_name()

# get group name of each object in Idf
idf$group_name(sorted = FALSE)

# get names of all available groups in underlying Idd
idf$group_name(all = TRUE)
}


## ------------------------------------------------
## Method `Idf$class_name`
## ------------------------------------------------

\dontrun{
# get names of all classes in Idf
idf$class_name()

# get names of all classes grouped by group names in Idf
idf$class_name(by_group = TRUE)

# get class name of each object in Idf
idf$class_name(sorted = FALSE)

# get names of all available classes in underlying Idd
idf$class_name(all = TRUE)

# get names of all available classes grouped by group names in
# underlying Idd
idf$class_name(all = TRUE, by_group = TRUE)
}


## ------------------------------------------------
## Method `Idf$is_valid_group`
## ------------------------------------------------

\dontrun{
# check if input is a valid group name in current Idf
idf$is_valid_group(c("Schedules", "Compliance Objects"))

# check if input is a valid group name in underlying Idd
idf$is_valid_group(c("Schedules", "Compliance Objects"), all = TRUE)
}


## ------------------------------------------------
## Method `Idf$is_valid_class`
## ------------------------------------------------

\dontrun{
# check if input is a valid class name in current Idf
idf$is_valid_class(c("Building", "ShadowCalculation"))

# check if input is a valid class name in underlying Idd
idf$is_valid_class(c("Building", "ShadowCalculation"), all = TRUE)
}


## ------------------------------------------------
## Method `Idf$definition`
## ------------------------------------------------

\dontrun{
# get the IddObject object for specified class
idf$definition("Version")
}


## ------------------------------------------------
## Method `Idf$object_id`
## ------------------------------------------------

\dontrun{
# get IDs of all objects in current Idf object
idf$object_id()

# get IDs of all objects in current Idf object, and merge them into a
# single integer vector
idf$object_id(simplify = TRUE)

# get IDs of objects in class Version and Zone
idf$object_id(c("Version", "Zone"))

# get IDs of objects in class Version and Zone, and merge them into a
# single integer vector
idf$object_id(c("Version", "Zone"), simplify = TRUE)
}


## ------------------------------------------------
## Method `Idf$object_name`
## ------------------------------------------------

\dontrun{
# get names of all objects in current Idf object
idf$object_name()

# get names of all objects in current Idf object, and merge them into
# a single character vector
idf$object_name(simplify = TRUE)

# get names of objects in class Version and Zone
idf$object_name(c("Version", "Zone"))

# get names of objects in class Version and Zone, and merge them into
# a single character vector
idf$object_name(c("Version", "Zone"), simplify = TRUE)
}


## ------------------------------------------------
## Method `Idf$object_num`
## ------------------------------------------------

\dontrun{
# get total number of objects
idf$object_num()

# get number of objects in class Zone and Schedule:Compact
idf$object_num(c("Zone", "Schedule:Compact"))
}


## ------------------------------------------------
## Method `Idf$is_valid_id`
## ------------------------------------------------

\dontrun{
idf$is_valid_id(c(51, 1000))
}


## ------------------------------------------------
## Method `Idf$is_valid_name`
## ------------------------------------------------

\dontrun{
idf$is_valid_name(c("Simple One Zone (Wireframe DXF)", "ZONE ONE", "a"))

# name matching is case-insensitive
idf$is_valid_name(c("simple one zone (wireframe dxf)", "zone one", "a"))
}


## ------------------------------------------------
## Method `Idf$object`
## ------------------------------------------------

\dontrun{
# get an object whose ID is 3
idf$object(3)

# get an object whose name is "simple one zone (wireframe dxf)"
# NOTE: object name matching is case-insensitive
idf$object("simple one zone (wireframe dxf)")
}


## ------------------------------------------------
## Method `Idf$objects`
## ------------------------------------------------

\dontrun{
# get objects whose IDs are 3 and 10
idf$objects(c(3,10))

# get objects whose names are "Simple One Zone (Wireframe DXF)" and "ZONE ONE"
# NOTE: object name matching is case-insensitive
idf$objects(c("Simple One Zone (Wireframe DXF)", "zone one"))
}


## ------------------------------------------------
## Method `Idf$object_unique`
## ------------------------------------------------

\dontrun{
# get the SimulationColtrol object
idf$object_unique("SimulationControl")

# S3 "[[" and "$" can also be used
idf$SimulationControl
idf[["SimulationControl"]]
}


## ------------------------------------------------
## Method `Idf$objects_in_class`
## ------------------------------------------------

\dontrun{
# get all objects in Zone class
idf$objects_in_class("Zone")

# S3 "[[" and "$" can also be used
idf$Zone
idf[["Zone"]]
}


## ------------------------------------------------
## Method `Idf$objects_in_group`
## ------------------------------------------------

\dontrun{
# get all objects in Schedules group
idf$objects_in_group("Schedules")
}


## ------------------------------------------------
## Method `Idf$object_relation`
## ------------------------------------------------

\dontrun{
# check each layer's reference of a construction named FLOOR
idf$object_relation("floor", "ref_to")

# check where is this construction being used
idf$object_relation("floor", "ref_by")
}


## ------------------------------------------------
## Method `Idf$objects_in_relation`
## ------------------------------------------------

\dontrun{
# get a construction named FLOOR and all materials it uses
idf$objects_in_relation("floor", "ref_to")

# get a construction named FLOOR and all surfaces that uses it
idf$objects_in_relation("floor", "ref_by", class = "BuildingSurface:Detailed")
}


## ------------------------------------------------
## Method `Idf$search_object`
## ------------------------------------------------

\dontrun{
# get all objects whose names contains "floor"
idf$search_object("floor", ignore.case = TRUE)
}


## ------------------------------------------------
## Method `Idf$dup`
## ------------------------------------------------

\dontrun{
# duplicate an object named "FLOOR"
idf$dup("floor") # New object name 'FLOOR_1' is auto-generated

# duplicate that object again by specifing object ID
idf$dup(16) # New object name 'FLOOR_2' is auto-generated

# duplicate that object two times and giving new names
idf$dup(new_floor = "floor", new_floor2 = 16)

# duplicate that object multiple times using variable inputs
floors_1 <- c(new_floor3 = "floor", new_floor4 = "floor")
floors_2 <- setNames(rep(16, 5), paste0("flr", 1:5))
idf$dup(floors_1, floors_2)
}


## ------------------------------------------------
## Method `Idf$add`
## ------------------------------------------------

\dontrun{
# add a new Building object with all default values
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = .())

# add a new Building object with all default values and comments
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = .(.comment = c("this is", "a new building")))

# add a new RunPeriod object with all possible fields
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = list(), RunPeriod = list("rp", 1, 1, 1, 31), .all = TRUE)

# add objects using variable inputs
empty <- empty_idf(8.8) # create an empty Idf
objs1 <- list(Schedule_Constant = list("const"), Building = list())
rp <- list(RunPeriod = list("rp", 2, 1, 2, 28))
empty$add(objs1, rp)
}


## ------------------------------------------------
## Method `Idf$set`
## ------------------------------------------------

\dontrun{
# modify an object by name (case-insensitive)
idf$set(r13layer = list(roughness = "smooth"))

# modify an object by ID
idf$set(..12 = list(roughness = "rough"))

# overwrite existing object comments
idf$set(r13layer = list(.comment = c("New comment")))

# assign default values to fields
idf$set(r13layer = list(solar_absorptance = NULL), .default = TRUE)

# set field values to blanks
idf$set(r13layer = list(solar_absorptance = NULL), .default = FALSE)

# set field values to blank and delete trailing fields
idf$set(r13layer = list(visible_absorptance = NULL), .default = FALSE)

# set field values to blank and keep blank fields
idf$set(r13layer = list(visible_absorptance = NULL), .default = FALSE, .empty = TRUE)

# set all fields in one class
idf$set(Material_NoMass := list(visible_absorptance = 0.9))

# set multiple objects in one class
idf$set(.("r13layer", "r31layer") := list(solar_absorptance = 0.8))
# above is equivalent to
idf$set(r13layer = list(solar_absorptance = 0.8),
        r31layer = list(solar_absorptance = 0.8)
)

# use variable input
sets <- list(r13layer = list(roughness = "smooth"))
idf$set(sets)
}


## ------------------------------------------------
## Method `Idf$del`
## ------------------------------------------------

\dontrun{
# delete objects using names
idf$object("Fraction") # ScheduleTypeLimits
idf$del("Fraction")

# delete objects using IDs
idf$objects(c(39, 40)) # Output:Variable
idf$del(39, 40)

# cannot delete objects that are referred by others
level_checks()$reference # reference-checking is enable by default
idf$del("r13layer") # error

# force to delete objects even thay are referred by others
idf$del("r13layer", .force = TRUE)

# delete objects and also objects that refer to them
idf$del("r31layer", .ref_by = TRUE) # Construction 'ROOF31' will be kept

# delete objects and also objects that they refer to
idf$del("extlights", .ref_to = TRUE) # Schedule 'AlwaysOn' will be kept

# delete objects and also other objects that refer to them recursively
idf$del("roof31", .ref_by = TRUE, .recursive = TRUE)

# delete objects using variable inputs
ids <- idf$object_id("Output:Variable", simplify = TRUE)
idf$del(ids)
}


## ------------------------------------------------
## Method `Idf$purge`
## ------------------------------------------------

\dontrun{
# purge unused "Fraction" schedule type
idf$purge("on/off") # ScheduleTypeLimits

# purge all unused schedule types
idf$purge(class = "ScheduleTypeLimits")

# purge all unused schedule related objects
idf$purge(group = "Schedules")
}


## ------------------------------------------------
## Method `Idf$duplicated`
## ------------------------------------------------

\dontrun{
# check if there are any duplications in the Idf
idf$duplicated(class = "ScheduleTypeLimits")

# check if there are any duplications in the schedule types
idf$duplicated(class = "ScheduleTypeLimits")

# check if there are any duplications in the schedule groups and
# material class
idf$duplicated(class = "Material", group = "Schedules")
}


## ------------------------------------------------
## Method `Idf$unique`
## ------------------------------------------------

\dontrun{
# remove duplications in the Idf
idf$unique(class = "ScheduleTypeLimits")

# remove duplications in the schedule types
idf$unique(class = "ScheduleTypeLimits")

# remove duplications in the schedule groups and material class
idf$unique(class = "Material", group = "Schedules")
}


## ------------------------------------------------
## Method `Idf$rename`
## ------------------------------------------------

\dontrun{
idf$objects(c("on/off", "test 352a"))
idf$rename(on_off = "on/off", test_352a = 51)
}


## ------------------------------------------------
## Method `Idf$insert`
## ------------------------------------------------

\dontrun{
# insert all material from another IDF
path_idf2 <- file.path(eplus_config(8.8)$dir, "ExampleFiles/5ZoneTDV.idf")
idf2 <- Idf$new(path_idf2)
idf$insert(idf2$Material)

# insert objects from same Idf is equivalent to using Idf$dup()
idf$insert(idf$SizingPeriod_DesignDay)
}


## ------------------------------------------------
## Method `Idf$load`
## ------------------------------------------------

\dontrun{
# load objects from character vectors
idf$load(
    c("Material,",
      "    mat,                     !- Name",
      "    MediumSmooth,            !- Roughness",
      "    0.667,                   !- Thickness {m}",
      "    0.115,                   !- Conductivity {W/m-K}",
      "    513,                     !- Density {kg/m3}",
      "    1381;                    !- Specific Heat {J/kg-K}"),

    "Construction, const, mat;"
)

# load objects from data.frame definitions
dt <- idf$to_table(class = "Material")
dt[field == "Name", value := paste(value, 1)]
dt[field == "Thickness", value := "0.5"]
idf$load(dt)

# by default, duplications are removed
idf$load(idf$to_table(class = "Material"))

# keep empty fields as they are
idf$load("Material, mat1, smooth, 0.5, 0.2, 500, 1000,,, 0.5;", .default = FALSE)

# keep trailing empty fields
idf$load("Material, mat2, smooth, 0.5, 0.2, 500, 1000,,,;",
    .default = FALSE, .empty = TRUE
)
}


## ------------------------------------------------
## Method `Idf$update`
## ------------------------------------------------

\dontrun{
# update objects from string definitions:
str <- idf$to_string("zone one", header = FALSE, format = "new_top")
str[8] <- "2," # Multiplier
idf$update(str)

# update objects from data.frame definitions:
dt <- idf$to_table("zone one")
dt[field == "Multiplier", value := "1"]
idf$update(dt)
}


## ------------------------------------------------
## Method `Idf$search_value`
## ------------------------------------------------

\dontrun{
# search values that contains "floor"
idf$search_value("floor", ignore.case = TRUE)

# search values that contains "floor" in class Construction
idf$search_value("floor", "Construction", ignore.case = TRUE)
}


## ------------------------------------------------
## Method `Idf$replace_value`
## ------------------------------------------------

\dontrun{
# search values that contains "win" and replace them with "windows"
idf$replace_value("win", "windows")
}


## ------------------------------------------------
## Method `Idf$validate`
## ------------------------------------------------

\dontrun{
idf$validate()

# check at predefined validate level
idf$validate("none")
idf$validate("draft")
idf$validate("final")

# custom validate checking components
idf$validate(custom_validate(auto_field = TRUE, choice = TRUE))
}


## ------------------------------------------------
## Method `Idf$is_valid`
## ------------------------------------------------

\dontrun{
idf$is_valid()

# check at predefined validate level
idf$is_valid("none")
idf$is_valid("draft")
idf$is_valid("final")

# custom validate checking components
idf$is_valid(custom_validate(auto_field = TRUE, choice = TRUE))
}


## ------------------------------------------------
## Method `Idf$to_string`
## ------------------------------------------------

\dontrun{
# get text format of the whole Idf
head(idf$to_string())

# get text format of the whole Idf, excluding the header and all comments
head(idf$to_string(comment = FALSE, header = FALSE))

# get text format of all objects in class Material
head(idf$to_string(class = "Material", comment = FALSE, header = FALSE))

# get text format of some objects
head(idf$to_string(c("floor", "zone one")))

# tweak output formatting
head(idf$to_string("floor", leading = 0, sep_at = 0))
}


## ------------------------------------------------
## Method `Idf$to_table`
## ------------------------------------------------

\dontrun{
# extract whole Idf data
idf$to_table()

# extract all data from class Material
idf$to_table(class = "Material")

# extract multiple object data
idf$to_table(c("FLOOR", "ZONE ONE"))

# keep value types and put actual values into a list column
idf$to_table(c("FLOOR", "ZONE ONE"), string_value = FALSE)$value

# add the unit to each value
idf$to_table(c("FLOOR", "ZONE ONE"), string_value = FALSE, unit = TRUE)

# get all possible fields
idf$to_table("ZONE ONE", all = TRUE)

# make sure all objects in same class have the same number of fields
idf$to_table(class = "Construction", align = TRUE)

# get a wide table with string values
idf$to_table(class = "Construction", wide = TRUE)

# get a wide table with actual values
idf$to_table(class = "OtherEquipment", wide = TRUE, string_value = FALSE)

# group extensible by extensible group number
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "group")

# group extensible by extensible group number and convert into a wide table
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "group", wide = TRUE)

# group extensible by extensible field index
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index")

# group extensible by extensible field index and convert into a wide table
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index", wide = TRUE)

# when grouping extensible, 'string_value' and 'unit' still take effect
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index",
    wide = TRUE, string_value = FALSE, unit = TRUE
)

# create table for new object input
idf$to_table(class = "BuildingSurface:Detailed", init = TRUE)
}


## ------------------------------------------------
## Method `Idf$external_deps`
## ------------------------------------------------

\dontrun{
idf$external_deps()
}


## ------------------------------------------------
## Method `Idf$is_unsaved`
## ------------------------------------------------

\dontrun{
idf$is_unsaved()
}


## ------------------------------------------------
## Method `Idf$save`
## ------------------------------------------------

\dontrun{
# save Idf as a new file
idf$save(tempfile(fileext = ".idf"))

# save and overwrite current file
idf$save(overwrite = TRUE)

# save the model with newly created and modified objects at the top
idf$save(overwrite = TRUE, format = "new_top")

# save the model to a new file and copy all external csv files used in
# "Schedule:File" class into the same folder
idf$save(path = file.path(tempdir(), "test1.idf"), copy_external = TRUE)
}


## ------------------------------------------------
## Method `Idf$run`
## ------------------------------------------------

\dontrun{
idf <- Idf$new(path_idf)
# save the model to tempdir()
idf$save(file.path(tempdir(), "test_run.idf"))

# use the first epw file in "WeatherData" folder in EnergyPlus v8.8
# installation path
epw <- list.files(file.path(eplus_config(8.8)$dir, "WeatherData"),
    pattern = "\\\\.epw$", full.names = TRUE)[1]

# if `dir` is NULL, the directory of IDF file will be used as simulation
# output directory
job <- idf$run(epw, dir = NULL)

# run simulation in the background
idf$run(epw, dir = tempdir(), wait = FALSE)

# copy all external files into the directory run simulation
idf$run(epw, dir = tempdir(), copy_external = TRUE)

# check for simulation errors
job$errors()

# get simulation status
job$status()

# get output directory
job$output_dir()

# re-run the simulation
job$run()

# get simulation results
job$report_data()
}


## ------------------------------------------------
## Method `Idf$last_job`
## ------------------------------------------------

\dontrun{
idf$last_job()
}


## ------------------------------------------------
## Method `Idf$geometry`
## ------------------------------------------------

\dontrun{
idf$geometry()
}


## ------------------------------------------------
## Method `Idf$view`
## ------------------------------------------------

\dontrun{
idf$view()
idf$view(render_by = "zone")
idf$view(render_by = "construction")
}


## ------------------------------------------------
## Method `Idf$print`
## ------------------------------------------------

\dontrun{
idf$print("group")
idf$print("class")
idf$print("object")
idf$print("field")

# order objects by there classes
idf$print("object", order = FALSE)
idf$print("field", order = FALSE)
}

}
\seealso{
\link{IdfObject} class for a single object in an IDF.
}
\author{
Hongyuan Jia
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Idf-new}{\code{Idf$new()}}
\item \href{#method-Idf-version}{\code{Idf$version()}}
\item \href{#method-Idf-path}{\code{Idf$path()}}
\item \href{#method-Idf-group_name}{\code{Idf$group_name()}}
\item \href{#method-Idf-class_name}{\code{Idf$class_name()}}
\item \href{#method-Idf-is_valid_group}{\code{Idf$is_valid_group()}}
\item \href{#method-Idf-is_valid_class}{\code{Idf$is_valid_class()}}
\item \href{#method-Idf-definition}{\code{Idf$definition()}}
\item \href{#method-Idf-object_id}{\code{Idf$object_id()}}
\item \href{#method-Idf-object_name}{\code{Idf$object_name()}}
\item \href{#method-Idf-object_num}{\code{Idf$object_num()}}
\item \href{#method-Idf-is_valid_id}{\code{Idf$is_valid_id()}}
\item \href{#method-Idf-is_valid_name}{\code{Idf$is_valid_name()}}
\item \href{#method-Idf-object}{\code{Idf$object()}}
\item \href{#method-Idf-objects}{\code{Idf$objects()}}
\item \href{#method-Idf-object_unique}{\code{Idf$object_unique()}}
\item \href{#method-Idf-objects_in_class}{\code{Idf$objects_in_class()}}
\item \href{#method-Idf-objects_in_group}{\code{Idf$objects_in_group()}}
\item \href{#method-Idf-object_relation}{\code{Idf$object_relation()}}
\item \href{#method-Idf-objects_in_relation}{\code{Idf$objects_in_relation()}}
\item \href{#method-Idf-search_object}{\code{Idf$search_object()}}
\item \href{#method-Idf-dup}{\code{Idf$dup()}}
\item \href{#method-Idf-add}{\code{Idf$add()}}
\item \href{#method-Idf-set}{\code{Idf$set()}}
\item \href{#method-Idf-del}{\code{Idf$del()}}
\item \href{#method-Idf-purge}{\code{Idf$purge()}}
\item \href{#method-Idf-duplicated}{\code{Idf$duplicated()}}
\item \href{#method-Idf-unique}{\code{Idf$unique()}}
\item \href{#method-Idf-rename}{\code{Idf$rename()}}
\item \href{#method-Idf-insert}{\code{Idf$insert()}}
\item \href{#method-Idf-load}{\code{Idf$load()}}
\item \href{#method-Idf-update}{\code{Idf$update()}}
\item \href{#method-Idf-paste}{\code{Idf$paste()}}
\item \href{#method-Idf-search_value}{\code{Idf$search_value()}}
\item \href{#method-Idf-replace_value}{\code{Idf$replace_value()}}
\item \href{#method-Idf-validate}{\code{Idf$validate()}}
\item \href{#method-Idf-is_valid}{\code{Idf$is_valid()}}
\item \href{#method-Idf-to_string}{\code{Idf$to_string()}}
\item \href{#method-Idf-to_table}{\code{Idf$to_table()}}
\item \href{#method-Idf-external_deps}{\code{Idf$external_deps()}}
\item \href{#method-Idf-is_unsaved}{\code{Idf$is_unsaved()}}
\item \href{#method-Idf-save}{\code{Idf$save()}}
\item \href{#method-Idf-run}{\code{Idf$run()}}
\item \href{#method-Idf-last_job}{\code{Idf$last_job()}}
\item \href{#method-Idf-geometry}{\code{Idf$geometry()}}
\item \href{#method-Idf-view}{\code{Idf$view()}}
\item \href{#method-Idf-print}{\code{Idf$print()}}
\item \href{#method-Idf-clone}{\code{Idf$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-new"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-new}{}}}
\subsection{Method \code{new()}}{
Create an \code{Idf} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$new(path, idd = NULL, encoding = "unknown")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{path}}{Either a path, a connection, or literal data (either a single
string or a raw vector) to an EnergyPlus Input Data File
(IDF). If a file path, that file usually has a extension
\code{.idf}.}

\item{\code{idd}}{Any acceptable input of \code{\link[=use_idd]{use_idd()}}. If \code{NULL}, which is the
default, the version of IDF will be passed to \code{\link[=use_idd]{use_idd()}}. If
the input is an \code{.ddy} file which does not have a version
field, the latest version of \link{Idf} cached will be used.}

\item{\code{encoding}}{The file encoding of input IDF. Should be one of
\code{"unknown"}, \verb{"Latin-1" and }"UTF-8"\verb{. The default is }"unknown"` which means that the file is encoded in the native
encoding.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
It takes an EnergyPlus Input Data File (IDF) as input and returns an
\code{Idf} object.

Currently, Imf file is not fully supported. All EpMacro lines will be treated
as normal comments of the nearest downwards object. If input is an Imf file,
a warning will be given during parsing. It is recommended to convert the Imf
file to an Idf file and use \link{ParametricJob} class to conduct
parametric analysis.
}

\subsection{Returns}{
An \code{Idf} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# example model shipped with eplusr from EnergyPlus v8.8
path_idf <- system.file("extdata/1ZoneUncontrolled.idf", package = "eplusr") # v8.8

# If neither EnergyPlus v8.8 nor Idd v8.8 was found, error will
# occur. If Idd v8.8 is found, it will be used automatically.
idf <- Idf$new(path_idf)

# argument `idd` can be specified explicitly using `use_idd()`
idf <- Idf$new(path_idf, idd = use_idd(8.8))

# you can set `download` arugment to "auto" in `use_idd()` if you
# want to automatically download corresponding IDD file when
# necessary
idf <- Idf$new(path_idf, use_idd(8.8, download = "auto"))

# Besides use a path to an IDF file, you can also provide IDF in literal
# string format
string_idf <-
    "
    Version, 8.8;
    Building,
        Building;                !- Name
    "

Idf$new(string_idf, use_idd(8.8, download = "auto"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-version"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-version}{}}}
\subsection{Method \code{version()}}{
Get the version of current \code{Idf}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$version()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$version()} returns the version of current \code{Idf} in a
\code{\link[base:numeric_version]{base::numeric_version()}} format. This makes it easy to direction
compare versions of different \code{Idf}s, e.g. \code{idf$version() > 8.6} or
\code{idf1$version() > idf2$version()}.
}

\subsection{Returns}{
A \code{\link[base:numeric_version]{base::numeric_version()}} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get version
idf$version()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-path"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-path}{}}}
\subsection{Method \code{path()}}{
Get the file path of current \code{Idf}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$path()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$path()} returns the full path of current \code{Idf} or \code{NULL} if the
\code{Idf} object is created using a character vector and not saved
locally.
}

\subsection{Returns}{
\code{NULL} or a single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get path
idf$path()

# return `NULL` if Idf is not created from a file
Idf$new("Version, 8.8;\n")$path()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-group_name"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-group_name}{}}}
\subsection{Method \code{group_name()}}{
Get names of groups
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$group_name(all = FALSE, sorted = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{all}}{If \code{FALSE}, only names of groups in current \code{Idf} object
will be returned. If \code{TRUE}, all group names in the underlying
\link{Idd} will be returned. Default: \code{FALSE}.}

\item{\code{sorted}}{Only applicable when \code{all} is \code{FALSE}. If \code{TRUE},
duplications in returned group or class names are removed, and
unique names are further sorted according to their occurrences
in the underlying \link{Idd}. Default: \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$group_name()} returns names of groups current \code{Idf} contains or
the underlying \link{Idd} object contains.
}

\subsection{Returns}{
A character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get names of all groups Idf contains
idf$group_name()

# get group name of each object in Idf
idf$group_name(sorted = FALSE)

# get names of all available groups in underlying Idd
idf$group_name(all = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-class_name"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-class_name}{}}}
\subsection{Method \code{class_name()}}{
Get names of classes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$class_name(all = FALSE, sorted = TRUE, by_group = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{all}}{If \code{FALSE}, only names of classes in current \code{Idf} object
will be returned. If \code{TRUE}, all class names in the underlying
\link{Idd} will be returned. Default: \code{FALSE}.}

\item{\code{sorted}}{Only applicable when \code{all} is \code{FALSE}. If \code{TRUE},
duplications in returned group or class names are removed, and
unique names are further sorted according to their occurrences
in the underlying \link{Idd}. Default: \code{TRUE}.}

\item{\code{by_group}}{Only applicable when \code{all} or \code{sorted} is \code{TRUE}. If
\code{TRUE}, a list is returned which separates class names by the
group they belong to.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$class_name()} returns names of classes current \code{Idf} contains or
the underlying \link{Idd} object contains.
}

\subsection{Returns}{
A character vector if \code{by_group} is \code{FALSE} and a list of
character vectors when \code{by_group} is \code{TRUE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get names of all classes in Idf
idf$class_name()

# get names of all classes grouped by group names in Idf
idf$class_name(by_group = TRUE)

# get class name of each object in Idf
idf$class_name(sorted = FALSE)

# get names of all available classes in underlying Idd
idf$class_name(all = TRUE)

# get names of all available classes grouped by group names in
# underlying Idd
idf$class_name(all = TRUE, by_group = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_valid_group"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_valid_group}{}}}
\subsection{Method \code{is_valid_group()}}{
Check if elements in input character vector are valid group names.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_valid_group(group, all = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{group}}{A character vector to check.}

\item{\code{all}}{If \code{FALSE}, check if input characters are valid group names
for current \code{Idf}. If \code{TRUE}, check if input characters are
valid group names for underlying \link{Idd}. Default: FALSE}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$is_valid_group()} returns \code{TRUE}s if given character vector
contains valid group names in the context of current \code{Idf} (when
\code{all} is \code{FALSE}) or current underlying \link{Idd} (when \code{all} is \code{TRUE}).

Note that case-sensitive matching is performed, which means that
\code{"Location and Climate"} is a valid group name but \code{"location and climate"} is not.
}

\subsection{Returns}{
A logical vector with the same length as input character
vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# check if input is a valid group name in current Idf
idf$is_valid_group(c("Schedules", "Compliance Objects"))

# check if input is a valid group name in underlying Idd
idf$is_valid_group(c("Schedules", "Compliance Objects"), all = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_valid_class"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_valid_class}{}}}
\subsection{Method \code{is_valid_class()}}{
Check if elements in input character vector are valid class names.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_valid_class(class, all = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A character vector to check.}

\item{\code{all}}{If \code{FALSE}, check if input characters are valid class names
for current \code{Idf}. If \code{TRUE}, check if input characters are
valid class names for underlying \link{Idd}. Default: FALSE}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$is_valid_class()} returns \code{TRUE}s if given character vector
contains valid class names in the context of current \code{Idf} (when
\code{all} is \code{FALSE}) or current underlying \link{Idd} (when \code{all} is \code{TRUE}),
and \code{FALSE}s otherwise.

Note that case-sensitive matching is performed, which means that
\code{"Version"} is a valid class name but \code{"version"} is not.
}

\subsection{Returns}{
A logical vector with the same length as input character
vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# check if input is a valid class name in current Idf
idf$is_valid_class(c("Building", "ShadowCalculation"))

# check if input is a valid class name in underlying Idd
idf$is_valid_class(c("Building", "ShadowCalculation"), all = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-definition"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-definition}{}}}
\subsection{Method \code{definition()}}{
Get the \link{IddObject} object for specified class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$definition(class = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A \strong{single} string of valid class name in current
\link{Idd}. If \code{NULL}, the underlying \link{Idd} object is returned.
Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$definition()} returns an \link{IddObject} of given class. \link{IddObject}
contains all data used for parsing and creating an \link{IdfObject}. For
details, please see \link{IddObject} class.
}

\subsection{Returns}{
An \link{IddObject} object if class is not \code{NULL} or an \link{Idd}
object if class is \code{NULL}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get the IddObject object for specified class
idf$definition("Version")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object_id"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object_id}{}}}
\subsection{Method \code{object_id()}}{
Get the unique ID for each object in specified classes in the \code{Idf}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object_id(class = NULL, simplify = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A character vector that contains valid class names for
current \code{Idf} object. If \code{NULL}, all classes in current \code{Idf}
object are used. Default: \code{NULL}.}

\item{\code{simplify}}{If \code{TRUE}, an integer vector contains object IDs of
all specified classes is returned. If \code{FALSE}, a named list
that contains object IDs for each specified class is returned.
Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
In \code{Idf}, each object is assigned with an integer as an universally
unique identifier (UUID) in the context of current \code{Idf}. UUID is
not reused even if the object associated is deleted.

\verb{$object_id()} returns an integer vector (when \code{simplify} is \code{TRUE})
or a named list (when \code{simplify} is \code{FALSE}) of integer vectors that
contain object IDs in each specified class. The returned list is
named using specified class names.
}

\subsection{Returns}{
An integer vector (when \code{simplify} is \code{TRUE}) or a named list
of integer vectors (when \code{simplify} is \code{FALSE}).
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get IDs of all objects in current Idf object
idf$object_id()

# get IDs of all objects in current Idf object, and merge them into a
# single integer vector
idf$object_id(simplify = TRUE)

# get IDs of objects in class Version and Zone
idf$object_id(c("Version", "Zone"))

# get IDs of objects in class Version and Zone, and merge them into a
# single integer vector
idf$object_id(c("Version", "Zone"), simplify = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object_name"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object_name}{}}}
\subsection{Method \code{object_name()}}{
Get names for objects in specified classes in the \code{Idf}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object_name(class = NULL, simplify = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A character vector that contains valid class names for
current \code{Idf}. If \code{NULL}, all classes in current \code{Idf} are
used. Default: \code{NULL}.}

\item{\code{simplify}}{If \code{TRUE}, a character vector contains object names
of all specified classes is returned. If \code{FALSE}, a named list
that contains a character vector for each specified class is
returned. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
In \code{Idf}, each object is assigned with a single string as the name
for it, if the class it belongs to has name attribute, e.g. class
\code{RunPeriod}, \code{Material} and etc. That name should be unique among all
objects in that class. EnergyPlus will fail with an error if
duplications are found among object names in a class.

\verb{$object_name()} returns a character vector (when \code{simplify} is
\code{TRUE}) or a named list (when \code{simplify} is \code{FALSE}) of character
vectors that contain object IDs in each specified class. The returned
list is named using specified class names. If specified class does
not have name attribute, \code{NA}s are returned.
}

\subsection{Returns}{
A character vector (when \code{simplify} is \code{TRUE}) or a named
list of character vectors (when \code{simplify} is \code{FALSE}).
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get names of all objects in current Idf object
idf$object_name()

# get names of all objects in current Idf object, and merge them into
# a single character vector
idf$object_name(simplify = TRUE)

# get names of objects in class Version and Zone
idf$object_name(c("Version", "Zone"))

# get names of objects in class Version and Zone, and merge them into
# a single character vector
idf$object_name(c("Version", "Zone"), simplify = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object_num"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object_num}{}}}
\subsection{Method \code{object_num()}}{
Get number of objects in specified classes in the \link{Idf} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object_num(class = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A character vector that contains valid class names for
underlying \link{Idd}. If \code{NULL}, all classes in current \code{Idf} are
used, and the total object number is returned. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$object_num()} returns an integer vector of object number in
specified classes. \code{0} is returned if there is no object in that
class.
}

\subsection{Returns}{
An integer vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get total number of objects
idf$object_num()

# get number of objects in class Zone and Schedule:Compact
idf$object_num(c("Zone", "Schedule:Compact"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_valid_id"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_valid_id}{}}}
\subsection{Method \code{is_valid_id()}}{
Check if elements in input integer vector are valid object IDs.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_valid_id(id, class = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{An integer vector to check.}

\item{\code{class}}{A single string indicates the class where the objects to
check against. If \code{NULL}, all classes in current \code{Idf} are
used. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$is_valid_id()} returns \code{TRUE}s if given integer vector
contains valid object IDs in current \code{Idf} object.
}

\subsection{Returns}{
A logical vector with the same length as input integer
vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$is_valid_id(c(51, 1000))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_valid_name"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_valid_name}{}}}
\subsection{Method \code{is_valid_name()}}{
Check if elements in input character vector are valid object names.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_valid_name(name, class = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A character vector to check.}

\item{\code{class}}{A single string indicates the class where the objects to
check against. If \code{NULL}, all classes in current \code{Idf} are
used. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$is_valid_name()} returns \code{TRUE}s if given character vector
contains valid object names in current \code{Idf} object.

Note that \strong{case-insensitive} matching is performed, which means
that \code{"rOoF"} is equivalent to \code{"roof"}. This behavior is consistent
in all methods that take object name(s) as input.
}

\subsection{Returns}{
A logical vector with the same length as input character
vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$is_valid_name(c("Simple One Zone (Wireframe DXF)", "ZONE ONE", "a"))

# name matching is case-insensitive
idf$is_valid_name(c("simple one zone (wireframe dxf)", "zone one", "a"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object}{}}}
\subsection{Method \code{object()}}{
Extract an \link{IdfObject} object using object ID or name.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object(which, class = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{A single integer specifying the object ID or a single
string specifying the object name.}

\item{\code{class}}{A character vector that contains valid class names for
current \code{Idf} object used to locate objects. If \code{NULL}, all
classes in current \code{Idf} object are used. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$object()} returns an \link{IdfObject} object specified by an object ID
or name.

Note that unlike object ID, which is always unique across the whole
\code{Idf} object, different objects can have the same name. If the name
given matches multiple objects, an error is issued showing what
objects are matched by the same name. This behavior is consistent in
all methods that take object name(s) as input. In this case, it is
suggested to directly use object ID instead of name.

Note that \strong{case-insensitive} matching is performed for object
names, which means that \code{"rOoF"} is equivalent to \code{"roof"}. This
behavior is consistent in all methods that take object name(s) as
input.
}

\subsection{Returns}{
An \link{IdfObject} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get an object whose ID is 3
idf$object(3)

# get an object whose name is "simple one zone (wireframe dxf)"
# NOTE: object name matching is case-insensitive
idf$object("simple one zone (wireframe dxf)")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-objects"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-objects}{}}}
\subsection{Method \code{objects()}}{
Extract multiple \link{IdfObject} objects using object IDs or names.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$objects(which)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{An integer vector specifying object IDs or a character
vector specifying object names.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$objects()} returns a named list of \link{IdfObject} objects using object
IDS or names. The returned list is named using object names.

Note that unlike object ID, which is always unique across the whole
\code{Idf} object, different objects can have the same name. If the name
given matches multiple objects, an error is issued showing what
objects are matched by the same name. This behavior is consistent in
all methods that take object name(s) as input. In this case, it is
suggested to directly use object ID instead of name.

Note that \strong{case-insensitive} matching is performed for object
names, which means that \code{"rOoF"} is equivalent to \code{"roof"}. This
behavior is consistent in all methods that take object name(s) as
input.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get objects whose IDs are 3 and 10
idf$objects(c(3,10))

# get objects whose names are "Simple One Zone (Wireframe DXF)" and "ZONE ONE"
# NOTE: object name matching is case-insensitive
idf$objects(c("Simple One Zone (Wireframe DXF)", "zone one"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object_unique"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object_unique}{}}}
\subsection{Method \code{object_unique()}}{
Extract the \link{IdfObject} in class with \code{unique-object} attribute.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object_unique(class)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A single string of valid class name for current \code{Idf}
object.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
For each version of an \code{Idf} object, the corresponding underlying
\link{Idd} describe how many objects can be defined in each class. Classes
that have \code{unique-object} attribute can only hold a single object,
e.g. \code{Version}, \code{SimulationControl} and etc. \verb{$object_unique()} can
be used to directly return the \link{IdfObject} in one \code{unique-object}
class. An error will be issued if there are multiple objects in that
class or input class is not an \code{unique-object} class. This makes sure
that \verb{$object_unique()} always returns a single \link{IdfObject}.

\code{Idf} class also provides custom S3 method of \code{$} and \code{[[} to
make it more convenient to get the \link{IdfObject} in \code{unique-object}
class. Basically, \code{idf$ClassName} and \code{idf[["ClassName"]]},
where \code{ClassName} is a single valid class name, is equivalent to
\code{idf$object_unique(ClassName)} if \code{ClassName} is an \code{unique-object}
class. For convenience, underscore-style names are allowed when using
\code{$}, e.g.  \code{Site_Location} is equivalent to \code{Site:Location}. For
instance, \code{idf$Site_Location} and also \code{idf[["Site_Location"]]} will
both return the \link{IdfObject}s in \code{Site:Location} class. Note that
unlike \verb{$object_unique()}, \code{idf$ClassName} and \code{idf[["ClassName"]]}
will directly return \code{NULL} instead of giving an error when
\code{ClassName} is not a valid class name in current \code{Idf} object. This
makes it possible to use \code{is.null(idf$ClassName)} to check if
\code{ClassName} is a valid class or not.
}

\subsection{Returns}{
An \link{IdfObject} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get the SimulationColtrol object
idf$object_unique("SimulationControl")

# S3 "[[" and "$" can also be used
idf$SimulationControl
idf[["SimulationControl"]]
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-objects_in_class"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-objects_in_class}{}}}
\subsection{Method \code{objects_in_class()}}{
Extract all \link{IdfObject} objects in one class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$objects_in_class(class)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A single string of valid class name for current \code{Idf}
object.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$objects_in_class()} returns a named list of all \link{IdfObject} objects
in specified class. The returned list is named using object names.

\code{Idf} class also provides custom S3 method of \code{$} and \code{[[} to
make it more convenient to get all \link{IdfObject} objects in one class.
Basically, \code{idf$ClassName} and \code{idf[["ClassName"]]}, where
\code{ClassName} is a single valid class name, is equivalent to
\code{idf$objects_in_class(ClassName)} if \code{ClassName} is not an
\code{unique-object} class. For convenience, \emph{underscore-style} names are
allowed, e.g.  \code{BuildingSurface_Detailed} is equivalent to
\code{BuildingSurface:Detailed} when using \code{$}. For instance,
\code{idf$BuildingSurface_Detailed} and also
\code{idf[["BuildingSurface:Detailed"]]} will both return all \link{IdfObject}
objects in \code{BuildingSurface:Detailed} class. Note that
unlike \verb{$objects_in_class()}, \code{idf$ClassName} and
\code{idf[["ClassName"]]} will directly return \code{NULL} instead of giving
an error when \code{ClassName} is not a valid class name in current \code{Idf}
object. This makes it possible to use \code{is.null(idf$ClassName)} to
check if \code{ClassName} is a valid class or not.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get all objects in Zone class
idf$objects_in_class("Zone")

# S3 "[[" and "$" can also be used
idf$Zone
idf[["Zone"]]
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-objects_in_group"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-objects_in_group}{}}}
\subsection{Method \code{objects_in_group()}}{
Extract all \link{IdfObject} objects in one group.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$objects_in_group(group)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{group}}{A single string of valid group name for current \code{Idf}
object.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$objects_in_group()} returns a named list of all \link{IdfObject} objects
in specified group. The returned list is named using object names.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get all objects in Schedules group
idf$objects_in_group("Schedules")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-object_relation"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-object_relation}{}}}
\subsection{Method \code{object_relation()}}{
Extract the relationship between object field values.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$object_relation(
  which,
  direction = c("all", "ref_to", "ref_by", "node"),
  object = NULL,
  class = NULL,
  group = NULL,
  depth = 0L,
  keep = FALSE,
  class_ref = c("both", "none", "all")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{A single integer specifying object ID or a single string
specifying object name.}

\item{\code{direction}}{The relation direction to extract. Should be either
\code{"all"}, \code{"ref_to"}, \code{"ref_by"} and \code{"node"}.}

\item{\code{object}}{A character vector of object names or an integer vector
of object IDs used for searching relations. Default: \code{NULL}.}

\item{\code{class}}{A character vector of class names used for searching
relations. Default: \code{NULL}.}

\item{\code{group}}{A character vector of group names used for searching
relations. Default: \code{NULL}.}

\item{\code{depth}}{If > 0, the relation is searched recursively. A
simple example of recursive reference: one material named
\code{mat} is referred by a construction named \code{const}, and \code{const}
is also referred by a surface named \code{surf}. If \code{NULL},
all possible recursive relations are returned. Default: \code{0}.}

\item{\code{keep}}{If \code{TRUE}, all fields of specified object are returned
regardless they have any relations with other objects or not.
If \code{FALSE}, only fields in specified object that have
relations with other objects are returned. Default: \code{FALSE}.}

\item{\code{class_ref}}{Specify how to handle class-name-references. Class
name references refer to references in like field \verb{Component 1 Object Type} in \code{Branch} objects. Their value refers to other
many class names of objects, instaed of referring to specific
field values. There are 3 options in total, i.e. \code{"none"},
\code{"both"} and \code{"all"}, with \code{"both"} being the default.
* \code{"none"}: just ignore class-name-references. It is a reasonable
option, as for most cases, class-name-references always come
along with field value references. Ignoring
class-name-references will not impact the most part of the
relation structure.
* \code{"both"}: only include class-name-references if this object
also reference field values of the same one. For example, if the
value of field \verb{Component 1 Object Type} is
\code{Coil:Heating:Water}, only the object that is referenced in the
next field \verb{Component 1 Name} is treated as referenced by
\verb{Component 1 Object Type}. This is the default option.
* \code{"all"}: include all class-name-references. For example, if the
value of field \verb{Component 1 Object Type} is
\code{Coil:Heating:Water}, all objects in \code{Coil:Heating:Water} will
be treated as referenced by that field. This is the most
aggressive option.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Many fields in \link{Idd} can be referred by others. For example, the
\verb{Outside Layer} and other fields in \code{Construction} class refer to the
\code{Name} field in \code{Material} class and other material related classes.
Here it means that the \verb{Outside Layer} field \strong{refers to} the \code{Name}
field and the \code{Name} field is \strong{referred by} the \verb{Outside Layer}. In
EnergyPlus, there is also a special type of field called \code{Node},
which together with \code{Branch}, \code{BranchList} and other classes define
the topography of the HVAC connections. A outlet node of a component
can be referred by another component as its inlet node, but can also
exists independently, such as zone air node.

\verb{$object_relation()} provides a simple interface to get this kind of
relation. It takes a single object ID or name and also a relation
direction, and returns an \code{IdfRelation} object which contains data
presenting such relation above. For instance, if
\code{model$object_relation("WALL-1", "ref_to")} gives results below:

\if{html}{\out{<div class="sourceCode">}}\preformatted{-- Refer to Others ------------------------
  Class: <Construction>
  \\- Object [ID:2] <WALL-1>
     \\- 2: "WD01";        !- Outside Layer
        v~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        \\- Class: <Material>
           \\- Object [ID:1] <WD01>
              \\- 1: "WD01";        !- Name
}\if{html}{\out{</div>}}

This means that the value \code{"WD01"} of \verb{Outside Layer} in a
construction named \code{WALL-1} refers to a material named \code{WD01}. All
those objects can be further easily extracted using
\verb{$objects_in_relation()} method described below.
}

\subsection{Returns}{
An \code{IdfRelation} object, which is a list of 3
\code{\link[data.table:data.table]{data.table::data.table()}}s named \code{ref_to}, \code{ref_by} and \code{node}.
Each \code{\link[data.table:data.table]{data.table::data.table()}} contains 24 columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# check each layer's reference of a construction named FLOOR
idf$object_relation("floor", "ref_to")

# check where is this construction being used
idf$object_relation("floor", "ref_by")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-objects_in_relation"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-objects_in_relation}{}}}
\subsection{Method \code{objects_in_relation()}}{
Extract multiple \link{IdfObject} objects referencing each others.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$objects_in_relation(
  which,
  direction = c("ref_to", "ref_by", "node"),
  object = NULL,
  class = NULL,
  group = NULL,
  depth = 0L,
  class_ref = c("both", "none", "all")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{A single integer specifying object ID or a single string
specifying object name.}

\item{\code{direction}}{The relation direction to extract. Should be one of
\code{"ref_to"}, \code{"ref_by"} or \code{"node"}.}

\item{\code{object}}{A character vector of object names or an integer vector
of object IDs used for searching relations. Default: \code{NULL}.}

\item{\code{class}}{A character vector of valid class names in the
underlying \link{Idd}. It is used to restrict the classes to be
returned. If \code{NULL}, all possible classes are considered and
corresponding \link{IdfObject} objects are returned if
relationships are found. Default: \code{NULL}.}

\item{\code{group}}{A character vector of valid group names in the
underlying \link{Idd}. It is used to restrict the groups to be
returned. If \code{NULL}, all possible groups are considered and
corresponding \link{IdfObject} objects are returned if
relationships are found. Default: \code{NULL}.}

\item{\code{depth}}{If > 0, the relation is searched recursively. A
simple example of recursive reference: one material named
\code{mat} is referred by a construction named \code{const}, and \code{const}
is also referred by a surface named \code{surf}. If \code{NULL},
all possible recursive relations are returned. Default: \code{0}.}

\item{\code{class_ref}}{Specify how to handle class-name-references. Class
name references refer to references in like field \verb{Component 1 Object Type} in \code{Branch} objects. Their value refers to other
many class names of objects, instaed of refering to specific
field values. There are 3 options in total, i.e. \code{"none"},
\code{"both"} and \code{"all"}, with \code{"both"} being the default.
* \code{"none"}: just ignore class-name-references. It is a reasonable
option, as for most cases, class-name-references always come
along with field value references. Ignoring
class-name-references will not impact the most part of the
relation structure.
* \code{"both"}: only include class-name-references if this object
also reference field values of the same one. For example, if the
value of field \verb{Component 1 Object Type} is
\code{Coil:Heating:Water}, only the object that is referenced in the
next field \verb{Component 1 Name} is treated as referenced by
\verb{Component 1 Object Type}. This is the default option.
* \code{"all"}: include all class-name-references. For example, if the
value of field \verb{Component 1 Object Type} is
\code{Coil:Heating:Water}, all objects in \code{Coil:Heating:Water} will
be treated as referenced by that field. This is the most
aggressive option.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$objects_in_relation()} returns a named list of \link{IdfObject} objects
that have specified relationship with given object. The first element of
returned list is always the specified object itself. If that
object does not have specified relationship with other objects in
specified \code{class}, a list that only contains specified object itself
is returned.

For instance, assuming that \code{const} is a valid object name in
\code{Construction} class, \code{idf$objects_in_relation("const", "ref_by", "BuildingSurface:Detailed")}
will return a named list of an \link{IdfObject} object named \code{const} and
also all other \link{IdfObject} objects in \code{BuildingSurface:Detailed}
class that refer to field values in \code{const}. Similarly,
\code{idf$objects_in_relation("const", "ref_to", "Material")}
will return a named list of an \link{IdfObject} object named \code{const} and
also all other \link{IdfObject} objects in \code{Material} class that \code{const}
refers to. This makes it easy to directly extract groups of related
objects and then use \verb{$insert()} method or other methods
described below to insert them or extract data.

There are lots of recursive references in a model. For instance, a
material can be referred by a construction, that construction can be
referred by a building surface, and that building surface can be
referred by a window on that surface. These objects related
recursively can be extracted by setting \code{recursive} to \code{TRUE}.
}

\subsection{Returns}{
An named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get a construction named FLOOR and all materials it uses
idf$objects_in_relation("floor", "ref_to")

# get a construction named FLOOR and all surfaces that uses it
idf$objects_in_relation("floor", "ref_by", class = "BuildingSurface:Detailed")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-search_object"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-search_object}{}}}
\subsection{Method \code{search_object()}}{
Extract multiple \link{IdfObject} objects using regular expression on
names.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$search_object(
  pattern,
  class = NULL,
  ignore.case = FALSE,
  perl = FALSE,
  fixed = FALSE,
  useBytes = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{pattern, ignore.case, perl, fixed, useBytes}}{All are directly
passed to \link[base:grep]{base::grepl}.}

\item{\code{class}}{A character vector of valid class names in the
underlying \link{Idd}. It is used to restrict the classes to be
returned. If \code{NULL}, all possible classes are considered and
corresponding \link{IdfObject} objects are returned if
\code{pattern} is met Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$search_object()} returns a named list of \link{IdfObject} objects whose
names meet the given regular expression in specified classes.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get all objects whose names contains "floor"
idf$search_object("floor", ignore.case = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-dup"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-dup}{}}}
\subsection{Method \code{dup()}}{
Duplicate existing objects.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$dup(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Integer vectors of object IDs and character vectors of
object names. If input is named, its name will be used as the
name of newly created objects.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$dup()} takes integer vectors of object IDs and character vectors of
object names, duplicates objects specified, and returns a list of
newly created \link{IdfObject} objects. The names of input are used as new
names for created \link{IdfObject}s. If input is not named, new names are
the names of duplicated objects with a suffix \code{"_1"}, \code{"_2"} and etc,
depending on how many times that object has been duplicated. Note an
error will be issued if trying to assign a new name to an object
which belongs to a class that does not have name attribute.

Assigning newly added objects with an existing name in current \code{Idf}
object is prohibited if current validation level includes object name
conflicting checking. For details, please see \code{level_checks()}.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# duplicate an object named "FLOOR"
idf$dup("floor") # New object name 'FLOOR_1' is auto-generated

# duplicate that object again by specifing object ID
idf$dup(16) # New object name 'FLOOR_2' is auto-generated

# duplicate that object two times and giving new names
idf$dup(new_floor = "floor", new_floor2 = 16)

# duplicate that object multiple times using variable inputs
floors_1 <- c(new_floor3 = "floor", new_floor4 = "floor")
floors_2 <- setNames(rep(16, 5), paste0("flr", 1:5))
idf$dup(floors_1, floors_2)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-add"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-add}{}}}
\subsection{Method \code{add()}}{
Add new objects.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$add(..., .default = TRUE, .all = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Lists of object definitions. Each list should be named
with a valid class name. There is a special element \code{.comment}
in each list, which will be used as the comments of newly
added object.}

\item{\code{.default}}{If \code{TRUE}, default values are used for those blank
fields if possible. If \code{FALSE}, empty fields are kept blank.
Default: \code{TRUE}.}

\item{\code{.all}}{If \code{TRUE}, all fields are added. If \code{FALSE}, only minimum
required fields are added. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$add()} takes new object definitions in list format, adds
corresponding objects in specified classes, returns a list of newly
added \link{IdfObject} objects. The returned list will be named using
newly added object names. Every list should be named using a valid
class name.  Underscore-style class name is allowed for class name.
Names in each list element are treated as field names. Values without
names will be inserted according to their position. There is a
special element named \code{.comment} in each list, which will be used as
the comments of newly added object.

Empty objects can be added using an empty list, e.g.
\code{idf$add(Building = list())}. All empty fields will be filled with
corresponding default value if \code{.default} is \code{TRUE}, leaving other
fields as blanks. However, adding blank objects may not be allowed if
there are required fields in that class and current validate level
includes missing-required-field checking. For what kind of validation
components will be performed during adding new objects, please see
\code{\link[=level_checks]{level_checks()}}.

Note that \code{.()} can be used as an alias as \code{list()}, e.g.
\code{idf$add(Building = .())} is equivalent to
\code{idf$add(Building = list())}.

Field name matching is \strong{case-insensitive}. For convenience,
underscore-style field names are also allowed, e.g. \code{eNd_MoNtH} is
equivalent to \verb{End Month}. This behavior is consistent among all
methods that take field names as input.

There is no need to give all field values if only specific fields are
interested, as long as other fields are not required. For example, to
define a new object in \code{RunPeriod} class, the following is enough (at
least for EnergyPlus v8.8):

\if{html}{\out{<div class="sourceCode">}}\preformatted{idf$add(
    RunPeriod = list(
        "my run period",
        begin_month = 1, begin_day_of_month = 1,
        end_month = 1, end_day_of_month = 31
    ),
    .default = TRUE
)
}\if{html}{\out{</div>}}

If not all field names are given, positions of those values without
field names are determined after those values with names. E.g. in
\code{idf$add(Construction = list("out_layer", name = "name"))},
\code{"out_layer"} will be treated as the value for field \verb{Outside Layer}
in \code{Construction} class, since the value for field \code{Name} has been
specified using explicit field name.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# add a new Building object with all default values
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = .())

# add a new Building object with all default values and comments
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = .(.comment = c("this is", "a new building")))

# add a new RunPeriod object with all possible fields
empty <- empty_idf(8.8) # create an empty Idf
empty$add(Building = list(), RunPeriod = list("rp", 1, 1, 1, 31), .all = TRUE)

# add objects using variable inputs
empty <- empty_idf(8.8) # create an empty Idf
objs1 <- list(Schedule_Constant = list("const"), Building = list())
rp <- list(RunPeriod = list("rp", 2, 1, 2, 28))
empty$add(objs1, rp)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-set"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-set}{}}}
\subsection{Method \code{set()}}{
Set values of existing objects.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$set(..., .default = TRUE, .empty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Lists of object definitions. Each list should be named
with a valid object name or ID denoted in style \code{..ID}. There
is a special element \code{.comment} in each list, which will be
used as new comments of modified object, overwriting existing
comments if any.}

\item{\code{.default}}{If \code{TRUE}, default values are used for those blank
fields if possible. If \code{FALSE}, empty fields are kept blank.
Default: \code{TRUE}.}

\item{\code{.empty}}{If \code{TRUE}, trailing empty fields are kept. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$set()} takes new field value definitions in list format, sets new
values for fields in objects specified, and returns a list of
modified \link{IdfObject}s. The returned list will be named using names of
modified objects. Every list in \verb{$set()} should be named with a
valid object name. Object ID can also be used but have to be combined
with prevailing two periods \code{..}, e.g. \code{..10} indicates the object
with ID \code{10}. Similar to
\href{../../eplusr/html/Idf.html#method-add}{\code{$add()}}, a
special element \code{.comment} in each list will be used as the \strong{new}
comments for modified object, overwriting the old ones. Names in list
element are treated as field names.

Note that \code{.()} can be used as an alias as \code{list()}, e.g.
\code{idf$set(Building = .(...))} is equivalent to
\code{idf$set(Building = list(...))}.

There is two special syntax in \verb{$set()}, which is inspired by the
\href{https://cran.r-project.org/package=eplusr}{data.table} package:
\itemize{
\item \code{class := list(field = value)}: Note the use of \verb{:=} instead of
\code{=}. The main difference is that, unlike \code{=}, the left hand side of
\verb{:=} should be a valid class name in current \code{Idf} object. It will
set the field of all objects in specified class to specified value.
\item \code{.(object, object) := list(field = value)}: Similar like above, but
note the use of \code{.()} in the left hand side. You can put multiple
object ID or names in \code{.()}. It will set the field of all specified
objects to specified value.
}

You can delete a field by assigning \code{NULL} to it, e.g. \code{list(fld = NULL)} means to delete the value of field \code{fld}, in the condition
that \code{.default} is \code{FALSE}, \code{fld} is not a required field and the
index of \code{fld} is larger than the number minimum fields required for
that class. If those conditions are not required, \code{fld} will be left
as blank if \code{.default} is \code{FALSE} or filled with default value if
\code{.default} is \code{TRUE}.

By default, trailing empty fields that are not required will be
removed and only minimum required fields are kept. For example, if
\code{rp} is an object in \code{RunPeriod} class in an \code{Idf} of version 8.8,
by default empty field with index larger than 11 will be removed
since they are all non-required fields. You can keep the trailing
empty fields by setting \code{.empty} to \code{TRUE}.

New fields that currently do not exist in that object can also be
set. They will be automatically added on the fly.

Field name matching is \strong{case-insensitive}. For convenience,
underscore-style field names are also allowed, e.g. \code{eNd_MoNtH} is
equivalent to \verb{End Month}.

If not all field names are given, positions of those values without
field names are determined after those values with names. E.g. in
\code{idf$set(floor = list("out_layer", name = "name"))}, \code{"out_layer"}
will be treated as the value for field \verb{Outside Layer} in an object
named \code{floor}, since the value for field \code{Name} has been specified
using explicit field name.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# modify an object by name (case-insensitive)
idf$set(r13layer = list(roughness = "smooth"))

# modify an object by ID
idf$set(..12 = list(roughness = "rough"))

# overwrite existing object comments
idf$set(r13layer = list(.comment = c("New comment")))

# assign default values to fields
idf$set(r13layer = list(solar_absorptance = NULL), .default = TRUE)

# set field values to blanks
idf$set(r13layer = list(solar_absorptance = NULL), .default = FALSE)

# set field values to blank and delete trailing fields
idf$set(r13layer = list(visible_absorptance = NULL), .default = FALSE)

# set field values to blank and keep blank fields
idf$set(r13layer = list(visible_absorptance = NULL), .default = FALSE, .empty = TRUE)

# set all fields in one class
idf$set(Material_NoMass := list(visible_absorptance = 0.9))

# set multiple objects in one class
idf$set(.("r13layer", "r31layer") := list(solar_absorptance = 0.8))
# above is equivalent to
idf$set(r13layer = list(solar_absorptance = 0.8),
        r31layer = list(solar_absorptance = 0.8)
)

# use variable input
sets <- list(r13layer = list(roughness = "smooth"))
idf$set(sets)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-del"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-del}{}}}
\subsection{Method \code{del()}}{
Delete existing objects
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$del(
  ...,
  .ref_by = FALSE,
  .ref_to = FALSE,
  .recursive = FALSE,
  .force = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{integer vectors of object IDs and character vectors of
object names in current \code{Idf} object.}

\item{\code{.ref_by}}{If \code{TRUE}, objects whose fields refer to input objects
will also be deleted. Default: \code{FALSE}.}

\item{\code{.ref_to}}{If \code{TRUE}, objects whose fields are referred by input
objects will also be deleted. Default: \code{FALSE}.}

\item{\code{.recursive}}{If \code{TRUE}, relation searching is performed
recursively, in case that objects whose fields refer to target
object are also referred by another object, and also objects
whose fields are referred by target object are also referred
by another object. Default: \code{FALSE}.}

\item{\code{.force}}{If \code{TRUE}, objects are deleted even if they are
referred by other objects.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$del()} takes integer vectors of object IDs and character vectors of
object names, and deletes objects specified.

If current \link[=level_checks]{validate level} includes reference
checking, objects will not be allowed to be deleted if they are
referred by other objects. For example, an error will be issued if
you want to delete one material that is referred by other
constructions, because doing so will result in invalid field value
references. You may bypass this if you really want to by setting
\code{.force} to \code{TRUE}.

When \code{.ref_by} or \code{.ref_to} is \code{TRUE}, objects will be deleted
only when they have and only have relation with input objects but not
any other objects. For example, a construction \code{const} consist of 4
different materials. If \code{.ref_to} is \code{TRUE}, that 4 materials will
only be deleted when they are only used in \code{const}, but not used in
any other objects.

There are recursively reference relations in \code{Idf} object. For
example, one material's name is referenced by one construction, and
that construction's name can be referred by another surface. You can
delete all of them by setting \code{.recursive} to \code{TRUE}.

If \code{.ref_by} is \code{TRUE}, objects whose fields refer to input objects
will also be deleted.

IF \code{.ref_to} is \code{TRUE}, objects whose fields
are referred by input objects will also be deleted.
}

\subsection{Returns}{
The modified \code{Idf} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# delete objects using names
idf$object("Fraction") # ScheduleTypeLimits
idf$del("Fraction")

# delete objects using IDs
idf$objects(c(39, 40)) # Output:Variable
idf$del(39, 40)

# cannot delete objects that are referred by others
level_checks()$reference # reference-checking is enable by default
idf$del("r13layer") # error

# force to delete objects even thay are referred by others
idf$del("r13layer", .force = TRUE)

# delete objects and also objects that refer to them
idf$del("r31layer", .ref_by = TRUE) # Construction 'ROOF31' will be kept

# delete objects and also objects that they refer to
idf$del("extlights", .ref_to = TRUE) # Schedule 'AlwaysOn' will be kept

# delete objects and also other objects that refer to them recursively
idf$del("roof31", .ref_by = TRUE, .recursive = TRUE)

# delete objects using variable inputs
ids <- idf$object_id("Output:Variable", simplify = TRUE)
idf$del(ids)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-purge"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-purge}{}}}
\subsection{Method \code{purge()}}{
Purge resource objects that are not used
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$purge(object = NULL, class = NULL, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{object}}{an integer vector of object IDs or a character vector
of object names in current \code{Idf} object. Default: \code{NULL}.}

\item{\code{class}}{A character vector of valid class names in current \code{Idf}
object. Default: \code{NULL}.}

\item{\code{group}}{A character vector of valid group names in current \code{Idf}
object. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$purge()} takes an integer vector of object IDs or a character
vectors of object names, and deletes resource objects specified that
are not used by any objects.

Here resource objects indicate all objects that can be referenced by
other objects, e.g. all schedules. \verb{$purge()} will ignore any inputs
that are not resources. If inputs contain objects from multiple
classes, references among them are also taken into account, which
means purging is performed hierarchically. If both materials and
constructions are specified, the latter will be purged first, because
it is possible that input constructions reference input materials.
}

\subsection{Returns}{
The modified \code{Idf} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# purge unused "Fraction" schedule type
idf$purge("on/off") # ScheduleTypeLimits

# purge all unused schedule types
idf$purge(class = "ScheduleTypeLimits")

# purge all unused schedule related objects
idf$purge(group = "Schedules")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-duplicated"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-duplicated}{}}}
\subsection{Method \code{duplicated()}}{
Determine duplicated objects
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$duplicated(object = NULL, class = NULL, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{object}}{an integer vector of object IDs or a character vector
of object names in current \code{Idf} object. Default: \code{NULL}.}

\item{\code{class}}{A character vector of valid class names in current \code{Idf}
object. Default: \code{NULL}.}

\item{\code{group}}{A character vector of valid group names in current \code{Idf}
object. Default: \code{NULL}.

If all \code{object}, \code{class} and \code{group} are \code{NULL}, duplication checking
is performed on the whole \code{Idf}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$duplicated()} takes an integer vector of object IDs or a character
vectors of object names, and returns a \code{\link[data.table:data.table]{data.table::data.table()}}
to show whether input objects contain duplications or not.

Here duplicated objects refer to objects whose field values are the
same except the names. Object comments are just ignored during
comparison.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} of 4 columns:
\itemize{
\item \code{class}: Character. Names of classes that input objects belong to
\item \code{id}: Integer. Input object IDs
\item \code{name}: Character. Input object names
\item \code{duplicate}: Integer. The IDs of objects that input objects
duplicate. If input object is not a duplication, \code{NA} is returned
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# check if there are any duplications in the Idf
idf$duplicated(class = "ScheduleTypeLimits")

# check if there are any duplications in the schedule types
idf$duplicated(class = "ScheduleTypeLimits")

# check if there are any duplications in the schedule groups and
# material class
idf$duplicated(class = "Material", group = "Schedules")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-unique"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-unique}{}}}
\subsection{Method \code{unique()}}{
Remove duplicated objects
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$unique(object = NULL, class = NULL, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{object}}{an integer vector of object IDs or a character vector
of object names in current \code{Idf} object. Default: \code{NULL}.}

\item{\code{class}}{A character vector of valid class names in current \code{Idf}
object. Default: \code{NULL}.}

\item{\code{group}}{A character vector of valid group names in current \code{Idf}
object. Default: \code{NULL}.

If all \code{object}, \code{class} and \code{group} are \code{NULL}, duplication checking
is performed on the whole \code{Idf}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$unique()} takes an integer vector of object IDs or a character
vectors of object names, and remove duplicated objects.

Here duplicated objects refer to objects whose field values are the
same except the names. Object comments are just ignored during
comparison.

\verb{$unique()} will only keep the first unique object and remove all
redundant objects. Value referencing the redundant objects will be
redirected into the unique object.
}

\subsection{Returns}{
The modified \code{Idf} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# remove duplications in the Idf
idf$unique(class = "ScheduleTypeLimits")

# remove duplications in the schedule types
idf$unique(class = "ScheduleTypeLimits")

# remove duplications in the schedule groups and material class
idf$unique(class = "Material", group = "Schedules")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-rename"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-rename}{}}}
\subsection{Method \code{rename()}}{
Rename existing objects
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$rename(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Integer vectors of valid object IDs and character vectors
of valid object names in current \code{Idf} object. Each element
should be named. Names of input vectors are used as the new
object names}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$rename()} takes named character vectors of object names and named
integer vectors of object IDs, renames specified objects to names of
input vectors and returns a list of renamed \link{IdfObject}s. The
returned list will be named using names of modified objects. An error
will be issued if trying to "rename" an object which does not have
name attribute. When renaming an object that is referred by other
objects, corresponding fields that refer to that object's name will
also be changed accordingly.
}

\subsection{Returns}{
A named list of renamed \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$objects(c("on/off", "test 352a"))
idf$rename(on_off = "on/off", test_352a = 51)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-insert"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-insert}{}}}
\subsection{Method \code{insert()}}{
Insert new objects from \link{IdfObject}s
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$insert(..., .unique = TRUE, .empty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{\link{IdfObject}s or lists of \link{IdfObject}s from same version as
current \code{Idf} object.}

\item{\code{.unique}}{If there are duplications in input \link{IdfObject}s or
there is same object in current \code{Idf} object, duplications in
input are removed. Default: \code{TRUE}.}

\item{\code{.empty}}{If \code{TRUE}, trailing empty fields are kept. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$insert()} takes \link{IdfObject}s or lists of \link{IdfObject}s as input,
inserts them into current \code{Idf} objects, and returns a list of
inserted \link{IdfObject}s. The returned list will be named using names of
inserted objects.

\verb{$insert()} is quite useful to insert objects from other \code{Idf}
objects. However, you cannot insert an \link{IdfObject} which comes from a
different version than current \code{Idf} object.

\verb{$insert()} will skip \link{IdfObject}s that have exactly same fields in
current \code{Idf} object. If input \link{IdfObject} has the same name as one
\link{IdfObject} in current \code{Idf} object but field values are not equal,
an error will be issued if current \link[=level_checks]{validate level}
includes conflicted-name checking.

By default, trailing empty fields that are not required will be
removed and only minimum required fields are kept. You can keep the
trailing empty fields by setting \code{.empty} to \code{TRUE}.
}

\subsection{Returns}{
A named list of inserted \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# insert all material from another IDF
path_idf2 <- file.path(eplus_config(8.8)$dir, "ExampleFiles/5ZoneTDV.idf")
idf2 <- Idf$new(path_idf2)
idf$insert(idf2$Material)

# insert objects from same Idf is equivalent to using Idf$dup()
idf$insert(idf$SizingPeriod_DesignDay)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-load"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-load}{}}}
\subsection{Method \code{load()}}{
Load new objects from characters or data.frames
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$load(..., .unique = TRUE, .default = TRUE, .empty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Character vectors or data.frames of object definitions.}

\item{\code{.unique}}{If \code{TRUE}, and there are duplications in input
\link{IdfObject}s or there is same object in current \code{Idf} object,
duplications in input are removed. Default: \code{TRUE}.}

\item{\code{.default}}{If \code{TRUE}, default values are filled for those blank
fields if possible. Default: \code{TRUE}.}

\item{\code{.empty}}{If \code{TRUE}, trailing empty fields are kept. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$load()} is similar to
\href{../../eplusr/html/Idf.html#method-insert}{\code{$insert()}},
except it takes directly character vectors or data.frames as
\link{IdfObject} definitions, insert corresponding objects into current
\code{Idf} object and returns a named list of newly added \link{IdfObject}s.
The returned list will be named using names of added objects. This
makes it easy to create objects using the output from\verb{$to_string()}
and \verb{$to_table()} method from
\href{../../eplusr/html/Idd.html#method-to_string}{\code{Idd}},
\href{../../eplusr/html/IddObject.html#method-to_string}{\code{IddObject}},
also from
\href{../../eplusr/html/Idf.html#method-to_string}{\code{Idf}},
and
\href{../../eplusr/html/IdfObject.html#method-to_string}{\code{IdfObject}},
class.

For object definitions in character vector format, they follow the
same rules as a normal IDF file:
\itemize{
\item Each object starts with a class name and a comma (\verb{,});
\item Separates each values with a comma (\verb{,});
\item Ends an object with a semicolon (\verb{;}) for the last value.
}

Each character vector can contain:
\itemize{
\item One single object, e.g. \code{c("Building,", "MyBuilding;")}, or
"Building, MyBuilding;".
\item Multiple objects, e.g. \code{c("Building, MyBuilding;", "SimulationControl, Yes")}.
}

You can also provide an option header to indicate if input objects
are presented in IP units, using \verb{!-Option ViewInIPunits}. If this
header does not exist, then all values are treated as in SI units.

For object definitions in data.frame format, it is highly recommended
to use \verb{$to_table()} method in
\href{../../eplusr/html/Idd.html#method-to_table}{\code{Idd}},
\link{Idd},
\href{../../eplusr/html/IddObject.html#method-to_table}{\code{IddObject}},
\link{IddObject},
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}},
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}},
class to create an acceptable data.frame template. A
valid definition requires at least three columns described below.
Note that column order does not matter.
\itemize{
\item \code{class}:Character type. Valid class names in the underlying
\link{Idd} object.
\item \code{index}:Integer type. Valid field indices for each class.
\item \code{value}:Character type or list type. Value for each field
to be added.
\itemize{
\item If character type, usually when \code{string_value} is \code{TRUE}
in method \verb{$to_table()} in
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}}
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}}
class. Note that
each value should be given as a string even if the corresponding
field is a numeric type.
\item If list type, usually when \code{string_value} is set to
\code{FALSE} in method\verb{$to_table()} in
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}}
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}}
class.
Each value should have the right type as the corresponding field
definition.  Otherwise, errors will be issued if current
\link[=level_checks]{validation level} includes invalid-type checking.
}
\item \code{id}: \strong{Optional}. Integer type. If input data.frame includes
multiple object definitions in a same class, values in \code{id} column
will be used to distinguish each definition. If \code{id} column does
not exists, it assumes that each definition is separated by \code{class}
column and will issue an error if there is any duplication in the
\code{index} column.
}

Note that \verb{$load()} assumes all definitions are from the same version
as current \code{Idf} object. If input definition is from different
version, parsing error may occur.

By default, trailing empty fields that are not required will be
removed and only minimum required fields are kept. You can keep the
trailing empty fields by setting \code{.empty} to \code{TRUE}.
}

\subsection{Returns}{
A named list of loaded \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# load objects from character vectors
idf$load(
    c("Material,",
      "    mat,                     !- Name",
      "    MediumSmooth,            !- Roughness",
      "    0.667,                   !- Thickness {m}",
      "    0.115,                   !- Conductivity {W/m-K}",
      "    513,                     !- Density {kg/m3}",
      "    1381;                    !- Specific Heat {J/kg-K}"),

    "Construction, const, mat;"
)

# load objects from data.frame definitions
dt <- idf$to_table(class = "Material")
dt[field == "Name", value := paste(value, 1)]
dt[field == "Thickness", value := "0.5"]
idf$load(dt)

# by default, duplications are removed
idf$load(idf$to_table(class = "Material"))

# keep empty fields as they are
idf$load("Material, mat1, smooth, 0.5, 0.2, 500, 1000,,, 0.5;", .default = FALSE)

# keep trailing empty fields
idf$load("Material, mat2, smooth, 0.5, 0.2, 500, 1000,,,;",
    .default = FALSE, .empty = TRUE
)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-update"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-update}{}}}
\subsection{Method \code{update()}}{
Update existing object values from characters or data.frames
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$update(..., .default = TRUE, .empty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Character vectors or data.frames of object definitions.}

\item{\code{.default}}{If \code{TRUE}, default values are filled for those blank
fields if possible. Default: \code{TRUE}.}

\item{\code{.empty}}{If \code{TRUE}, trailing empty fields are kept. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$update()} is similar to
\href{../../eplusr/html/Idf.html#method-set}{\code{$set()}}, except
it takes directly character vectors or data.frames as \link{IdfObject}
definitions, updates new values for fields in objects specified, and
returns a named list of modified \link{IdfObject}s. The returned list will
be named using names of modified objects. This makes it easy to
update object values using the output from \verb{$to_string()} and
\verb{$to_table} method from
\href{../../eplusr/html/Idf.html#method-to_string}{\code{Idf}},
and
\href{../../eplusr/html/IdfObject.html#method-to_string}{\code{IdfObject}},
class.

The format of object definitions is similar to \verb{$load()}.

For object definitions in character vector format, object names are
used to locate which objects to update. Objects that have name
attribute should have valid names. This means that there is no way to
update object names using character vector format, but this can be
achieved using data.frame format as it uses object IDs instead of
object names to locate objects. The format of acceptable characters
follows the same rules as a normal IDF file:
\itemize{
\item Each object starts with a class name and a comma (\verb{,});
\item Separates each values with a comma (\verb{,});
\item Ends an object with a semicolon (\verb{;}) for the last value.
}

Each character vector can contain:
\itemize{
\item One single object, e.g. \code{c("Building,", "MyBuilding;")}, or
"Building, MyBuilding;".
\item Multiple objects, e.g. \code{c("Building, MyBuilding;", "SimulationControl, Yes")}.
}

You can also provide an option header to indicate if input objects
are presented in IP units, using \verb{!-Option ViewInIPunits}. If this
header does not exist, then all values are treated as in SI units.

For object definitions in data.frame format, it is highly recommended
to use \verb{$to_table()} method in
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}},
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}},
class to create an acceptable data.frame template. A valid definition
requires three columns described below. Note that column order does
not matter.
\itemize{
\item \code{id}: Integer type. Valid IDs of objects to update.
\item \code{index}:Integer type. Valid field indices for each object.
\item \code{value}:Character type or list type. Value for each field
to be added.
\itemize{
\item If character type, usually when \code{string_value} is \code{TRUE}
in method \verb{$to_table()} in
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}}
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}}
class. Note that
each value should be given as a string even if the corresponding
field is a numeric type.
\item If list type, usually when \code{string_value} is set to
\code{FALSE} in method \verb{$to_table()} in
\href{../../eplusr/html/Idf.html#method-to_table}{\code{Idf}}
and
\href{../../eplusr/html/IdfObject.html#method-to_table}{\code{IdfObject}}
class.
Each value should have the right type as the corresponding field
definition.  Otherwise, errors will be issued if current
\link[=level_checks]{validation level} includes invalid-type checking.
}
}

Note that \verb{$update()} assumes all definitions are from the same version
as current \code{Idf} object. If input definition is from different
version, parsing error may occur.

By default, trailing empty fields that are not required will be
removed and only minimum required fields are kept. You can keep the
trailing empty fields by setting \code{.empty} to \code{TRUE}.
}

\subsection{Returns}{
A named list of updated \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# update objects from string definitions:
str <- idf$to_string("zone one", header = FALSE, format = "new_top")
str[8] <- "2," # Multiplier
idf$update(str)

# update objects from data.frame definitions:
dt <- idf$to_table("zone one")
dt[field == "Multiplier", value := "1"]
idf$update(dt)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-paste"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-paste}{}}}
\subsection{Method \code{paste()}}{
Paste new objects from IDF Editor
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$paste(in_ip = FALSE, ver = NULL, unique = TRUE, empty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{in_ip}}{Set to \code{TRUE} if the IDF file is open with \code{Inch-Pound}
view option toggled. Numeric values will automatically
converted to SI units if necessary. Default: \code{FALSE}.}

\item{\code{ver}}{The version of IDF file open by IDF Editor, e.g. \code{8.6},
\code{"8.8.0"}. If \code{NULL}, assume that the file has the same
version as current Idf object. Default: \code{NULL}.}

\item{\code{unique}}{If \code{TRUE}, and there are duplications in copied objects
from IDF Editor or there is same object in current Idf,
duplications in input are removed. Default: \code{TRUE}.}

\item{\code{empty}}{If \code{TRUE}, trailing empty fields are kept. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$paste()} reads the contents (from clipboard) of copied objects from IDF
Editor (after hitting \verb{Copy Obj} button), inserts corresponding
objects into current \code{Idf} object and returns a named list of newly
added \link{IdfObject}s. The returned list will be named using names of
added objects. As IDF Editor is only available on Windows platform,
\verb{$paste()} only works on Windows too.

There is no version data copied to the clipboard when copying objects in
IDF Editor. \verb{$paste()} assumes the file open in IDF Editor has the
same version as current \code{Idf} object. This may not be always true.
Please check the version before running \verb{$paste()}, or explicitly
specify the version of file opened by IDF Editor using \code{ver}
parameter. Parsing error may occur if there is a version mismatch.

By default, trailing empty fields that are not required will be
removed and only minimum required fields are kept. You can keep the
trailing empty fields by setting \code{.empty} to \code{TRUE}.
}

\subsection{Returns}{
A named list of loaded \link{IdfObject} objects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-search_value"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-search_value}{}}}
\subsection{Method \code{search_value()}}{
Search objects by field values using regular expression
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$search_value(
  pattern,
  class = NULL,
  ignore.case = FALSE,
  perl = FALSE,
  fixed = FALSE,
  useBytes = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{pattern, ignore.case, perl, fixed, useBytes}}{All of them are
directly passed to \link[base:grep]{base::grepl} and
\link[base:grep]{base::gsub}.}

\item{\code{class}}{A character vector of invalid class names in current
\code{Idf} object to search for values. If \code{NULL}, all classes are
used. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$search_value()} returns a list of \link{IdfObject}s that contain values
which match the given pattern. If no matched found, \code{NULL} is
returned invisibly. The returned list will be named using names of
matched objects.

Note that during matching, all values are treated as characters,
including numeric values.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search values that contains "floor"
idf$search_value("floor", ignore.case = TRUE)

# search values that contains "floor" in class Construction
idf$search_value("floor", "Construction", ignore.case = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-replace_value"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-replace_value}{}}}
\subsection{Method \code{replace_value()}}{
Replace object field values using regular expression
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$replace_value(
  pattern,
  replacement,
  class = NULL,
  ignore.case = FALSE,
  perl = FALSE,
  fixed = FALSE,
  useBytes = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{pattern, replacement, ignore.case, perl, fixed, useBytes}}{All of
them are directly passed to \link[base:grep]{base::grepl} and
\link[base:grep]{base::gsub}.}

\item{\code{class}}{A character vector of invalid class names in current
\code{Idf} object to search for values. If \code{NULL}, all classes are
used. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$replace_value()} returns a list of \link{IdfObject}s whose values have
been replaced using given pattern. If no matched found, \code{NULL} is
returned invisibly. The returned list will be named using names of
matched objects.

Note that during matching, all values are treated as characters,
including numeric values.

Modifying object values using regular expression is not recommended.
Consider to use
\href{../../eplusr/html/Idf.html#method-set}{\code{$set()}}
and
\href{../../eplusr/html/Idf.html#method-update}{\code{$update()}}
if possible.
\link[=level_checks]{Validation} rules also apply during replacing.
}

\subsection{Returns}{
A named list of \link{IdfObject} objects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search values that contains "win" and replace them with "windows"
idf$replace_value("win", "windows")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-validate"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-validate}{}}}
\subsection{Method \code{validate()}}{
Check possible object field value errors
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$validate(level = eplusr_option("validate_level"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{level}}{One of \code{"none"}, \code{"draft"}, \code{"final"} or a list of 10
elements with same format as \code{\link[=custom_validate]{custom_validate()}} output.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$validate()} checks if there are errors in current \code{Idf} object
under specified validation level and returns an \code{IdfValidity} object.
\verb{$validate()} is useful to help avoid some common errors before
running the model. By default, validation is performed when calling
all methods that modify objects, e.g.
\href{../../eplusr/html/Idf.html#method-dup}{\code{$dup()}}
\href{../../eplusr/html/Idf.html#method-add}{\code{$add()}},
\href{../../eplusr/html/Idf.html#method-set}{\code{$set()}},
\href{../../eplusr/html/Idf.html#method-del}{\code{$del()}},
and etc.

In total, there are 10 different validate checking components:
\itemize{
\item \code{required_object}: Check if required objects are missing in current
\code{Idf}.
\item \code{unique_object}: Check if there are multiple objects in one
unique-object class. An unique-object class means that there should
be at most only one object existing in that class.
\item \code{unique_name}: Check if all objects in each class have unique names.
\item \code{extensible}: Check if all fields in an extensible group have
values. An extensible group is a set of fields that should be
treated as a whole, such like the X, Y and Z vertices of a building
surfaces. An extensible group should be added or deleted together.
\code{extensible} component checks if there are some, but not all,
fields in an extensible group are empty.
\item \code{required_field}: Check if all required fields have values.
\item \code{auto_field}: Check if all fields filled with value \code{"Autosize"} and
\code{"Autocalculate"} are actual autosizable and autocalculatable
fields or not.
\item \code{type}: Check if all fields have value types complied with their
definitions, i.e. character, numeric and integer fields should be
filled with corresponding type of values.
\item \code{choice}: Check if all choice fields are filled with valid choice
values.
\item \code{range}: Check if all numeric fields have values within prescibed
ranges.
\item \code{reference}: Check if all fields whose values refer to other fields
are valid.
}

The \code{level} argument controls what checkings should be performed.
\code{level} here is just a list of 10 element which specify the toggle
status of each component. You can use helper \code{\link[=custom_validate]{custom_validate()}} to
get that list and pass it directly to \code{level}.

There are 3 predefined validate level that indicates different
combinations of checking components, i.e. \code{none}, \code{draft} and
\code{final}. Basically, \code{none} level just does not perform any
checkings; \code{draft} includes 5 components, i.e. \code{auto_field}, \code{type},
\code{unique_name}, \code{choice} and \code{range}; and \code{final} level includes all
10 components. You can always get what components each level contains
using \code{\link[=level_checks]{level_checks()}}. By default, the result from
\code{eplusr_option("validate_level")} is passed to \code{level}. If not set,
\code{final} level is used.

Underneath, an \code{IdfValidity} object which \verb{$validate()} returns is a
list of 13 element as shown below. Each element or several elements
represents the results from a single validation checking component.
\itemize{
\item \code{missing_object}: Result of \code{required_object} checking.
\item \code{duplicate_object}: Result of \code{unique_object} checking.
\item \code{conflict_name}: Result of \code{unique_name} checking.
\item \code{incomplete_extensible}: Result of \code{extensible} checking.
\item \code{missing_value}: Result of \code{required_field} checking.
\item \code{invalid_autosize}: Result of \code{auto_field} checking for invalid
\code{Autosize} field values.
\item \code{invalid_autocalculate}: Result of \code{auto_field} checking for
invalid \code{Autocalculate} field values.
\item \code{invalid_character}: Result of \code{type} checking for invalid
character field values.
\item \code{invalid_numeric}: Result of \code{type} checking for invalid
numeric field values.
\item \code{invalid_integer}: Result of \code{type} checking for invalid
integer field values.
\item \code{invalid_choice}: Result of \code{choice} checking.
\item \code{invalid_range}: Result of \code{range} checking.
\item \code{invalid_reference}: Result of \code{reference} checking.
}

Except \code{missing_object}, which is a character vector of class names
that are missing, all other elements are
\link[data.table:data.table]{data.table} with 9 columns containing data
of invalid field values:
\itemize{
\item \code{object_id}: IDs of objects that contain invalid values
\item \code{object_name}: names of objects that contain invalid values
\item \code{class_id}: indexes of classes that invalid objects belong to
\item \code{class_name}: names of classes that invalid objects belong to
\item \code{field_id}: indexes (at Idd level) of object fields that are invalid
\item \code{field_index}: indexes of object fields in corresponding that are invalid
\item \code{field_name}: names (without units) of object fields that are invalid
\item \code{units}: SI units of object fields that are invalid
\item \code{ip_units}: IP units of object fields that are invalid
\item \code{type_enum}: An integer vector indicates types of invalid fields
\item \code{value_id}: indexes (at Idf level) of object field values that are invalid
\item \code{value_chr}: values (converted to characters) of object fields that are
invalid
\item \code{value_num}: values (converted to numbers in SI units) of object fields
that are invalid
}

Knowing the internal structure of \code{IdfValidity}, it is easy to extract
invalid \link{IdfObject}s you interested in. For example, you can get all IDs of
objects that contain invalid value references using
\code{model$validate()$invalid_reference$object_id}. Then using
\href{../../eplusr/html/Idf.html#method-set}{\code{$set()}}
method to correct them.

Different validate result examples are shown below:
\itemize{
\item No error is found:

\if{html}{\out{<div class="sourceCode">}}\preformatted{v No error found.
}\if{html}{\out{</div>}}

Above result shows that there is no error found after conducting all
validate checks in specified validate level.
\item Errors are found:

\if{html}{\out{<div class="sourceCode">}}\preformatted{ x [2] Errors found during validation.
=========================================================================

-- [2] Invalid Autocalculate Field --------------------------------------
   Fields below cannot be `autocalculate`:

    Class: <AirTerminal:SingleDuct:VAV:Reheat>
    \\- Object [ID:176] <SPACE5-1 VAV Reheat>
       +- 17: AUTOCALCULATE, !- Maximum Flow per Zone Floor Area During Reheat \{m3/s-m2\}
       \\- 18: AUTOCALCULATE; !- Maximum Flow Fraction During Reheat
}\if{html}{\out{</div>}}

Above result shows that after all validate components
performed under current validate level, 2 invalid field values
are found. All of them are in a object named \verb{SPACE5-1 VAV Reheat}
with ID \code{176}. They are invalid because those two fields do not
have an autocalculatable attribute but are given \code{AUTOCALCULATE}
value. Knowing this info, one simple way to fix the
error is to correct those two fields by doing:

\if{html}{\out{<div class="sourceCode">}}\preformatted{idf$set(..176 =
    list(`Maximum Flow per Zone Floor Area During Reheat` = "autosize",
         `Maximum Flow Fraction During Reheat` = "autosize"
    )
)
}\if{html}{\out{</div>}}
}
}

\subsection{Returns}{
An \code{IdfValidity} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$validate()

# check at predefined validate level
idf$validate("none")
idf$validate("draft")
idf$validate("final")

# custom validate checking components
idf$validate(custom_validate(auto_field = TRUE, choice = TRUE))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_valid"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_valid}{}}}
\subsection{Method \code{is_valid()}}{
Check if there is any error in current \code{Idf}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_valid(level = eplusr_option("validate_level"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{level}}{One of \code{"none"}, \code{"draft"}, \code{"final"} or a list of 10
elements with same format as \code{\link[=custom_validate]{custom_validate()}} output.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$is_valid()} checks if there are errors in current \code{Idf} object
under specified validation level and returns \code{TRUE} or \code{FALSE}
accordingly. For detailed description on validate checking, see
\href{../../eplusr/html/Idf.html#method-validate}{\code{$validate()}}
documentation above.
}

\subsection{Returns}{
A single logical value of \code{TRUE} or \code{FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$is_valid()

# check at predefined validate level
idf$is_valid("none")
idf$is_valid("draft")
idf$is_valid("final")

# custom validate checking components
idf$is_valid(custom_validate(auto_field = TRUE, choice = TRUE))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-to_string"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-to_string}{}}}
\subsection{Method \code{to_string()}}{
Format \code{Idf} as a character vector
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$to_string(
  which = NULL,
  class = NULL,
  comment = TRUE,
  header = TRUE,
  format = eplusr_option("save_format"),
  leading = 4L,
  sep_at = 29L
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{Either an integer vector of valid object IDs or a
character vector of valid object names. If \code{NULL}, the whole
\code{Idf} object is converted. Default: \code{NULL}.}

\item{\code{class}}{A character vector of class names. If \code{NULL}, all
classed in current \code{Idf} object is converted. Default: \code{NULL}.}

\item{\code{comment}}{If \code{FALSE}, all comments will not be included.
Default: \code{TRUE}.}

\item{\code{header}}{If \code{FALSE}, the header will not be included. Default:
\code{TRUE}.}

\item{\code{format}}{Specific format used when formatting. Should be one of
\code{"asis"}, \code{"sorted"}, \code{"new_top"}, and \code{"new_bot"}.
\itemize{
\item If \code{"asis"}, \code{Idf} object will be formatted in the same way as it
was when first read. If \code{Idf} object does not contain any format
saving option, which is typically the case when the model was not
saved using eplusr or IDFEditor, \code{"sorted"} will be used.
\item \code{"sorted"}, \code{"new_top"} and \code{"new_bot"} are the same as the save
options \code{"Sorted"}, \code{"Original with New at Top"}, and \code{"Original with New at Bottom"} in IDFEditor. Default:
\code{eplusr_option("save_format")}.
}}

\item{\code{leading}}{Leading spaces added to each field. Default: \code{4L}.}

\item{\code{sep_at}}{The character width to separate value string and field
string. Default: \code{29L} which is the same as IDF Editor.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$to_string()} returns the text format of parts or whole \code{Idf}
object.
}

\subsection{Returns}{
A character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get text format of the whole Idf
head(idf$to_string())

# get text format of the whole Idf, excluding the header and all comments
head(idf$to_string(comment = FALSE, header = FALSE))

# get text format of all objects in class Material
head(idf$to_string(class = "Material", comment = FALSE, header = FALSE))

# get text format of some objects
head(idf$to_string(c("floor", "zone one")))

# tweak output formatting
head(idf$to_string("floor", leading = 0, sep_at = 0))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-to_table"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-to_table}{}}}
\subsection{Method \code{to_table()}}{
Format \code{Idf} as a data.frame
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$to_table(
  which = NULL,
  class = NULL,
  string_value = TRUE,
  unit = FALSE,
  wide = FALSE,
  align = FALSE,
  all = FALSE,
  group_ext = c("none", "group", "index"),
  force = FALSE,
  init = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{Either an integer vector of valid object IDs or a
character vector of valid object names. If \code{NULL}, the whole
\code{Idf} object is converted. Default: \code{NULL}.}

\item{\code{class}}{A character vector of class names. If \code{NULL}, all
classed in current \code{Idf} object is converted. Default: \code{NULL}.}

\item{\code{string_value}}{If \code{TRUE}, all field values are returned as
character. If \code{FALSE}, \code{value} column in returned
\link[data.table:data.table]{data.table} is a list column with
each value stored as corresponding type. Note that if the
value of numeric field is set to \code{"Autosize"} or
\code{"Autocalculate"}, it is left as it is, leaving the returned
type being a string instead of a number. Default: \code{TRUE}.}

\item{\code{unit}}{Only applicable when \code{string_value} is \code{FALSE}. If
\code{TRUE}, values of numeric fields are assigned with units using
\code{\link[units:units]{units::set_units()}} if applicable. Default: \code{FALSE}.}

\item{\code{wide}}{Only applicable if target objects belong to a same class.
If \code{TRUE}, a wide table will be returned, i.e. first three
columns are always \code{id}, \code{name} and \code{class}, and then every
field in a separate column. Note that this requires all
objects specified must from the same class.
Default: \code{FALSE}.}

\item{\code{align}}{If \code{TRUE}, all objects in the same class will have the
same field number. The number of fields is the same as the
object that have the most fields among objects specified.
Default: \code{FALSE}.}

\item{\code{all}}{If \code{TRUE}, all available fields defined in IDD for the
class that objects belong to will be returned. Default:
\code{FALSE}.}

\item{\code{group_ext}}{Should be one of \code{"none"}, \code{"group"} or \code{"index"}.
If not \code{"none"}, \code{value} column in returned
\code{\link[data.table:data.table]{data.table::data.table()}} will be converted into a list.
If \code{"group"}, values from extensible fields will be grouped by the
extensible group they belong to. For example, coordinate
values of each vertex in class \code{BuildingSurface:Detailed} will
be put into a list. If \code{"index"}, values from extensible fields
will be grouped by the extensible field indice they belong to.
For example, coordinate values of all x coordinates will be
put into a list. If \code{"none"}, nothing special will be done.
Default: \code{"none"}.}

\item{\code{force}}{If \code{TRUE}, \code{wide} can be \code{TRUE} even though there are
multiple classes in input. This can result in a data.table
with lots of columns. But may be useful when you know that
target classes have the exact same fields, e.g.
\code{Ceiling:Adiabatic} and \code{Floor:Adiabatic}. Default: \code{FALSE}.}

\item{\code{init}}{If \code{TRUE}, a table for new object input will be returned
with all values filled with defaults. In this case, \code{object}
input will be ignored. The \code{id} column will be filled with
possible new object IDs. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$to_table()} returns a \link[data.table:data.table]{data.table} that
contains core data of specified objects.
The returned \link[data.table:data.table]{data.table} has 5 columns:
\itemize{
\item \code{id}: Integer type. Object IDs.
\item \code{name}: Character type. Object names.
\item \code{class}: Character type. Current class name.
\item \code{index}: Integer type. Field indexes.
\item \code{field}: Character type. Field names.
\item \code{value}: Character type if \code{string_value} is \code{TRUE} or list type if
\code{string_value} is \code{FALSE} or \code{group_ext} is not \code{"none"}. Field values.
}

Note that when \code{group_ext} is not \code{"none"}, \code{index} and \code{field}
values will not match the original field indices and names. In this
case, \code{index} will only indicate the indices of sequences. For
\code{field} column, specifically:
\itemize{
\item When \code{group_ext} is \code{"group"}, each field name in a extensible group
will be abbreviated using \code{\link[=abbreviate]{abbreviate()}} with \code{minlength} being
\code{10L} and all abbreviated names will be separated by \code{|} and
combined together. For example, field names in the extensible group
(\verb{Vertex 1 X-coordinate}, \verb{Vertex 1 Y-coordinate}, \verb{Vertex 1 Z-coordinate}) in class \code{BuildiBuildingSurface:Detailed} will be
merged into one name \code{Vrtx1X-crd|Vrtx1Y-crd|Vrtx1Z-crd}.
\item When \code{group_ext} is \code{"index"}, the extensible group indicator in field
names will be removed. Take the same example as above, the
resulting field names will be \verb{Vertex X-coordinate}, \verb{Vertex Y-coordinate}, and \verb{Vertex Z-coordinate}.
}
}

\subsection{Returns}{
A \link[data.table:data.table]{data.table} with 6 columns (if
\code{wide} is \code{FALSE}) or at least 6 columns (if \code{wide} is \code{TRUE}).
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# extract whole Idf data
idf$to_table()

# extract all data from class Material
idf$to_table(class = "Material")

# extract multiple object data
idf$to_table(c("FLOOR", "ZONE ONE"))

# keep value types and put actual values into a list column
idf$to_table(c("FLOOR", "ZONE ONE"), string_value = FALSE)$value

# add the unit to each value
idf$to_table(c("FLOOR", "ZONE ONE"), string_value = FALSE, unit = TRUE)

# get all possible fields
idf$to_table("ZONE ONE", all = TRUE)

# make sure all objects in same class have the same number of fields
idf$to_table(class = "Construction", align = TRUE)

# get a wide table with string values
idf$to_table(class = "Construction", wide = TRUE)

# get a wide table with actual values
idf$to_table(class = "OtherEquipment", wide = TRUE, string_value = FALSE)

# group extensible by extensible group number
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "group")

# group extensible by extensible group number and convert into a wide table
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "group", wide = TRUE)

# group extensible by extensible field index
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index")

# group extensible by extensible field index and convert into a wide table
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index", wide = TRUE)

# when grouping extensible, 'string_value' and 'unit' still take effect
idf$to_table(class = "BuildingSurface:Detailed", group_ext = "index",
    wide = TRUE, string_value = FALSE, unit = TRUE
)

# create table for new object input
idf$to_table(class = "BuildingSurface:Detailed", init = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-external_deps"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-external_deps}{}}}
\subsection{Method \code{external_deps()}}{
Get external file dependencies that the Idf needs for simulation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$external_deps(full = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{full}}{If \code{TRUE}, a \link[data.table:data.table]{data.table} is
returned giving details about the objects and fields that use those
external file dependencies. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$external_deps()} returns information of files that are used as
external resources for the simulation.

Currently, classes below are checked:
\itemize{
\item \code{Schedule:File:Shading}
\item \code{Schedule:File}
\item \code{Construction:WindowDataFile}
\item \code{ExternalInterface:FunctionalMockupUnitImport}
\item \code{ExternalInterface:FunctionalMockupUnitImport:From:Variable}
\item \code{ExternalInterface:FunctionalMockupUnitImport:To:Schedule}
\item \code{ExternalInterface:FunctionalMockupUnitImport:To:Actuator}
\item \code{ExternalInterface:FunctionalMockupUnitImport:To:Variable}
\item \code{Table:IndependentVariable}
\item \code{Table:Lookup}
}

Note that, for \code{ExternalInterface:FunctionalMockupUnitImport} and
\verb{ExternalInterface:FunctionalMockupUnitImport:*}, resources of FMU
will also be extracted.
}

\subsection{Returns}{
When \code{full} is \code{FALSE}, which is the default, a character vector.

When \code{full} is \code{TRUE}, a \link[data.table:data.table]{data.table} of 8
columns:
\itemize{
\item \code{id}: Integer type. Object IDs.
\item \code{name}: Character type. Object names.
\item \code{class}: Character type. Current class name.
\item \code{index}: Integer type. Field indexes.
\item \code{field}: Character type. Field names.
\item \code{value}: Character type. Field values.
\item \code{path}: Character type. Full file paths.
\item \code{exist}: Logical type. \code{TRUE} if file exists, \code{FALSE} otherwise.
}

If there are any FMUs using external file resources, the returned
data.table will have an attribute named \code{extra} which is a list
giving the FMU name and external file resources it use.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$external_deps()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-is_unsaved"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-is_unsaved}{}}}
\subsection{Method \code{is_unsaved()}}{
Check if there are unsaved changes in current \code{Idf}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$is_unsaved()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$is_unsaved()} returns \code{TRUE} if there are modifications on the
model since it was read or since last time it was saved, and returns
\code{FALSE} otherwise.
}

\subsection{Returns}{
A single logical value of \code{TRUE} or \code{FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$is_unsaved()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-save"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-save}{}}}
\subsection{Method \code{save()}}{
Save \code{Idf} object as an IDF file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$save(
  path = NULL,
  format = eplusr_option("save_format"),
  overwrite = FALSE,
  copy_external = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{path}}{A path where to save the IDF file. If \code{NULL}, the path of
the \code{Idf} itself, i.e.
\href{../../eplusr/html/Idf.html#method-path}{\code{$path()}},
will be used.}

\item{\code{format}}{Specific format used when formatting. Should be one of
\code{"asis"}, \code{"sorted"}, \code{"new_top"}, and \code{"new_bot"}.
\itemize{
\item If \code{"asis"}, \code{Idf} object will be formatted in the same way as it
was when first read. If \code{Idf} object does not contain any format
saving option, which is typically the case when the model was not
saved using eplusr or IDFEditor, \code{"sorted"} will be used.
\item \code{"sorted"}, \code{"new_top"} and \code{"new_bot"} are the same as the save
options \code{"Sorted"}, \code{"Original with New at Top"}, and \code{"Original with New at Bottom"} in IDFEditor. Default:
\code{eplusr_option("save_format")}.
}}

\item{\code{overwrite}}{Whether to overwrite the file if it already exists.
Default: \code{FALSE}.}

\item{\code{copy_external}}{If \code{TRUE}, the external files extracted from
\verb{$external_deps()} will also be copied into the same directory.
The values of file paths in the \code{Idf} will be changed into
relative path automatically. This makes it possible to create
fully reproducible simulation conditions. If \code{FALSE}, the
values of those fields that reference external file paths will
be updated to absolute paths. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$save()} formats current \code{Idf} object, saves it as an IDF file and
returns the path of saved file invisibly. After saving,
\href{../../eplusr/html/Idf.html#method-path}{\code{$path()}}
will also be updated to return the path of saved file.
}

\subsection{Returns}{
A length-one character vector, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# save Idf as a new file
idf$save(tempfile(fileext = ".idf"))

# save and overwrite current file
idf$save(overwrite = TRUE)

# save the model with newly created and modified objects at the top
idf$save(overwrite = TRUE, format = "new_top")

# save the model to a new file and copy all external csv files used in
# "Schedule:File" class into the same folder
idf$save(path = file.path(tempdir(), "test1.idf"), copy_external = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-run"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-run}{}}}
\subsection{Method \code{run()}}{
Run simulation using EnergyPlus
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$run(
  weather,
  dir = NULL,
  wait = TRUE,
  force = FALSE,
  copy_external = FALSE,
  echo = wait
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{weather}}{A path to an \code{.epw} file or an \link{Epw} object. \code{weather}
can also be \code{NULL} which will force design-day-only
simulation. Note that this needs at least one
\code{Sizing:DesignDay} object exists in the \code{Idf}.}

\item{\code{dir}}{The directory to save the simulation results. If \code{NULL},
the folder of \code{Idf} path will be used. Default: \code{NULL}.}

\item{\code{wait}}{Whether to wait until the simulation completes and print
the standard output and error of EnergyPlus. If \code{FALSE}, the
simulation will run in the background. Default is \code{TRUE}.}

\item{\code{force}}{Only applicable when the last simulation runs with
\code{wait} equals to \code{FALSE} and is still running. If \code{TRUE},
current running job is forced to stop and a new one will
start. Default: \code{FALSE}.}

\item{\code{copy_external}}{If \code{TRUE}, the external files that current \code{Idf}
object depends on will also be copied into the simulation
output directory. The values of file paths in the Idf will be
changed automatically. This ensures that the output directory
will have all files needed for the model to run. Default is
\code{FALSE}.}

\item{\code{echo}}{Only applicable when \code{wait} is \code{TRUE}. Whether to show
standard output and error from EnergyPlus. Default: same as
\code{wait}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$run()} calls corresponding version of EnergyPlus to run the current
\code{Idf} object together with specified weather. The model and the
weather used will be copied into the output directory. An \link{EplusJob}
object is returned which provides detailed info of the simulation and
methods to collect simulation results. Please see \link{EplusJob} for
details.

eplusr uses the EnergyPlus command line interface which was
introduced since EnergyPlus 8.3.0. So \verb{$run()} only supports models
with version no lower than 8.3.0.

When calling \verb{$run()}, eplusr will do steps below to make sure the
output collecting methods work as expected. Please note that this may
result in an IDF file that may not be exactly same as your current
\code{Idf} object.
\itemize{
\item eplusr uses EnergyPlus SQL output for extracting simulation
results. In order to do so, an object in \code{Output:SQLite} class with
\verb{Option Type} value being \code{SimpleAndTabular} will be automatically
created if it does not exists.
\item In order to make sure \code{.rdd} (Report Data Dictionary) and \code{.mdd}
(Meter Data Dictionary) files are created during simulation, an
object in \code{Output:VariableDictionary} class with \verb{Key Field} value
being \code{IDF} will be automatically created if it does not exists.
}
}

\subsection{Returns}{
An \link{EplusJob} object of current simulation.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf <- Idf$new(path_idf)
# save the model to tempdir()
idf$save(file.path(tempdir(), "test_run.idf"))

# use the first epw file in "WeatherData" folder in EnergyPlus v8.8
# installation path
epw <- list.files(file.path(eplus_config(8.8)$dir, "WeatherData"),
    pattern = "\\\\.epw$", full.names = TRUE)[1]

# if `dir` is NULL, the directory of IDF file will be used as simulation
# output directory
job <- idf$run(epw, dir = NULL)

# run simulation in the background
idf$run(epw, dir = tempdir(), wait = FALSE)

# copy all external files into the directory run simulation
idf$run(epw, dir = tempdir(), copy_external = TRUE)

# check for simulation errors
job$errors()

# get simulation status
job$status()

# get output directory
job$output_dir()

# re-run the simulation
job$run()

# get simulation results
job$report_data()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-last_job"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-last_job}{}}}
\subsection{Method \code{last_job()}}{
Get the last simulation job
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$last_job()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$last_job()} returns the last \link{EplusJob} object that was created
using
\href{../../eplusr/html/Idf.html#method-run}{\code{$run()}}. If the
\code{Idf} hasn't been run yet, \code{NULL} is returned.
}

\subsection{Returns}{
\code{NULL} or an \link{EplusJob} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$last_job()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-geometry"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-geometry}{}}}
\subsection{Method \code{geometry()}}{
Extract \code{Idf} geometries
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$geometry()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$geometry()} extracts all geometry objects into an \link{IdfGeometry}
object. \code{IdfGeometry} is an abstraction of a collection of geometry
in an \link{Idf}. It provides more detail methods to query geometry
properties, update geometry vertices and visualize geometry in 3D
using the \href{https://cran.r-project.org/package=rgl}{rgl} package.
}

\subsection{Returns}{
An \link{IdfGeometry} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$geometry()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-view"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-view}{}}}
\subsection{Method \code{view()}}{
View 3D \code{Idf} geometry
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$view(
  new = FALSE,
  render_by = "surface_type",
  wireframe = TRUE,
  x_ray = FALSE,
  axis = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{new}}{If \code{TRUE}, a new rgl window will be open using
\code{\link[rgl:rgl.open]{rgl::rgl.open()}}. If \code{FALSE}, existing rgl window will be
reused if possible. Default: \code{FALSE}.}

\item{\code{render_by}}{A single string specifying the way of rendering the
geometry. Possible values are:
\itemize{
\item \code{"surface_type"}: Default. Render the model by surface type
model. Walls, roofs, windows, doors, floors, and shading
surfaces will have unique colors.
\item \code{"boundary"}: Render the model by outside boundary condition.
Only surfaces that have boundary conditions will be rendered
with a color. All other surfaces will be white.
\item \code{"construction"}: Render the model by surface constructions.
\item \code{"zone"}: Render the model by zones assigned.
\item \code{"normal"}: Render the model by surface normal. The outside
face of a heat transfer face will be rendered as white and the
inside face will be rendered as red.
}}

\item{\code{wireframe}}{If \code{TRUE}, the wireframe of each surface will be
shown. Default: \code{TRUE}.}

\item{\code{x_ray}}{If \code{TRUE}, all surfaces will be rendered translucently.
Default: \code{FALSE}.}

\item{\code{axis}}{If \code{TRUE}, the X, Y and Z axes will be drawn at the
global origin. Default: \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$view()} uses the \href{https://cran.r-project.org/package=rgl}{rgl}
package to visualize the IDF geometry in 3D in a similar way as
\href{https://openstudio.net/}{OpenStudio}.

\verb{$view()} returns an \link{IdfViewer} object which can be used to further
tweak the viewer scene.

In the rgl window, you can control the view using your mouse:
\itemize{
\item Left button: Trackball
\item Right button: Pan
\item Middle button: Field-of-view (FOV). '0' means orthographic
projection.
\item Wheel: Zoom
}
}

\subsection{Returns}{
An \link{IdfViewer} object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$view()
idf$view(render_by = "zone")
idf$view(render_by = "construction")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-print"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-print}{}}}
\subsection{Method \code{print()}}{
Print \code{Idf} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$print(zoom = "class", order = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{zoom}}{Control how detailed of the Idf object should be printed.
Should be one of \code{"group"}, \code{"class"}, \code{"object"} and
\code{"field"}. Default: \code{"group"}.
\itemize{
\item \code{"group"}: all group names current existing are shown with prevailing
square bracket showing how many \strong{C}lasses existing in that group.
\item \code{"class"}: all class names are shown with prevailing square bracket
showing how many \strong{O}bjects existing in that class, together with
parent group name of each class.
\item \code{"object"}: all object IDs and names are shown, together with parent
class name of each object.
\item \code{"field"}: all object IDs and names, field names and values are shown,
together with parent class name of each object.
}}

\item{\code{order}}{Only applicable when \code{zoom} is \code{"object"} or \code{"field"}.
If \code{TRUE}, objects are shown as the same order in the IDF. If
\code{FALSE}, objects are grouped and ordered by classes. Default:
\code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$print()} prints the \code{Idf} object according to different detail
level specified using the \code{zoom} argument.

With the default \code{zoom} level \code{object}, contents of the \code{Idf} object
is printed in a similar style as you see in IDF Editor, with
additional heading lines showing \code{Path}, \code{Version} of the \code{Idf}
object. Class names of objects are ordered by group and the number of
objects in classes are shown in square bracket.
}

\subsection{Returns}{
The \code{Idf} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
idf$print("group")
idf$print("class")
idf$print("object")
idf$print("field")

# order objects by there classes
idf$print("object", order = FALSE)
idf$print("field", order = FALSE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Idf-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Idf-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Idf$clone(deep = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
