% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SRU.R
\name{SRU}
\alias{SRU}
\alias{SRUcalc}
\alias{cut_in}
\alias{plot.SRU}
\alias{print.SRU}
\title{Standardized Resource Use (SRU)}
\usage{
SRU(prob, death, unit, los, los.exp, class, score, plot = FALSE, type = 1,
  digits = 2, digits2 = 5, originals = FALSE)

\method{print}{SRU}(x, ...)

\method{plot}{SRU}(x, ..., xlim = range(x$rates[, 2]),
  ylim = range(x$rates[, 1]), xlab = "SMR", ylab = "SRU",
  points.arg = list(pch = 21, col = "white", bg = "cadetblue3", cex = 1.5),
  med.arg = list(col = "dodgerblue4", lwd = 2, lty = 1), tert.arg = list(col
  = "darkorange2", lty = 2, lwd = 1), auto.legend = TRUE, leg.arg = list(x =
  "top", bty = "n", xpd = NA, inset = -0.2, ncol = 2), bty = "n")

cut_in(score, los, death, unit, days, min = 200, exc.ICU = TRUE,
  complete = FALSE, digits = 5)

SRUcalc(prob, death, unit, los, score, digits = 2)
}
\arguments{
\item{prob}{Individual predicted death (ranging from 0 to 1) in a vector.}

\item{death}{Observed death. Accepted values are 0 (absence) or 1 (presence) in a vector.}

\item{unit}{A character or factor variable indicating the ICU where the patient is admitted.}

\item{los}{A numeric variable indicating the observed length of stay for each patient.}

\item{los.exp}{Estimated length of stay (LOS). This argument is optional and will be required only if \code{type = 2}. If the user has an alternative model to estimate the individual LOS, the predicted individual LOS should passed to this argument. If this is the case, the predicted ICU LOS is estimated as the mean of the individual LOS predictions of these groups.}

\item{class}{A factor variable indicating the class of severity score (e.g. SAPS 3). In the case of SAPS 3, this is a cut in the SAPS 3 score grouping patients into severity classes. This will be required if the argument \code{original = FALSE} and NAs are not allowed; if \code{original = TRUE}, class is ignored.}

\item{score}{A numeric vector with the Acute Physiology Score (SAPS) 3 score for each admission. The function will use this argument to know to wich severity class each patient will assigned to. It's used only when \code{originals = TRUE} and ignored otherwise. NAs are not allowed.}

\item{plot}{Logical; If \code{TRUE} plots a SMR versus SRU scatter plot.}

\item{type}{Way to calculate SRU. If \code{type = 1}, it does as the original article (default); which is to estimate the ICU's expected LOS by multiplying the overall average days within each severity class by the number of survivors in the same severity class in that ICU, and summing the expected LOS for each severity class in that ICU. If \code{type = 2}, then the user must provide the \code{los.exp} (expected LOS) for each subject (i.e. from a prediction model), and the function will estimate the ICU's expected LOS as the mean of the individual LOS.}

\item{digits, digits2}{Integer indicating the number of decimals to be used in the output.}

\item{originals}{Logical; If \code{TRUE} it uses the severity classes and average days as the original article and will override the \code{class} argument if any. It requires the \code{score} argument and it must be the SAPS 3 score. We recommend not to set it \code{TRUE}, unless you really know what you are doing. Even if one wished to have severity classes identical to the original paper, it is better to set the severity classes before running the analysis. Like this, the function will estimate the average days from the data instead of using the fixed average days from the original paper.}

\item{x}{For \code{print.SRU} or \code{plot.SRU}, an object of class 'SRU'.}

\item{...}{Arguments to be passed to \code{\link[graphics]{plot.default}} or to \code{\link[base]{print}}.}

\item{xlim, ylim}{Limits of x and y axis for \code{plot.SRU}.}

\item{xlab, ylab}{Labels of x and y axis for \code{plot.SRU}.}

\item{points.arg}{List of arguments passed to \code{\link[graphics]{points}} for plotting points correponding to ICUs' SMR and SRU.}

\item{med.arg}{List of arguments passed to \code{\link[graphics]{abline}} for plotting lines corresponding to SRU and SMR medians.}

\item{tert.arg}{List of arguments passed to \code{\link[graphics]{abline}} for plotting lines corresponding to SRU and SMR tertiles.}

\item{auto.legend}{Logical; If \code{TRUE}, prints a legend with parameters in \code{leg.arg} arguments.}

\item{leg.arg}{List of arguments passed to \code{\link[graphics]{legend}} for plotting legends in \code{plot.SRU}.}

\item{bty}{A character string which determined the type of box which is drawn about plots. See  \code{\link[graphics]{par}}}

\item{days}{For \code{cut_in}, this is a vector with days which one wants to average days to match. See example.}

\item{min}{For \code{cut_in}, this is the minimum desired quantity of patients in each severity class (default = 200) to estimate the average days.}

\item{exc.ICU}{Logical; For \code{cut_in}, if \code{TRUE}, ICUs without surviving patients are ignored.}

\item{complete}{Logical; For \code{cut_in}, if \code{TRUE}, shows additional information about severity classes.}
}
\value{
Two tables: one with information about severity classes and the respective quantities required to estimate the expected LOS, and another with information about ICUs classified as Most Efficient (ME) or Least Efficient (LE).
\itemize{
\item \code{Sev} Severity class.
\item\code{Total} Total of patients.
\item \code{Surv} Total of survivors.
\item \code{Total.LOS} Total length of stay (days).
\item \code{AvDays} Average days to produce a survivor.
\item \code{N.Unit} Quantity of ICUs.
\item \code{N.Pat} Quantity of patients.
\item\code{SMR} Standardized Mortality Ratio Mean (standard deviation).
\item \code{SRU} Standardized Resource Use Mean (standard deviation).
}

Most Efficient ICUs have SRU, SMR < median. Least Efficient ICUs have SRU, SMR > median.

\code{cut_in} returns a vector with the limits to cut the severity score.

\code{SRUcalc} returns a table with:
\itemize{
\item \code{Unit} ICUs names.
\item\code{SMR or SRU} Standardized Rate.
\item \code{N} Number of subjects analyzed.
\item \code{Observed} Observed number of deaths.
\item \code{Expected} Expected number of deaths.
\item \code{LOS_esp} Expected length of stay.
}
}
\description{
\code{SRU} calculates the standardized resource use for ICUs (Intensive Care Units) from information regarding individual patients admissions. The resource use is represented by the patients' length of stay (LOS). Therefore the SRU for each unit is defined as the observed LOS divided by the expected LOS for each ICU. To estimate the expected LOS for each ICU one must define a severity score, here defined by the SAPS 3 score. In theory, the 'score' could be any score/probability estimating death for each ICU admission.

The \code{plot.SRU} function will return a \code{\link{SMR}} versus SRU scatter plot with its medians and tertiles. Thus, it classifies each unit in the quadrants formed by these two medians: most efficient (ME) is the lower left quadrant (SRU and SMR below the median);  least efficient (LE) is the upper right quadrant (SRU and SMR above the median); and least achieving (LE) - the lower right wuadrant(SRU below and SMR above the median); and over achieving (OA) - the upper left quadrant (SRU above and SMR below the median).

 \code{print.SRU} Prints a object of class 'SRU'.

 \code{cut_in} is used to find limits to define severity classes which are used in \code{SRU} function. The severity classes are necessary to calculate average days to produce one survivor and consequently to estimate the expected LOS in each ICU. Its rationale is to find the severity classes limits that yelds a desired average days of survivors. At some point in time, we made a study to test if different arrangements of the severity classes would yeld different classifications in the efficiency quadrants. Despite this study did not show any difference from the original approach, we left the function in the package. Therefore, any arbitrary severity classes should yeld the same results.

 \code{SRUcalc} is a simpler function to estimte SRU and returns, for each unit, the SRU value, the observed and expected number of deaths, and the observed and expected LOS.
}
\examples{

# Loading the dataset
data(icu)

# Removing data with inapropriate values and some editing
icu <- icu[-which(icu$los < 0 ),]
icu$Saps3DeathProbabilityStandardEquation <- icu$Saps3DeathProbabilityStandardEquation / 100

# Setting classes acording to limits of SAPS 3 score
days <- seq(1,100)
cut_lims <- cut_in(icu$Saps3Points, icu$los, icu$UnitDischargeName,
                   icu$Unit, days, exc.ICU = TRUE)
icu$class <- cut(icu$Saps3Points, breaks = cut_lims, include.lowest = TRUE)

# Estimating the SRU
x <- SRU(prob = icu$Saps3DeathProbabilityStandardEquation,
death = icu$UnitDischargeName, unit = icu$Unit,
los = icu$los, score = icu$Saps3Points,
originals = TRUE, type = 1, plot = FALSE)
x
plot(x)

# To see the units rankings and individual SMR and SRU, ordering by its SRU
x$rates[order(x$rates$sru),]

# SRU with diferent severity classes created by cut_in function
y <- SRU(prob = icu$Saps3DeathProbabilityStandardEquation,
death = icu$UnitDischargeName, unit = icu$Unit,
los = icu$los, score = icu$Saps3Points,
originals = FALSE, type = 1, plot = FALSE, class = icu$class)
y

# Using SRUcalc
SRUcalc(prob = icu$Saps3DeathProbabilityStandardEquation,
        death = icu$UnitDischargeName, unit = icu$Unit, los = icu$los,
        score = icu$Saps3Points)

rm(x, y, days, icu, cut_lims)

}
\author{
Lunna Borges and Pedro Brasil
}
\references{
Rothen HU, Stricker K, Einfalt J, Bauer P, Metnitz PGH, Moreno RP, Takala J (2007) Variability in outcome and resource use in intensive care units. Intensive Care Med 33:1329-1336
}
\seealso{
\code{\link{SMR}}, \code{\link{reclass}}, \code{\link{funnel}}
}

