#' plot.NestsResult Plot the embryo growth
#' @title Plot the embryo growth
#' @author Marc Girondot
#' @return Nothing
#' @param x A result file generated by searchR
#' @param ... Parameters for plot()
#' @param NestsResult A NestsResult file generated by searchR
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if result is not provided, or replace the one in NestsResult. Use SE=NA to remove SE from NestResult
#' @param temperatures Timeseries of temperatures formatted using formatNests(). Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings
#' @param weight Weights of the different nests to estimate likelihood
#' @param stopattest TRUE or FALSE. If TRUE, the model stops when proxy of size reached the mean test size.
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be displayed. Only one series can be displayed at a time.
#' @param TSP.borders The limits of TSP in stages. See embryo.stages parameter.
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided to estimate TSP borders. See note.
#' @param replicate.CI Number of replicates to estimate CI. If 1, no CI is estimated.
#' @param metric.end.incubation The expected metric at the end of incubation. Used to calibrate TSP size. If NULL, take the maximum Mean of the test parameter. If NA, use the actual final size. Can be a vector and is recycled if necessary.
#' @param col.stages The color of the stages
#' @param col.temperatures The color of the temperatures
#' @param col.TSP The color of the TSP
#' @param col.PT The color of the pivotal temperature
#' @param col.S The color of the size or mass. Can be a vector (useful when series="all" option).
#' @param lty.temperatures Type of line for temperatures
#' @param lwd.temperatures Width of line for temperatures
#' @param ylimT Range of temperatures to be displayed
#' @param ylimS Range of size to be displayed
#' @param xlim Range of incubation days to be displayed
#' @param show.metric TRUE or FALSE, does the plot of embryo metric is shown?
#' @param show.stages TRUE or FALSE, does the embryo stages should be displayed?
#' @param show.TSP TRUE or FALSE, does the TSP boders should be displayed?
#' @param show.third TRUE or FALSE, does the first and second third boders should be displayed?
#' @param show.CI TRUE or FALSE, do the confidence intervals should be displayed?
#' @param show.temperatures TRUE or FALSE, does the temperatures should be displayed?
#' @param show.PT TRUE or FALSE, does the pivotal temperature should be displayed?
#' @param show.fioritures If FALSE, set show.PT, show.temperatures, show.stages, show.TSP, show.third, show.CI to FALSE
#' @param PT Value for pivotal temperature, mean and SE
#' @param show.test TRUE or FALSE, does the hatchling size should be displayed
#' @param lab.third Label for 2nd third of incubation
#' @param at.lab.third Position of Label for 2nd third of incubation [default=4]
#' @param lab.PT Label for Pivotal Temperature
#' @param lab.stages Label for Stages
#' @param xlab Label for axis
#' @param ylabT Label for temperature axis
#' @param ylabS Label for size axis
#' @param mar Parameter mar used for plot
#' @param add If TRUE, all the curves are shown on the same graph 
#' @param progress If FALSE, the progress bar is not shown (useful for use with sweave or knitr)
#' @description Plot the embryo growth from one or several nests.\cr
#' The embryo.stages is a named vector with relative size as compared to final size at the beginning of the stage. Names are the stages.\cr
#' For example for SCL in Caretta caretta, embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33) indicates that the stages 21 begins at the relative size of 8.4/39.33.\cr
#' Series can be indicated as the name of the series, its number or succesion of TRUE or FALSE. "all" indicates that all series must be printed.\cr
#' show.fioritures parameter does not affect show.test option.\cr
#' Note: three species have predefined embryo stages. embryo.stages parameter can take the values:\cr
#' \itemize{
#'   \item \code{Caretta caretta.SCL}
#'   \item \code{Chelonia mydas.SCL}
#'   \item \code{Emys orbicularis.SCL}
#'   \item \code{Emys orbicularis.mass}
#'   }
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45), series=1,  
#' 	SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479), 
#' 	embryo.stages="Caretta caretta.SCL")
#' # to plot all the nest at the same time, use
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45),  
#' 	series="all", show.fioritures=FALSE, add=TRUE, 
#' 	embryo.stages="Caretta caretta.SCL")
#' # to use color different for series
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45), add=TRUE, 
#' 	series="all", show.fioritures=FALSE, col.S=c(rep("black", 5), rep("red", 6)), 
#' 	embryo.stages="Caretta caretta.SCL")
#' }
#' @method plot NestsResult
#' @export


plot.NestsResult <-
function(x, ..., parameters=NULL, fixed.parameters=NULL, 
         SE=NULL, temperatures=NULL, derivate=NULL, 
         test=NULL, stopattest=FALSE, M0=NULL, weight=NULL, series="all",
         TSP.borders=NULL, embryo.stages=NULL,
         replicate.CI=100, 
         metric.end.incubation=NULL,
         col.stages="blue", col.PT="red", col.TSP="gray", 
         col.temperatures="green", col.S="black", 
         lty.temperatures=1, lwd.temperatures=2, ylimT=NULL, ylimS=NULL, xlim=NULL,
         show.stages=TRUE, show.TSP=TRUE, show.third=TRUE, show.CI=TRUE,  
         show.metric=TRUE, 
         show.fioritures=TRUE, 
         show.temperatures=TRUE, show.PT=TRUE, PT=c(mean=NA, SE=NA), show.test=TRUE, 
         add=FALSE, 
         lab.third="2nd third of incubation", at.lab.third=4, lab.PT="PT", lab.stages="Stages", 
         mar = c(4, 5, 4, 5) + 0.3, 
         xlab="Days of incubation", ylabT=expression("Temperatures in " * degree * "C"), 
         ylabS= "Embryo metric", progress=TRUE) {

#  parameters=NULL; fixed.parameters=NULL; SE=NULL; temperatures=NULL; derivate=NULL; test=NULL; stopattest=FALSE; M0=NULL; weight=NULL; series="all"; TSP.borders=NULL; embryo.stages="Caretta caretta.SCL"; replicate.CI=100; metric.end.incubation=NULL; col.stages="blue"; col.PT="red"; col.TSP="gray"; col.temperatures="green"; col.S="black"; lty.temperatures=1; lwd.temperatures=2; ylimT=NULL; ylimS=NULL; xlim=NULL; show.stages=TRUE; show.TSP=TRUE; show.third=TRUE; show.CI=TRUE;  show.metric=TRUE; show.fioritures=TRUE; show.temperatures=TRUE; show.PT=TRUE; PT=c(mean=NA, SE=NA); show.test=TRUE; add=FALSE; lab.third="2nd third of incubation"; at.lab.third=4; lab.PT="PT"; lab.stages="Stages"; mar = c(4, 5, 4, 5) + 0.3; xlab="Days of incubation"; ylabT=expression("Temperatures in " * degree * "C"); ylabS= "Embryo metric"; progress=TRUE
#  ylimS <- c(0, 35); ylimT <- c(20, 32); x <- resultNest_4p  
  
  TSP.list <- embryogrowth::TSP.list
  
  if(is.null(embryo.stages) & ((show.stages) | (show.TSP))) {
    stop("You must indicate embryo stages.")
  }
  
  if (class(embryo.stages)=="character") {
    estages <- TSP.list[[gsub(" ", "_", embryo.stages)]]
    if (is.null(estages)) {
      stop("The TSP for ", embryo.stages, " does not exist")
    } else {
      embryo.stages <- estages[, "metric"]
      names(embryo.stages) <- estages[, "stages"]
      TSP.borders <- c(attributes(estages)$TSP.begin.stages, attributes(estages)$TSP.end.stages)
    }
  }
  
  
  NestsResult <- x
  
  if (is.null(temperatures)) temperatures <- NestsResult$data
  if (is.null(derivate)) derivate <- NestsResult$derivate
  if (is.null(weight)) weight <- NestsResult$weight
  if (is.null(test)) test <- NestsResult$test
  if (is.null(M0)) M0 <- NestsResult$M0
  if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
  if (is.null(SE)) SE <- NestsResult$SE
  if (is.null(parameters)) parameters <- NestsResult$par
  
  
  if (all(is.na(SE[]))) SE <- NULL
  
  if (is.null(SE)) {
    show.CI <- FALSE
  }
  
# 27/1/2013
if (!show.fioritures) {
	show.PT <- FALSE
	show.temperatures <- FALSE
	show.stages <- FALSE
	show.TSP <- FALSE
	show.CI <- FALSE
	show.third <- FALSE
}
  
  if (!replicate.CI | replicate.CI<2 | !show.CI) {
    replicate.CI <- 1
    show.CI <- FALSE
  }
  
###############################
#### je compte le nombre de series  faire
###############################


NbTS <- temperatures[["IndiceT"]][3]
if (series[[1]]=="all") {
	series<-rep(TRUE, NbTS)
} else {
	if (any(!is.logical(series))) {
		if (is.numeric(series)) {
			seriesx <- rep(FALSE, NbTS)
			seriesx[series] <- TRUE
		} else {
			seriesx <- (names(temperatures[1:NbTS])==series)
		}
		series <- seriesx
	} else {
# c'est des valeurs logiques, je verifie si le bon nombre, sinon je complete
		if (length(series)!=NbTS) {
			series <- rep(series, NbTS)
			series <- series[1:NbTS]
		}
	}
}

nbseries <- sum(series)

if (nbseries==0) {
	stop("No series has been selected")
}

# fais du recyclage des couleurs des sries si ncessaire
cptcol.S <- 0
col.S <- rep(col.S, nbseries)[1:nbseries]

########################
#### Je teste les series
########################
if (progress) pb<-txtProgressBar(min=0, max=NbTS, style=3)

for(seriesx in 1:NbTS) {
  
    if (progress) setTxtProgressBar(pb, seriesx)
  
  if (series[seriesx]) {
    cptcol.S <- cptcol.S+1
    metric.summary <- info.nests(NestsResult=NULL, parameters=parameters, 
                         fixed.parameters=fixed.parameters, 
                         SE=SE, temperatures=temperatures, derivate=derivate, 
                         test=test, stopattest=stopattest, M0=M0, series=seriesx,
                         TSP.borders=TSP.borders, embryo.stages=embryo.stages,
                         replicate.CI=replicate.CI, weight=NULL, out=c("summary", "metric"), 
                         fill=60, 
                         SexualisationTRN=NULL, metric.end.incubation=metric.end.incubation,
                         progress=FALSE)
     par(mar=mar)
     
     troispoints <- list(...)
     # troispoints <- list()
     
     if (add==FALSE | cptcol.S==1) {
      par(new=FALSE)
      L1 <- list(type = "n", axes = TRUE, bty = "n", xlab = xlab, ylab = ylabS, lwd=2, yaxs="r", xaxs="r", ylim=ylimS, xlim=xlim, main="", col=col.S[cptcol.S])
      L2 <- modifyList(list(x=metric.summary$metric[[1]]$Time/(60*24), y=metric.summary$metric[[1]]$SCL), troispoints)
      do.call(plot, modifyList(L1, L2) )
     }
     
     ylimS <- unname(ScalePreviousPlot()$ylim[1:2])
     xlim <- unname(ScalePreviousPlot()$xlim[1:2])
     
     if (show.TSP) {
       par(new=TRUE)
       plot(1,1, xlim=xlim, ylim=ylimS, axes=FALSE, type="n", xlab="", ylab="")
       x1 <- xlim[1]
       x2 <- xlim[2]
       ty1 <- attributes(metric.summary$metric[[1]])$metric.begin.tsp
       ty2 <- attributes(metric.summary$metric[[1]])$metric.end.tsp
       polygon(x=c(x1, x2, x2, x1), y=c(ty1, ty1, ty2, ty2), col=col.TSP, border=NA)
#       text(x=x1+5, y=ty1+(ty2-ty1)/2, labels="TSP")
       x1 <- attributes(metric.summary$metric[[1]])$time.begin.tsp/(24*60)
       x2 <- attributes(metric.summary$metric[[1]])$time.end.tsp/(24*60)
       ty1 <- ylimS[1]
       ty2 <- ylimS[2]
       polygon(x=c(x1, x2, x2, x1), y=c(ty1, ty1, ty2, ty2), col=col.TSP, border=NA)
#       text(x=x1+(x2-x1)/2, y=ty1+5, labels="TSP")
     }
     
     
     if (show.temperatures) {
       # na.omit(metric.summary$metric[[1]][, c("Time", "TempC")])
       x <- na.omit(metric.summary$metric[[1]][, c("Time", "TempC")])[, "Time"]/(60*24)
       y <- na.omit(metric.summary$metric[[1]][, c("Time", "TempC")])[, "TempC"]
       
       Lx <- modifyList(as.list(troispoints), list(x=x, y=y, axes = FALSE, xlab = "", ylab = "", main="")) 
       L <- modifyList(list(type = "l", bty = "n", xlab = "", ylab = "", xlim=xlim, ylim=ylimT, main="", las=1, col=col.temperatures, lty=lty.temperatures, lwd=lwd.temperatures), Lx) 
       par(new=TRUE)
       do.call(plot, L) 
       
       cex.y2 <- par("cex.axis")
       if (!is.null(troispoints$cex.axis)) cex.y2 <- troispoints$cex.axis
       cex.lab2 <- par("cex.lab")
       if (!is.null(troispoints$cex.lab)) cex.lab2 <- troispoints$cex.lab
       
       axis(side=4, ylim=ylimT, las=1, cex.axis=cex.y2)
       mtext(ylabT, side=4, line=3, cex=cex.lab2*par("cex"))
       
       if (show.PT & !is.na(PT[1])) {
         
         segments(0, PT[1], xlim[2]+xlim[2]*0.05, PT[1], lwd=2, col=col.PT, xpd=TRUE)
         text(xlim[2]+6, PT[1], xpd=TRUE, labels=lab.PT, col=col.PT, cex=0.8)
         if (!is.na(PT[2])) {
           segments(0, PT[1]+PT[2]*2, xlim[2]+xlim[2]*0.05, PT[1]+PT[2]*2, lwd=2, col=col.PT, lty=2, xpd=TRUE)
           segments(0, PT[1]-PT[2]*2, xlim[2]+xlim[2]*0.05, PT[1]-PT[2]*2, lwd=2, col=col.PT, lty=2, xpd=TRUE)
         }
         
       }
     }
     
     if (show.TSP) {
       par(new=TRUE)
       plot(1,1, xlim=xlim, ylim=ylimS, axes=FALSE, type="n", xlab="", ylab="")
       x1 <- xlim[1]
       x2 <- xlim[2]
       ty1 <- attributes(metric.summary$metric[[1]])$metric.begin.tsp
       ty2 <- attributes(metric.summary$metric[[1]])$metric.end.tsp
#       polygon(x=c(x1, x2, x2, x1), y=c(ty1, ty1, ty2, ty2), col=col.TSP, border=NA)
       text(x=x1+5, y=ty1+(ty2-ty1)/2, labels="TSP")
       x1 <- attributes(metric.summary$metric[[1]])$time.begin.tsp/(24*60)
       x2 <- attributes(metric.summary$metric[[1]])$time.end.tsp/(24*60)
       ty1 <- ylimS[1]
       ty2 <- ylimS[2]
#       polygon(x=c(x1, x2, x2, x1), y=c(ty1, ty1, ty2, ty2), col=col.TSP, border=NA)
       text(x=x1+(x2-x1)/2, y=ty1+5, labels="TSP")
     }
     
     par(new=TRUE)
     L1 <- list(type = "n", axes = FALSE, bty = "n", xlab = "", ylab = "", yaxs="r", xaxs="r", ylim=ylimS, xlim=xlim, main="")
     L2 <- c(list(x=metric.summary$metric[[1]]$Time/(60*24), y=metric.summary$metric[[1]]$SCL), as.list(troispoints))
     do.call(plot, modifyList(L1, L2) )
     
     if (show.metric) {
      x <- metric.summary$metric[[1]]$Time/(60*24)
       y <- metric.summary$metric[[1]]$SCL
       L1 <- list(type = "l", lwd=2, col=col.S[cptcol.S])
      L2 <- c(list(x=x, y=y), as.list(troispoints))
     
      do.call(plot_add, modifyList(L1, L2) )
     }
     
     if (show.CI) {
       yp <- y+2*metric.summary$metric[[1]]$se.SCL
       L2 <- c(list(x=x, y=yp, lty=2), as.list(troispoints))
       do.call(plot_add, modifyList(L1, L2) )
       yp <- y-2*metric.summary$metric[[1]]$se.SCL
       L2 <- c(list(x=x, y=yp, lty=2), as.list(troispoints))
       do.call(plot_add, modifyList(L1, L2) )
     }
     

     
     if (show.test) {
       mean.ts <- attributes(metric.summary$metric[[1]])$test.mean
       sd.ts <- attributes(metric.summary$metric[[1]])$test.sd
       segments(0, mean.ts, xlim[2]+0.05*xlim[2], mean.ts, lwd=2, xpd=TRUE)
       segments(0, mean.ts-2*sd.ts,  xlim[2]+0.05*xlim[2], mean.ts-2*sd.ts, lwd=1, lty=2, xpd=TRUE)
       segments(0, mean.ts+2*sd.ts,  xlim[2]+0.05*xlim[2], mean.ts+2*sd.ts, lwd=1, lty=2, xpd=TRUE)
     }
     
     if (show.third) {
       x1 <- attributes(metric.summary$metric[[1]])$time.begin.middlethird/(24*60)
       x2 <- attributes(metric.summary$metric[[1]])$time.end.middlethird/(24*60)
       ty1 <- ylimS[1]
       ty2 <- ylimS[2]+1
       
           segments(x1, ty1, x1, ty2, lwd=1, lty=3, xpd=TRUE)
           segments(x2, ty1, x2, ty2, lwd=1, lty=3, xpd=TRUE)
           
           arrows(x1, ylimS[2]+1, x2, ylimS[2]+1, code=3, xpd=TRUE)
           text(x1+(x2-x1)/2, ylimS[2]+at.lab.third, labels=lab.third, xpd=TRUE)
     }
     
     
     if (show.stages) {
       ## on affiche les stades
       
       for(i in 1:length(embryo.stages)) {
         y1=embryo.stages[i]*attributes(metric.summary$metric[[1]])$metric.end.incubation
         segments(0, y1, xlim[2]-4, y1, lwd=1, lty=2, col=col.stages)
         text(xlim[2]-3*(as.numeric(i)%%2), y1, labels=names(embryo.stages[i]), col=col.stages, cex=0.7)
       }
       par(xpd=TRUE)
       text(xlim[2]-2, ylimS[2]+3, labels=lab.stages, col=col.stages, cex=0.7)
     }
     
 
     
     # je rtablis l'axe de la mtrique
     #  par(new=TRUE)
     #  L1 <- list(type = "n", axes = FALSE, bty = "n", xlab = "", ylab = "", yaxs="r", xaxs="r", ylim=ylimS, xlim=xlim, main="")
     #  L2 <- c(list(x=metric.summary$metric[[1]]$Time/(60*24), y=metric.summary$metric[[1]]$SCL), as.list(troispoints))
     #  do.call(plot, modifyList(L1, L2) )
     
     
     # fin de test si on doit faire la srie
  }
  
  # fin des sries
}

# fin de la fonction
}


