% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynsimGG.R
\name{dynsimGG}
\alias{dynsimGG}
\title{Plot dynamic simulation results from \code{dynsim}}
\usage{
dynsimGG(obj, lsize = 1, color, alpha = 0.5, xlab = "\\nTime",
  ylab = "Predicted Value\\n", title = "", leg.name = "Scenario",
  leg.labels, legend = "legend", shockplot.var, shockplot.ylab,
  shockplot.heights = c(12, 4), shockplot.heights.units = c("cm", "cm"))
}
\arguments{
\item{obj}{a \code{dynsim} class object.}

\item{lsize}{size of the smoothing line. Default is 1. See
ggplot2.}

\item{color}{character string. Specifies the color of the lines and ribbons.
If only one scenario is to be plotted then it can either be a single color
value using any color value allowed by ggplot2. The default is
the hexadecimal color \code{"#2B8CBE"}. If more than one scenario is to be
plotted then a color brewer palette is set. The default is\code{"Set1"}. See
\code{\link{scale_colour_brewer}}.}

\item{alpha}{numeric. Alpha (e.g. transparency) for the ribbons. Default is
\code{alpha = 0.1}. See ggplot2.}

\item{xlab}{a label for the plot's x-axis.}

\item{ylab}{a label of the plot's y-axis.}

\item{title}{the plot's main title.}

\item{leg.name}{name of the legend (if applicable).}

\item{leg.labels}{character vector specifying the labels for each scenario in
the legend.}

\item{legend}{specifies what type of legend to include (if applicable). The
default is \code{legend = "legend"}. To hide the legend use
\code{legend = FALSE}. See \code{\link{discrete_scale}} for more details.}

\item{shockplot.var}{character string naming the one shock variable to plot
fitted values of over time specified underneath the main plot.}

\item{shockplot.ylab}{character string for the shockplot's y-axis label.}

\item{shockplot.heights}{numeric vector with of length 2 with units of
the main and shockplot height plots.}

\item{shockplot.heights.units}{a character vector of length 2 with the
unit types for the values in \code{shockplot.heights}.
See \code{\link{unit}} for details.}
}
\description{
\code{dynsimGG} uses ggplot2 to plot dynamic simulation results
created by \code{\link{dynsim}}.
}
\details{
Plots dynamic simulations of autoregressive relationships from
\code{\link{dynsim}}. The central line is the mean of the simulation
distributions. The outer ribbon is the furthest extent of the simulation
distributions' central intervals found in \code{\link{dynsim}} with the
\code{sig} argument. The middle ribbons plot the limits of the simulation
distributions' central 50% intervals.
}
\examples{
# Load package
library(DataCombine)

# Load Grunfeld data
data(grunfeld, package = "dynsim")

# Create lag invest variable
grunfeld <- slide(grunfeld, Var = "invest", GroupVar = "company",
               NewVar = "InvestLag")

# Convert company to factor for fixed-effects specification
grunfeld$company <- as.factor(grunfeld$company)
# Estimate basic model
M1 <- lm(invest ~ InvestLag + mvalue + kstock + company, data = grunfeld)

# Set up scenarios for company 4
attach(grunfeld)
Scen1 <- data.frame(InvestLag = mean(InvestLag, na.rm = TRUE),
                    mvalue = quantile(mvalue, 0.05),
                    kstock = quantile(kstock, 0.05),
                    company4 = 1)
Scen2 <- data.frame(InvestLag = mean(InvestLag, na.rm = TRUE),
                    mvalue = mean(mvalue),
                    kstock = mean(kstock),
                    company4 = 1)
Scen3 <- data.frame(InvestLag = mean(InvestLag, na.rm = TRUE),
                    mvalue = quantile(mvalue, 0.95),
                    kstock = quantile(kstock, 0.95),
                    company4 = 1)
detach(grunfeld)

# Combine into a single list
ScenComb <- list(Scen1, Scen2, Scen3)

## Run dynamic simulations without shocks
Sim1 <- dynsim(obj = M1, ldv = "InvestLag", scen = ScenComb, n = 20)

# Create plot legend label
Labels <- c("5th Percentile", "Mean", "95th Percentile")

# Plot
dynsimGG(Sim1, leg.labels = Labels)

## Run dynamic simulations with shocks

# Create data frame of shock values
mShocks <- data.frame(times = c(5, 10), kstock = c(100, 1000))

# Run simulations
Sim2 <- dynsim(obj = M1, ldv = "InvestLag", scen = ScenComb, n = 20,
               shocks = mShocks)

# Plot
dynsimGG(Sim2, leg.labels = Labels)

# Plot with accompanying shock plot
dynsimGG(Sim2, leg.labels = Labels, shockplot.var = "kstock")

}

