% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_dose_draw.R
\name{f_dose_draw_1}
\alias{f_dose_draw_1}
\title{Drug Dispensing Data Simulation for One Iteration}
\usage{
f_dose_draw_1(
  i,
  common_time_model,
  fit_k0,
  fit_t0,
  fit_t1,
  fit_ki,
  fit_ti,
  fit_di,
  vf_ongoing,
  vf_ongoing1,
  vf_new1,
  treatment_by_drug_df,
  l,
  t
)
}
\arguments{
\item{i}{The iteration number.}

\item{common_time_model}{A Boolean variable that indicates whether
a common time model is used for drug dispensing visits.}

\item{fit_k0}{The model fit for the number of skipped
visits between randomization and the first drug dispensing visit.}

\item{fit_t0}{The model fit for the gap time between randomization
and the first drug dispensing visit when there is no visit skipping.}

\item{fit_t1}{The model fit for the gap time between randomization
and the first drug dispensing visit when there is visit skipping.}

\item{fit_ki}{The model fit for the number of skipped
visits between two consecutive drug dispensing visits.}

\item{fit_ti}{The model fit for the gap time between two
consecutive drug dispensing visits.}

\item{fit_di}{The model fit for the dispensed doses at drug
dispensing visits.}

\item{vf_ongoing}{The observed drug dispensing data for ongoing
patients with drug dispensing records. It includes the following
variables: \code{draw}, \code{drug}, \code{drug_name}, \code{dose_unit},
\code{usubjid}, \code{arrivalTime}, \code{treatment},
\code{treatment_description}, \code{time}, \code{totalTime},
\code{V}, \code{C}, and \code{D}.}

\item{vf_ongoing1}{A data frame for the last observed drug dispensing
date for ongoing patients with drug dispensing records, with or without
the associated drug information. For the common time model, it includes
the following variables: \code{draw}, \code{usubjid},
\code{arrivalTime}, \code{treatment}, \code{treatment_description},
\code{time}, \code{totalTime}, \code{V}, \code{C}, and \code{D}.
For separate time models, it includes the following variables:
\code{draw}, \code{drug}, \code{drug_name}, \code{dose_unit},
\code{usubjid}, \code{arrivalTime}, \code{treatment},
\code{treatment_description}, \code{time}, \code{totalTime},
\code{V}, \code{C}, and \code{D}.}

\item{vf_new1}{A data frame for the randomization date for new patients
and ongoing patients with no drug dispensing records, with or without the
associated drug information. For the common time model, it includes
the following variables: \code{draw}, \code{usubjid},
\code{arrivalTime}, \code{treatment}, \code{treatment_description},
\code{time}, \code{totalTime}, \code{V}, \code{C}, and \code{D}.
For separate time models, it includes the following variables:
\code{draw}, \code{drug}, \code{drug_name}, \code{dose_unit},
\code{usubjid}, \code{arrivalTime}, \code{treatment},
\code{treatment_description}, \code{time}, \code{totalTime},
\code{V}, \code{C}, and \code{D}.}

\item{treatment_by_drug_df}{A data frame indicating the treatments
associated with each drug, including the following variables:
\code{treatment}, \code{drug}, \code{drug_name}, and
\code{dose_unit}.}

\item{l}{Number of drugs.}

\item{t}{A vector of new time points for drug dispensing prediction.}
}
\value{
A list of two components:
\itemize{
\item \code{dosing_subject_newi}: A data frame for the drug dispensing
data at the subject level by date for ongoing and new subjects
for the given iteration. It contains the following variables:
\code{draw}, \code{drug}, \code{drug_name}, \code{dose_unit},
\code{usubjid}, \code{day}, \code{dose}, \code{arrivalTime},
\code{treatment}, \code{treatment_description}, \code{time},
and \code{totalTime}.
\item \code{dosing_summary_newi}: A data frame for the drug dispensing
summary data by drug, time, and simulation draw for ongoing and
new subjects for the given iteration. It includes the following
variables: \code{drug}, \code{drug_name}, \code{dose_unit},
\code{t}, \code{draw}, and \code{total_dose_b}.
}
}
\description{
Simulates drug dispensing data for one iteration.
}
\examples{

\donttest{
set.seed(431)
library(dplyr)

df <- df2 \%>\%
  mutate(arrivalTime = as.numeric(randdt - trialsdt + 1))

vf <- visitview2 \%>\%
  inner_join(df, by = "usubjid") \%>\%
  mutate(day = as.numeric(date - randdt + 1)) \%>\%
  select(drug, drug_name, dose_unit, usubjid, treatment,
         treatment_description, arrivalTime,
         time, event, dropout, day, dispensed_quantity) \%>\%
  group_by(drug, drug_name, dose_unit, usubjid, treatment,
           treatment_description, arrivalTime,
           time, event, dropout, day) \%>\%
  summarise(dose = sum(dispensed_quantity), .groups = "drop_last") \%>\%
  mutate(cum_dose = cumsum(dose)) \%>\%
  group_by(drug, drug_name, dose_unit, usubjid) \%>\%
  mutate(row_id = row_number())

pred <- eventPred::getPrediction(
  df = df,
  to_predict = "event only",
  target_d = 250,
  event_model = "log-logistic",
  dropout_model = "none",
  pilevel = 0.95,
  nyears = 3,
  nreps = 200,
  showsummary = FALSE,
  showplot = FALSE,
  by_treatment = TRUE)
newEvents <- pred$event_pred$newEvents

treatment_by_drug_df <- vf \%>\%
  group_by(treatment, drug, drug_name, dose_unit) \%>\%
  slice(n()) \%>\%
  select(treatment, drug, drug_name, dose_unit)

fit <- f_dispensing_models(
  vf, dosing_schedule_df,
  model_k0 = "zero-inflated poisson", model_t0 = "log-logistic",
  model_t1 = "least squares",
  model_ki = "zero-inflated poisson", model_ti = "least squares",
  model_di = "linear mixed-effects model",
  nreps = 200, showplot = FALSE)

trialsdt = df$trialsdt[1]
cutoffdt = df$cutoffdt[1]
t0 = as.numeric(cutoffdt - trialsdt + 1)
nyears = 3
t1 = t0 + nyears*365
t = c(seq(t0, t1, 30), t1)

nreps = length(unique(newEvents$draw))
l = length(unique(treatment_by_drug_df$drug))

### dosing data for ongoing patients ###
vf1 <- vf \%>\%
  filter(event == 0) \%>\%
  select(drug, drug_name, dose_unit, usubjid, day, dose)

# ongoing subjects with dosing records
unames <- unique(vf1$usubjid)

# replicate nreps times
vf1_rep = tibble(draw = 1:nreps) \%>\%
  cross_join(vf1)

df1 <- newEvents \%>\%
  filter(usubjid \%in\% unames) \%>\%
  select(-c(event, dropout))

vf_ongoing <- vf1_rep \%>\%
  inner_join(df1, by = c("draw", "usubjid"))

### new patients and ongoing patients with no dosing records ###
df_new <- newEvents \%>\%
  filter(!(usubjid \%in\% unames))

vf_new <- purrr::map_dfr(
  1:l, function(h) {
    df_new \%>\%
      inner_join(treatment_by_drug_df \%>\% filter(drug == h),
                 by = "treatment")
  }) \%>\% select(-c(event, dropout))

# only keep the last record for each patient in each draw
vf_ongoing1 <- vf_ongoing \%>\%
  group_by(draw, usubjid) \%>\%
  slice(n()) \%>\%
  mutate(V = day - 1,
         C = as.numeric(t0 - arrivalTime),
         D = pmin(time - 1, t1 - arrivalTime)) \%>\%
  select(-c(drug, drug_name, dose_unit, day, dose))

### new patients and ongoing patients with no dosing records ###
vf_new1 <- vf_new \%>\%
  group_by(draw, usubjid) \%>\%
  slice(n()) \%>\%
  mutate(V = 0,
         C = as.numeric(t0 - arrivalTime),
         D = pmin(time - 1, t1 - arrivalTime)) \%>\%
  select(-c(drug, drug_name, dose_unit))

# first iteration to extract subject and summary data
list1 <- f_dose_draw_1(
  1, fit$common_time_model,
  fit$fit_k0, fit$fit_t0, fit$fit_t1,
  fit$fit_ki, fit$fit_ti, fit$fit_di,
  vf_ongoing, vf_ongoing1, vf_new1,
  treatment_by_drug_df, l, t)

head(list1$dosing_subject_newi)
head(list1$dosing_summary_newi)
}

}
\seealso{
\code{\link{f_fit_t0}}, \code{\link{f_fit_ki}},
\code{\link{f_fit_ti}}, \code{\link{f_fit_di}}
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
