#' Relabel the Predictors in a Tidy Data Frame of Regression Results
#'
#' \code{relabel_predictors} is a convenience function for relabeling the predictors in a tidy data.frame to be passed to \code{\link[dotwhisker]{dwplot}}
#'
#' @param x Either a plot generated by \code{\link[dotwhisker]{dwplot}} or a tidy data.frame to be passed to \code{\link[dotwhisker]{dwplot}}
#' @param replace A named character vector, with new values as values, and old values as names
#'
#' @return The function returns a tidy data.frame.
#'
#' @examples
#' library(broom)
#' library(dplyr)
#'
#' data(mtcars)
#' m1 <- lm(mpg ~ wt + cyl + disp, data = mtcars)
#' m1_df <- broom::tidy(m1) %>%
#'          relabel_predictors(c("(Intercept)" = "Intercept",
#'                               wt = "Weight",
#'                               cyl = "Cylinder",
#'                               disp = "Displacement"))
#' dwplot(m1_df)
#'
#' dwplot(m1, show_intercept = TRUE) %>%
#'     relabel_predictors(c("(Intercept)" = "Intercept",
#'                               wt = "Weight",
#'                               cyl = "Cylinder",
#'                               disp = "Displacement"))
#'
#'
#' @importFrom plyr revalue
#'
#' @export

relabel_predictors <- function(x, replace = NULL) {
    if (is.data.frame(x)) {
        x$term <- plyr::revalue(x$term, replace = replace)
        return(x)
    }
    else if (is.ggplot(x)) {
        x$data$term <- plyr::revalue(x$data$term, replace = replace)
        x <- dwplot(x$data)
        return(x)
    } else stop("x should be either a tidy data.frame or a plot generated by dwplot")
}
