#' External validity indices
#'
#' \strong{E}xternal \strong{v}alidity indices compare a predicted clustering
#' result with a reference class or gold standard.
#'
#' `ev_nmi` calculates the normalized mutual information
#'
#' @param pred.lab predicted labels generated by classifier
#' @param ref.lab reference labels for the observations
#' @param method method of computing the entropy. Can be any one of "emp", "mm",
#'   "shrink", or "sg".
#'
#' @return `ev_nmi` returns the normalized mutual information.
#' @references Strehl A, Ghosh J. Cluster ensembles: a knowledge reuse framework
#'   for combining multiple partitions. J. Mach. Learn. Res. 2002;3:583-617.
#' @note `ev_nmi` is adapted from [infotheo::mutinformation()]
#' @author Johnson Liu, Derek Chiu
#' @name external_validity
#' @export
#'
#' @examples
#' set.seed(1)
#' E <- matrix(rep(sample(1:4, 1000, replace = TRUE)), nrow = 100, byrow =
#'               FALSE)
#' x <- sample(1:4, 100, replace = TRUE)
#' y <- sample(1:4, 100, replace = TRUE)
#' ev_nmi(x, y)
#' ev_confmat(x, y)
ev_nmi <- function(pred.lab, ref.lab, method = "emp") {
  U <- data.frame(ref.lab, pred.lab)
  Hyx <- infotheo::entropy(U, method)
  Hx <- infotheo::entropy(pred.lab, method)
  Hy <- infotheo::entropy(ref.lab, method)
  I <- ifelse(Hx + Hy - Hyx < 0, 0, Hx + Hy - Hyx)
  NMI <- I / sqrt(Hx * Hy)
  NMI
}

#' @details `ev_confmat` calculates a variety of statistics associated with
#'   confusion matrices.
#'
#' @return `ev_confmat` returns a vector of the following metrics: overall
#'   accuracy, Cohen's kappa, no information rate, accuracy p-value. Statistics
#'   are difficult to compare when there are multiclass comparisons. We hence
#'   also report averaged statistics for: sensitivity, specificity, PPV, NPV,
#'   prevalence, detection rate, detection prevalence, accuracy, and balanced
#'   accuracy.
#' @rdname external_validity
#' @export
ev_confmat <- function(pred.lab, ref.lab) {
  if (!all(unique(pred.lab) %in% unique(ref.lab))) {
    stop("Cluster labels should be the same in the predicted and reference
         classes.")
  }
  # Relabel predicted classes
  pred.relab <- relabel_class(pred.lab, ref.lab) %>%
    factor(levels = sort(unique(ref.lab)))

  # Confusion matrix, column/row sums, total, true/false positives/negatives
  CM <- table(pred.relab, ref.lab)
  clm <- colSums(CM)
  rwm <- rowSums(CM)
  N <-  sum(CM)
  TP <- diag(CM)
  FP <- clm - TP
  FN <- rwm - TP
  TN <- N - (TP + FP + FN)

  # Overall confusion matrix statistics
  overall <- caret::confusionMatrix(CM) %>%
    magrittr::use_series(overall) %>%
    magrittr::extract(c("Accuracy", "Kappa", "AccuracyNull",
                        "AccuracyPValue")) %>%
    magrittr::set_names(c("Overall Accuracy", "Cohen's kappa",
                          "No Information Rate", "P-Value [Acc > NIR]"))

  # Combine with averaged statistics
  avgd_stats <- data.frame(TP, TN, clm, rwm) %>%
    dplyr::transmute_(.dots = stats::setNames(
      list(~TP / clm, ~TN / (N - clm), ~TP / rwm, ~TN / (N - rwm), ~TP / N,
           ~(TP + TN) / N, ~(Sensitivity + Specificity) / 2),
      c("Sensitivity", "Specificity", "PPV", "NPV", "Detection Rate",
        "Accuracy", "Balanced Accuracy"))) %>%
    colMeans() %>%
    magrittr::set_names(paste("Average", names(.)))

  c(overall, avgd_stats)
}
