% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_cashflow.R
\name{plot_cashflow}
\alias{plot_cashflow}
\title{Cashflow plot for Monte Carlo simulation results}
\usage{
plot_cashflow(
  mcSimulation_object,
  cashflow_var_name,
  x_axis_name = "Timeline of intervention",
  y_axis_name = "Cashflow",
  legend_name = "Quantiles (\%)",
  legend_labels = c("5 to 95", "25 to 75", "median"),
  color_25_75 = "grey40",
  color_5_95 = "grey70",
  color_median = "blue",
  facet_labels = cashflow_var_name,
  base_size = 11,
  ...
)
}
\arguments{
\item{mcSimulation_object}{is a data frame of Monte Carlo simulations of cashflow outputs (in wide format, see example). Usually the "mcSimulation_object.csv" file generated by \code{\link[decisionSupport:mcSimulation]{mcSimulation}} function.}

\item{cashflow_var_name}{is the name (character string) for the variable name used to define cashflow in the returned list of outputs from the \code{\link[decisionSupport:mcSimulation]{mcSimulation}} function. If multiple decision options are provided this will produce a comparative panel plot.}

\item{x_axis_name}{is the name (character string) for the title of the timeline of the intervention to be printed on the x axis in quotes.}

\item{y_axis_name}{is the name (character string) for the title of the units of the cashflow to be printed on the y axis.}

\item{legend_name}{is the name (character string) for the title of the legend}

\item{legend_labels}{is the name (character string) for the labels of the legend. The default is inner, outer and median quantiles, i.e. `c("5 to 95", "25 to 75", "median")` and replacements should follow the same order}

\item{color_25_75}{is the color for the shade fill of the 25-75\% quantile from the grDevices colors. The default is "grey40".}

\item{color_5_95}{is the color for the shade fill of the 5-95\% quantile from the grDevices colors. The default is "grey70".}

\item{color_median}{is the color for the median line from the grDevices colors. The default is  "blue".}

\item{facet_labels}{are the names (character string) for the decisions. The default is the cashflow_var_name parameter.}

\item{base_size}{is the base text size to be used for the plot. The default is 11, this is the \code{\link[ggplot2:ggtheme]{ggplot2::ggtheme}} default}

\item{...}{accepts arguments to be passed to \code{\link[ggplot2:ggtheme]{ggplot2::ggtheme}}}
}
\value{
This function returns a plot of classes \code{'gg'}, 
and \code{'ggplot'}. This allows the user to
continue editing some features of the plots through the syntax 
\code{'+'}.
}
\description{
Creates a cashflow plot of the returned list of related outputs from the \code{\link[decisionSupport:mcSimulation]{mcSimulation}} function using \code{\link{ggplot2}}
}
\details{
This function automatically defines quantiles (5 to 95\% and 25 to 75\%) as well as a value for the median.
}
\examples{
 
# Plotting the cashflow:

# Create the estimate object (for multiple options):

variable = c("revenue_option1", "costs_option1", "n_years", 
             "revenue_option2", "costs_option2")
distribution = c("norm", "norm", "const", "norm", "norm")
lower = c(10000,  5000, 10, 8000,  500)
upper = c(100000, 50000, 10, 80000,  30000)

costBenefitEstimate <- as.estimate(variable, distribution, lower, upper)

# Define the model function without name for the return value:

profit1 <- function(x) {
  
cashflow_option1 <- vv(revenue_option1 - costs_option1, n = n_years, var_CV = 100)
cashflow_option2 <- vv(revenue_option2 - costs_option2, n = n_years, var_CV = 100)

return(list(Revenues_option1 = revenue_option1,
            Revenues_option2 = revenue_option2,
            Costs_option1 = costs_option1,
            Costs_option2 = costs_option2,
            Cashflow_option_one = cashflow_option1,
            Cashflow_option_two = cashflow_option2))
}

# Perform the Monte Carlo simulation:

predictionProfit1 <- mcSimulation(estimate = costBenefitEstimate,
                                  model_function = profit1,
                                  numberOfModelRuns = 10000,
                                  functionSyntax = "plainNames")


# Plot the cashflow distribution over time

plot_cashflow(mcSimulation_object = predictionProfit1, cashflow_var_name = "Cashflow_option_one",
              x_axis_name = "Years with intervention",
              y_axis_name = "Annual cashflow in USD",
              color_25_75 = "green4", color_5_95 = "green1",
              color_median = "red")

##############################################################
# Example 2 (Plotting the cashflow with panels for comparison):

# Compare the cashflow distribution over time for multiple decision options  

plot_cashflow(mcSimulation_object = predictionProfit1, 
              cashflow_var_name = c("Cashflow_option_one", "Cashflow_option_two"),
              x_axis_name = "Years with intervention",
              y_axis_name = "Annual cashflow in USD",
              color_25_75 = "green4", color_5_95 = "green1",
              color_median = "red", 
              facet_labels = c("Option 1", "Option 2"))
  
}
\references{
Lanzanova Denis, Cory Whitney, Keith Shepherd, and Eike Luedeling. “Improving Development Efficiency through Decision Analysis: Reservoir Protection in Burkina Faso.” Environmental Modelling & Software 115 (May 1, 2019): 164–75. \doi{10.1016/j.envsoft.2019.01.016}.
}
\author{
Eduardo Fernandez (\email{efernand@uni-bonn.de})

Cory Whitney (\email{cory.whitney@uni-bonn.de})
}
\keyword{Monte-Carlo}
\keyword{cashflow}
\keyword{decision-analysis}
\keyword{decisionSupport}
\keyword{risk}
\keyword{uncertainty}
