#' Download Asset From A Databrary Session.
#'
#' @description Databrary stores file types (assets) of many types. This
#' function downloads assets in a data frame generated by list_session_assets().
#'
#' @param session_df A data frame as generated by list_session_assets_2().
#' @param target_dir A character string. Directory to save the downloaded file.
#' Default is directory named after the session_id.
#' @param overwrite A logical value. Overwrite an existing file. Default is TRUE.
#' @param make_portable_fn A logical value. Replace characters in file names
#' that are not broadly portable across file systems. Default is FALSE.
#' @param timeout_secs An integer. The seconds an httr2 request will run before
#' timing out. Default is 600 (10 min). This is to handle very large files.
#' @param vb A logical value. If TRUE provides verbose output. Default is FALSE.
#' @param rq A list in the form of an `httr2` request object. Default is NULL.
#'
#' @returns Full file names to the downloaded assets or NULL.
#'
#' @examples
#' \donttest{
#' \dontrun{
#' download_session_assets_fr_df() # Downloads all of the files from session
#' 9807 in Databrary volume 1.
#'
#' # Just the CSVs
#' v1 <- list_session_assets()
#' v1_csv <- dplyr::filter(v1, format_extension == "csv")
#' download_session_assets_fr_df(v1_csv, vb = TRUE)
#' }
#' }
#' @export
download_session_assets_fr_df <-
  function(session_df = list_session_assets(),
           target_dir = tempdir(),
           overwrite = TRUE,
           make_portable_fn = FALSE,
           timeout_secs = 600,
           vb = FALSE,
           rq = NULL) {
    # Check parameters
    assertthat::assert_that(is.data.frame(session_df))
    assertthat::assert_that("session_id" %in% names(session_df))
    assertthat::assert_that("session_id" %in% names(session_df))
    assertthat::assert_that("asset_id" %in% names(session_df))
    assertthat::assert_that("format_extension" %in% names(session_df))
    assertthat::assert_that("asset_name" %in% names(session_df))
    
    assertthat::assert_that(length(target_dir) == 1)
    assertthat::assert_that(is.character(target_dir))
    if (!dir.exists(target_dir)) {
      if (vb) {
        message("Target directory not found: ", target_dir)
        message("Creating: ", target_dir)
      }
      dir.create(target_dir, recursive = TRUE)
    } else {
      if (vb)
        message("Target directory exists: ", target_dir)
      if (overwrite) {
        if (vb)
          message("`overwrite` is TRUE. Overwriting directory: ", target_dir)
      } else {
        if (vb)
          message("`overwrite` is FALSE. Cannot continue.")
        return(NULL)
      }
    }
    assertthat::is.writeable(target_dir)
    
    assertthat::assert_that(length(overwrite) == 1)
    assertthat::assert_that(is.logical(overwrite))
    
    assertthat::assert_that(length(make_portable_fn) == 1)
    assertthat::assert_that(is.logical(make_portable_fn))
    
    assertthat::is.number(timeout_secs)
    assertthat::assert_that(length(timeout_secs) == 1)
    assertthat::assert_that(timeout_secs > 0)
    
    assertthat::assert_that(length(vb) == 1)
    assertthat::assert_that(is.logical(vb))
    
    assertthat::assert_that(is.null(rq) |
                              ("httr2_request" %in% class(rq)))
    
    if (vb)
      message("Downloading n=",
              dim(session_df)[1],
              " files to /",
              target_dir)
    purrr::map(
      1:dim(session_df)[1],
      download_single_session_asset_fr_df,
      session_df,
      target_dir = target_dir,
      overwrite = overwrite,
      make_portable_fn = make_portable_fn,
      timeout_secs = timeout_secs,
      vb = vb,
      rq = rq,
      .progress = TRUE
    ) |>
      purrr::list_c()
    #fns
  }
