\name{dlm}
\alias{dlm}

\title{
Implement finite distributed lag model
}
\description{
Applies distributed lag models with one or multiple predictor(s).
}
\usage{
dlm(formula , data , x , y , q , remove )
}

\arguments{
  \item{formula}{A \code{formula} object for the model to be fitted. In the case of multiple predictor series, the model should be entered via a formula object. 
  }
  \item{data}{  A \code{data.frame} including all dependent and independent series. In the case of multiple predictor series, the data should be entered via the \code{data} argument.
  }
  \item{x}{
  A vector including the observations of predictor time series. This is not restricted to \code{ts} objects. If the series are supplied by \code{data} 
  }
  \item{y}{
  A vector including the observations of dependent time series. This is not restricted to \code{ts} objects.
  } 
  \item{q}{
  An integer representing finite lag length.
  }  
  \item{remove}{
  A list object showing the lags to be removed from the model for each independent series in its elements. Please see the details for the construction of this argument.
  }
 
}

\details{
When a decision made on a variable, some of the related variables would be effected through time. For example, when income tax rate is increased, this would reduce expenditures of consumers on goods and services, which reduces profits of suppliers, which reduces the demand for productive inputs, which reduces the profits of the input suppliers, and so on (Judge and Griffiths, 2000). These effects occur over the future time periods; hence, they are distributed across the time.    

In a distributed-lag model, the effect of an independent variable \eqn{X} on a dependent variable \eqn{Y} occurs over the time. Therefore, DLMs are dynamic models. A linear finite DLM with one independent variable is written as follows:


\deqn{
      Y_{t} = \alpha +\sum_{s = 0}^{q}\beta_{s}X_{t-s}+\epsilon_{t},
      }

where \eqn{\epsilon_{t}} is a stationary error term with \eqn{E(\epsilon_{t})=0, Var(\epsilon_{t})=\sigma^{2},Cov(\epsilon_{t},\epsilon_{s})=0}.

When there is only one predictor series, both of \code{model} and \code{formula} objects can be used. But when they are supplied, both \code{x} and \code{y} arguments should be \code{NULL}.

The variable names in \code{formula} must match with the names of variables in \code{data} argument and it must be in the form of a generic formula for R functions. 

The argument \code{data} contains dependent series and independent series. Required lags of dependent series are generated by the \code{dlm} function automatically. 

The argument \code{remove = list()} is used to specify which lags will be removed from the full model. Each element of the list \code{remove} shows particular lags that will be removed from each independent series. Notice that it is possible to fit a model with different lag lengths for each independent series by removing the appropriate lags of independent series with \code{remove} argument. To remove the main series from the model include \code{0} within the elements of \code{remove}.

To remove the intercept from the model, if a \code{formula} is entered, just include "-1" in the model formula. Otherwise, include "-1" in the element \code{remove} of the list \code{remove}. See the examples below for implementation details.

The standard function \code{summary()} prints model summary for the model of interest.

}

\value{

\item{model}{An object of class \code{lm}.}

\item{designMatrix}{The design matrix composed of transformed z-variables.}

\item{k}{The number of independent series. This is returned if multiple independent series are entered.}

\item{q}{The lag length.}

\item{removed}{A list or vector showing the removed lags from the model for independent series. Returns \code{NULL} if the fitted model is full.}

\item{formula}{Model formula of the fitted model. This is returned if multiple independent series are entered.}

\item{data}{A \code{data.frame} including all dependent and independent series. This is returned if multiple independent series are entered.}
}

\references{
B.H. Baltagi. \emph{Econometrics}, Fifth Ed. Springer, 2011.

R.C. Hill, W.E. Griffiths, G.G. Judge. \emph{Undergraduate Econometrics}. Wiley, 2000. 
}

\author{
Haydar Demirhan

Maintainer: Haydar Demirhan <haydar.demirhan@rmit.edu.au>
}

\examples{
# Only one independent series
data(seaLevelTempSOI)
model.dlm = dlm(x = seaLevelTempSOI$LandOcean,  
                y = seaLevelTempSOI$GMSL , q = 5)
summary(model.dlm)
residuals(model.dlm)
coef(model.dlm)
fitted(model.dlm)

removed = list(x = c(1))
model.dlm = dlm(x = seaLevelTempSOI$LandOcean,  
                y = seaLevelTempSOI$GMSL , q = 5,
                remove = removed)
summary(model.dlm)

# Remove intercept as well
removed = list(x = c(1,-1))
model.dlm = dlm(x = seaLevelTempSOI$LandOcean,  
                y = seaLevelTempSOI$GMSL , q = 5,
                remove = removed)
summary(model.dlm)

removed = list(x = c(0,1))
model.dlm = dlm(x = seaLevelTempSOI$LandOcean,  
                y = seaLevelTempSOI$GMSL , q = 5,
                remove = removed)
summary(model.dlm)

model.dlm = dlm(formula =  GMSL ~ LandOcean , 
                 data = seaLevelTempSOI , q = 5)
summary(model.dlm)

removed = list(x = c(0,1))
model.dlm = dlm(formula =  GMSL ~ LandOcean , 
               data = seaLevelTempSOI , q = 5,
                remove = removed)
summary(model.dlm)

# Remove intercept as well
removed = list(x = c(0,-1))
model.dlm = dlm(formula =  GMSL ~ LandOcean , 
                data = seaLevelTempSOI , q = 2,
                remove = removed)
summary(model.dlm)

# Multiple independent series
data(M1Germany)
data = M1Germany[1:144,]
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
                 data = data.frame(data) , q = 4)
summary(model.dlm)

removed = list(interest = c(1,3), logm1 = c(2))
removed
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
                 data = data.frame(data) , q = 4 , remove = removed)
summary(model.dlm)

removed = list(interest = c(0,1,3), logm1 = c(0,2))
removed
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
                 data = data.frame(data) , q = 4 , remove = removed)
summary(model.dlm)

removed = list( logm1 = c(1,2))
removed
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
                 data = data.frame(data) , q = 4 , remove = removed)
summary(model.dlm)
}
