% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.cure.model.R
\name{fit.cure.model}
\alias{fit.cure.model}
\title{Parametric cure model}
\usage{
fit.cure.model(
  formula,
  data,
  formula.surv = NULL,
  type = c("mixture", "nmixture"),
  dist = c("weibull", "exponential", "lognormal", "weiwei", "weiexp", "gmw"),
  link = c("logit", "loglog", "identity", "probit"),
  bhazard = NULL,
  covariance = TRUE,
  link.mix = c("logit", "loglog", "identity", "probit"),
  control = list(maxit = 10000),
  method = "Nelder-Mead",
  init = NULL
)
}
\arguments{
\item{formula}{Formula for modelling the cure proportion. The left hand side
has to be of the form \code{Surv(time, status)}.}

\item{data}{Data frame in which to interpret the variable names in \code{formula} and \code{formula.surv}.}

\item{formula.surv}{List of formulas for each parameter in the parametric distribution (see details).}

\item{type}{A character indicating the type of cure model.
Possible values are \code{mixture} (default) and \code{nmixture}.}

\item{dist}{The parametric distribution of the survival of the uncured.}

\item{link}{Character. Specifies the link function of the cure proportion.}

\item{bhazard}{Background hazard.}

\item{covariance}{Logical. If \code{TRUE} (default), the covariance matrix is computed.}

\item{link.mix}{Character. Specifies the link function for the mixture parameter in a
weibull-weibull mixture model and weibull-exponential model.\cr Only used when \code{dist = "weiwei"} and \code{dist = "weiexp"}.}

\item{control}{List of control parameters passed to \code{optim}.}

\item{method}{Optimization method passed to \code{optim}.}

\item{init}{Initial values for the maximum likelihood optimization.
If not provided, the optimization will start in 0.}
}
\value{
An object of class \code{cm} containing the estimated parameters of the cure model.
The appropriate link functions taken on \eqn{\pi} and the \eqn{\theta_i}'s are linear in the covariates corresponding to their respective parameter estimates.
}
\description{
This function fits parametric cure models using simple parametric distributions.
}
\details{
If \code{type = "mixture"}, the function fits the model,
\deqn{S(t|z) = \pi(z) + [1 - \pi(z)] S_u(t|z),}
and if \code{type = "nmixture"}, the function fits the model,
\deqn{S(t|z) = \pi(z)^{\widetilde F(t)},}
where z is a vector of covariates. The \code{formula.surv} argument is used to model
\eqn{S_u(t)} (1 - \eqn{\widetilde F(t)}). It is a \code{list} of formulas with as many entries as there are
parameters in the chosen parametric distribution. If not specified, all formulas are assumed to be \code{~1}.
The ith formula, i.e., \code{formula.surv[[i]]} refers to \eqn{\theta_i} in the below survival functions.\cr\cr
Exponential model:
\deqn{S_u(t) = \exp\left(-t \theta_1\right).}
Weibull model:
\deqn{S_u(t) = \exp\left(-\theta_1 t^{\theta_2}\right).}
Log-normal model:
\deqn{S_u(t) = 1 - \Phi\left(\frac{\log(t) - \theta_1}{\theta_2}\right).}
Weibull-exponential mixture model:
\deqn{S_u(t) = \theta_1\exp\left(-\theta_2 t^{\theta_3}\right) + (1 - \theta_1)\exp\left(-\theta_4 t\right).}
Weibull-Weibull mixture model:
\deqn{S_u(t) = \theta_1\exp\left(-\theta_2 t^{\theta_3}\right) + (1 - \theta_1)\exp\left(-\theta_4 t^{\theta_5}\right).}
Generalized modified Weibull distribution:
\deqn{S_u(t) = 1-\left(1 - \exp\left(-\theta_1 t ^ \theta_2 \exp(\theta_3 t)\right)\right) ^ \theta_4.}
In the Weibull-exponential and Weibull-Weibull mixture models, the link function for the mixture component is controlled by \code{link.mix}.
The remaining parameters are modelled using an exponential link function except \eqn{\theta_1} in the log-normal model,
which is modelled using the identity. Parameters are not transformed back to the original scale in
the outputted object and related \code{print.cm} and \code{summary.cm} functions
}
\examples{
##Use data cleaned version of the colon cancer data from the rstpm2 package
data("colonDC")
set.seed(2)
colonDC <- colonDC[sample(1:nrow(colonDC), 500), ]

##Extract general population hazards
colonDC$bhaz <- general.haz(time = "FU", rmap = list(age = "agedays", sex = "sex", year= "dx"),
                            data = colonDC, ratetable = survexp.dk)


###Without covariates
##Fit weibull mixture cure model
fit.wei <- fit.cure.model(Surv(FUyear, status) ~ 1, data = colonDC, bhazard = "bhaz",
                          type = "mixture", dist = "weibull", link = "logit")

##Plot various summaries of the model (see ?predict.cm)
plot(fit.wei)
plot(fit.wei, time = seq(0, 40, length.out = 100))
plot(fit.wei, type = "hazard")
plot(fit.wei, type = "survuncured")
plot(fit.wei, type = "probcure")

#Fit a weibull-weibull mixture cure model
fit.weiwei <- fit.cure.model(Surv(FUyear, status) ~ 1, data = colonDC, bhazard = "bhaz",
                          type = "mixture", dist = "weiwei", link = "logit")

#Compare to the weibull model
plot(fit.wei, ci = FALSE)
plot(fit.weiwei, add = TRUE, col = 2, ci = FALSE)

###With covariates
##Fit weibull mixture cure model with age effect for both components of the Weibull model
fit <- fit.cure.model(Surv(FUyear, status) ~ age, data = colonDC, bhazard = "bhaz",
                      formula.surv = list(~ age, ~ age),
                      type = "mixture", dist = "weibull", link = "logit")

##Plot model for age 50 and 60
plot(fit, newdata = data.frame(age = 60),
     time = seq(0, 15, length.out = 100), ci = FALSE)
plot(fit, newdata = data.frame(age = 50),
     time = seq(0, 15, length.out = 100), ci = FALSE, add = TRUE, col = 2)

plot(fit, newdata = data.frame(age = 60),
     time = seq(0, 15, length.out = 100), ci = FALSE, type = "hazard")
plot(fit, newdata = data.frame(age = 50),
     time = seq(0, 15, length.out = 100), ci = FALSE, type = "hazard", add = TRUE, col = 2)
}
