ctKalmanTIP <- function(sf,tipreds='all',subject=1,...){
  if(tipreds[1] %in% 'all') tipreds <- sf$ctstanmodel$TIpredNames
  if(length(subject) > 1) stop('>1 subject!')
  
  sdat <- standatact_specificsubjects(standata = sf$standata,subjects = subject)
  sdat$tipredsdata[,sf$ctstanmodel$TIpredNames] <- 0 #set all tipreds to zero

  #create datalong structure
  dat <- data.frame(id=sdat$subject,time=sdat$time)
  datti <- suppressWarnings(merge(dat,data.frame(id=sdat$subject,time=sdat$time,sdat$tipredsdata),all=TRUE))
  datti <- datti[order(datti[[sf$ctstanmodelbase$subjectIDname]],datti$time),]
  colnames(datti)[1:2] <- c(sf$ctstanmodelbase$subjectIDname,sf$ctstanmodelbase$timeName)
  addm <- matrix(NA,nrow=nrow(dat),ncol=length(sf$ctstanmodel$manifestNames))
  colnames(addm) <- sf$ctstanmodel$manifestNames
  tdpreds <- sdat$tdpreds
  colnames(tdpreds) <- sf$ctstanmodel$TDpredNames
  dat<-cbind(datti,addm,tdpreds)
  # dat <- merge(datti,addm)
  
  tisd <- apply(sf$standata$tipredsdata,2,sd,na.rm=TRUE)
  timu <- apply(sf$standata$tipredsdata,2,mean,na.rm=TRUE)

  newdat=dat
  for(tip in 1:length(tipreds)){
    for(direction in c(-1,1)){
      tdat <- newdat
      tdat[[sf$ctstanmodelbase$subjectIDname]] <- paste0(tipreds[tip],
        ifelse(direction==1,' high',' low'))
      tdat[,tipreds[tip]] <- tisd[tip] * direction
      #add to full dat
      dat <- rbind(dat,tdat)
    }
  }
  sf$standata <- suppressMessages(ctStanData(sf$ctstanmodel,dat,optimize=TRUE))
  ctKalman(fit = sf,subjects=1:sf$standata$nsubjects,realid=TRUE,...)
}

#' ctKalman 
#'
#' Outputs predicted, updated, and smoothed estimates of manifest indicators and latent states, 
#' with covariances, for specific subjects from data fit with \code{\link{ctStanFit}}, 
#' based on either the mode (if optimized) or mean (if sampled) of parameter distribution.
#' 
#' @param fit fit object as generated by \code{\link{ctStanFit}}.
#' @param timerange Either 'asdata' to just use the observed data range, or a numeric vector of length 2 denoting start and end of time range, 
#' allowing for estimates outside the range of observed data. Ranges smaller than the observed data
#' are ignored.
#' @param timestep Either 'asdata' to just use the observed data 
#' (which also requires 'asdata' for timerange) or a positive numeric value
#' indicating the time step to use for interpolating values. Lower values give a more accurate / smooth representation,
#' but take a little more time to calculate. Currently unavailable for ctStan fits.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param removeObs Logical. If TRUE, observations (but not covariates)
#' are set to NA, so only expectations based on
#' parameters and covariates are returned. 
#' @param plot Logical. If TRUE, plots output instead of returning it. 
#' See \code{\link{plot.ctKalmanDF}} 
#' (Stan based fit) for the possible arguments.
#' @param realid Output using original (not necessarily integer sequence) subject id's?
#' @param ... additional arguments to pass to \code{\link{plot.ctKalmanDF}}.
#' @return Returns a list containing matrix objects etaprior, etaupd, etasmooth, y, yprior, 
#' yupd, ysmooth, prederror, time, loglik,  with values for each time point in each row. 
#' eta refers to latent states and y to manifest indicators - y itself is thus just 
#' the input data. 
#' Covariance matrices etapriorcov, etaupdcov, etasmoothcov, ypriorcov, yupdcov, ysmoothcov,  
#' are returned in a row * column * time array. 
#' Some outputs are unavailable for ctStan fits at present.
#' If plot=TRUE, nothing is returned but a plot is generated.
#' @examples
#' \donttest{
#' 
#' #Basic
#' ctKalman(ctstantestfit, timerange=c(0,60), plot=TRUE)
#' 
#' #Multiple subjects, y and yprior, showing plot arguments
#' plot1<-ctKalman(ctstantestfit, timerange=c(0,60), timestep=.1, plot=TRUE,
#'   subjects=2:3, 
#'   kalmanvec=c('y','yprior'),
#'   errorvec=c(NA,'ypriorcov')) #'auto' would also have achieved this
#'   
#'  #modify plot as per normal with ggplot
#'  print(plot1+ggplot2::coord_cartesian(xlim=c(0,10)))
#'  
#'  #or generate custom plot from scratch:#'  
#'  k=ctKalman(ctstantestfit, timerange=c(0,60), timestep=.1, subjects=2:3)
#'  library(ggplot2)
#'  ggplot(k[k$Element %in% 'yprior',],
#'    aes(x=Time, y=value,colour=Subject,linetype=Row)) +
#'    geom_line() +
#'    theme_bw()
#'
#'  }
#' @export

ctKalman<-function(fit, timerange='asdata', timestep='auto',
  subjects=1, removeObs = FALSE, plot=FALSE, realid=FALSE,...){
  type=NA
  if('ctStanFit' %in% class(fit)) type='stan' 
  if('ctsemFit' %in% class(fit)) type ='omx'
  if(is.na(type)) stop('fit object is not from ctFit or ctStanFit!')
  
  subjects <- sort(subjects) #in case not entered in ascending order
  if(type=='stan'){
    if(timestep=='auto'){
      if(fit$standata$intoverstates==1) timestep=sd(fit$standata$time,na.rm=TRUE)/50 else timestep ='asdata'
    }
    if(all(timerange == 'asdata')) timerange <- range(fit$standata$time[fit$standata$subject %in% subjects])
    if(timestep != 'asdata' && fit$ctstanmodel$continuoustime) {
      if(fit$ctstanmodel$continuoustime != TRUE) stop('Discrete time model fits must use timestep = "asdata"')
      times <- seq(timerange[1],timerange[2],timestep)
      fit$standata <- standataFillTime(fit$standata,times)
    } 
    idstore <- fit$standata$subject
    if(length(fit$stanfit$stanfit@sim)==0) {
      fit$standata <- standatact_specificsubjects(fit$standata, subjects = subjects)
      idstore <- as.integer(subjects[fit$standata$subject])
    }
    if(!length(fit$stanfit$stanfit@sim)==0) fit$standata$dokalmanrows <-
      as.integer(fit$standata$subject %in% subjects |
          as.logical(match(unique(fit$standata$subject),fit$standata$subject)))
    
    if(removeObs){
      sapply(c('nobs_y','nbinary_y','ncont_y','whichobs_y','whichbinary_y','whichcont_y'),
        function(x) fit$standata[[x]][] <<- 0L)
      fit$standata$Y[] <- 99999
    }
    out <- ctStanKalman(fit,pointest=length(fit$stanfit$stanfit@sim)==0, 
      collapsefunc=mean, indvarstates = FALSE) #extract state predictions
    out$id <- idstore #as.integer(subjects[out$id]) #get correct subject indicators

    out <- meltkalman(out)
    out=out[!(out$Subject %in% subjects) %in% FALSE,]
    if(realid){
      out$Subject <- as.integer(out$Subject)
      out$Subject <- factor(fit$standata$idmap[
        match(out$Subject,fit$standata$idmap[,2]),1])
    }
  }
  
  # if(type !='stan'){
  #   if(timestep=='auto') timestep=1
  #   out<-list()
  #   if(timerange[1] != 'asdata' & timestep[1] == 'asdata') stop('If timerange is not asdata, a timestep must be specified!')
  #   
  #   # if(!is.null(datalong)) { #adjust ids and colnames as needed
  #   #   datalong <- makeNumericIDs(datalong, fit$ctstanmodel$subjectIDname,fit$ctstanmodel$timeName) #ensure id's are appropriate
  #   #   colnames(datalong)[colnames(datalong)==fit$ctstanmodel$subjectIDname] <- 'subject'
  #   #   colnames(datalong)[colnames(datalong)==fit$ctstanmodel$timeName] <- 'time'
  #   # }
  #   
  #   # if(is.null(datalong)) { #get relevant data
  #     
  #     if(is.null(fit$mxobj$expectation$P0)) { #if not fit with kalman filter then data needs rearranging
  #       datalong=suppressMessages(ctWideToLong(datawide = fit$mxobj$data$observed[subjects,,drop=FALSE],
  #         Tpoints=fit$ctmodelobj$Tpoints,
  #         n.manifest=fit$ctmodelobj$n.manifest,manifestNames = fit$ctmodelobj$manifestNames,
  #         n.TDpred=fit$ctmodelobj$n.TDpred,TDpredNames = fit$ctmodelobj$TDpredNames,
  #         n.TIpred = fit$ctmodelobj$n.TIpred, TIpredNames = fit$ctmodelobj$TIpredNames))
  #       datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT'))
  #       datalong[,'id'] <- subjects[datalong[,'id'] ]
  #     } else {
  #       datalong=fit$mxobj$data$observed
  #       datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT1'))
  #     }
  #     colnames(datalong)[colnames(datalong) == 'id'] <- 'subject'
  #     
  #     
  #   # }
  #   
  #   
  #   
  #   if(!all(subjects %in% datalong[,'subject'])) stop('Invalid subjects specified!')
  #   
  #   for(subjecti in subjects){
  #     #setup subjects data, interpolating and extending as necessary
  #     sdat=datalong[datalong[,'subject'] == subjecti,,drop=FALSE]
  #     if(timestep != 'asdata' || timerange[1] != 'asdata') {
  #       if(timerange[1]=='asdata') stimerange <- range(sdat[,'time']) else {
  #         stimerange <- timerange
  #         if(timerange[1] > min(sdat[,'time']) || timerange[2] < max(sdat[,'time']) ) stop('Specified timerange must contain all subjects time ranges!')
  #       }
  #       snewtimes <- seq(stimerange[1],stimerange[2],timestep)
  #       snewdat <- array(NA,dim=c(length(snewtimes),dim(sdat)[-1]),dimnames=list(c(),dimnames(sdat)[[2]]))
  #       snewdat[,'time'] <- snewtimes
  #       snewdat[,fit$ctmodelobj$TDpredNames] <- 0
  #       sdat <- rbind(sdat,snewdat)
  #       sdat[,'time'] <- round(sdat[,'time'],10)
  #       sdat<-sdat[!duplicated(sdat[,'time']),,drop=FALSE]
  #       sdat <- sdat[order(sdat[,'time']),,drop=FALSE]
  #       sdat[,c(fit$ctmodelobj$manifestNames,fit$ctmodelobj$TDpredNames)] [sdat[,c(fit$ctmodelobj$manifestNames,fit$ctmodelobj$TDpredNames)]==99999] <- NA
  #       sdat[,'subject'] <- subjecti
  #     }
  #     
  #     #get parameter matrices
  #     # 
  #     model <- summary(fit)
  #     # model <- model$
  #     
  #     #get kalman estimates
  #     
  #     out[[paste('subject',subjecti)]]<-Kalman(kpars=model,
  #       datalong=sdat,
  #       manifestNames=fit$ctmodelobj$manifestNames,
  #       latentNames=fit$ctmodelobj$latentNames,
  #       TDpredNames=fit$ctmodelobj$TDpredNames,
  #       idcol='subject',
  #       timecol='time')
  #   }
  #   class(out) <- c('ctKalman',class(out))
  # }#end old kalman
  
  if(plot) {
    plot(x=out,subjects=subjects,...)
  } else return(out)
}




#' Plots Kalman filter output from ctKalman.
#'
#' @param x Output from \code{\link{ctKalman}}. In general it is easier to call 
#' \code{\link{ctKalman}} directly with the \code{plot=TRUE} argument, which calls this function.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param kalmanvec string vector of names of any elements of the output you wish to plot, 
#' the defaults of 'y' and 'ysmooth' plot the original data, 'y', 
#' and the estimates of the 'true' value of y given all data. Replacing 'y' by 'eta' will 
#' plot latent states instead (though 'eta' alone does not exist) and replacing 'smooth' 
#' with 'upd' or 'prior' respectively plots updated (conditional on all data up to current time point)
#' or prior (conditional on all previous data) estimates.
#' @param errorvec vector of names indicating which kalmanvec elements to plot uncertainty bands for. 
#' 'auto' plots all possible.
#' @param elementNames if NA, all relevant object elements are included -- e.g. if yprior is in the kalmanvec
#' argument, all manifest variables are plotted, and likewise for latent states if etasmooth was specified.
#' Alternatively, a character vector specifying the manifest and latent names to plot explicitly can be specified.
#' @param plot if FALSE, plots are not generated and the ggplot object is simply returned invisibly.
#' @param errormultiply Numeric denoting the multiplication factor of the std deviation of errorvec objects. 
#' Defaults to 1.96, for 95\% intervals.
#' @param polygonsteps Number of steps to use for uncertainty band shading. 
#' @param polygonalpha Numeric for the opacity of the uncertainty region.
#' @param facets when multiple subjects are included in multivariate plots, the default is to facet plots 
#' by variable type. This can be set to NA for no facets, or \code{variable(Subject)} for facetting by subject.
#' @param ... not used.
#' @return A ggplot2 object. Side effect -- Generates plots.
#' @method plot ctKalmanDF
#' @export
#' @examples
#' if(w32chk()){
#'
#' 
#' ### Get output from ctKalman
#' x<-ctKalman(ctstantestfit,subjects=2,timestep=.01)
#' 
#' ### Plot with plot.ctKalmanDF
#' plot(x, subjects=2)
#' 
#' ###Single step procedure:
#' ctKalman(ctstantestfit,subjects=2,
#'   kalmanvec=c('y','yprior'),
#'   elementNames=c('Y1','Y2'), 
#'   plot=TRUE,timestep=.01)
#' }
plot.ctKalmanDF<-function(x, subjects=unique(x$Subject), kalmanvec=c('y','yprior'),
  errorvec='auto', errormultiply=1.96,plot=TRUE,elementNames=NA,
  polygonsteps=10,polygonalpha=.1,
  facets=vars(Variable),
  ...){
  
  kdf <- (x)
  
  if(!'ctKalmanDF' %in% class(kdf)) stop('not a ctKalmanDF object')
  
  if(1==99) Time <- Value <- Subject <- Row <- Variable <- Element <- NULL
  colnames(kdf)[colnames(kdf) %in% 'Row'] <- "Variable"
  colnames(kdf)[colnames(kdf) %in% 'value'] <- "Value"
  
  if(any(!is.na(elementNames))) kdf <- subset(kdf,Variable %in% elementNames)
  
  klines <- kalmanvec[grep('(prior)|(upd)|(smooth)',kalmanvec)]
  if(all(errorvec %in% 'auto')) errorvec <- klines
  errorvec <- errorvec[grep('(prior)|(upd)|(smooth)',errorvec)]
  # kpoints<- kalmanvec[-grep('(prior)|(upd)|(smooth)',kalmanvec)]
  colvec=ifelse(length(subjects) > 1, 'Subject', 'Variable')
  ltyvec <- setNames( rep(NA,length(kalmanvec)),kalmanvec)
  ltyvec[kalmanvec %in% klines] = setNames(1:length(klines),klines)
  # if(length(kalmanvec) > length(klines)) ltyvec <- 
  #   c(setNames(rep(0,length(kalmanvec)-length(klines)),kalmanvec[!kalmanvec %in% klines]),
  #     ltyvec)
  shapevec<-ltyvec
  shapevec[shapevec>0] <- NA
  shapevec[is.na(ltyvec)] <- 19
  # 
  d<-subset(kdf,Element %in% kalmanvec)
  
  g <- ggplot(d,
    aes_string(x='Time',y='Value',colour=colvec,linetype='Element',shape='Element')) +
    scale_linetype_manual(breaks=names(ltyvec),values=ltyvec)+
    scale_shape_manual(breaks=names(shapevec),values=shapevec) +
    # labs(linetype='Element',shape='Element',colour='Element',fill='Element')+
    scale_fill_discrete(guide='none')
  
  if(!is.na(facets[1]) && length(subjects) > 1 && length(unique(subset(kdf,Element %in% kalmanvec)$Variable)) > 1){
    g <- g+ facet_wrap(facets,scales = 'free') 
  }
  
  polygonsteps <- polygonsteps + 1
  polysteps <- seq(errormultiply,0,-errormultiply/(polygonsteps+1))[c(-polygonsteps+1)]
  if(any(!is.na(errorvec))){
    for(si in polysteps){
      # alphasum <- alphasum + polygonalpha/polygonsteps
      
      d2 <- subset(d,Element %in% errorvec)
      d2$sd <- d2$sd *si
      
      if(length(subjects) ==1){
        g<- g+ 
          geom_ribbon(data=d2,aes(ymin=(Value-sd),x=Time,
            ymax=(Value+sd),fill=(Variable)),
            alpha=ifelse(si== polysteps[1],.05,polygonalpha/polygonsteps),
            inherit.aes = FALSE,linetype=0)
        if(si== polysteps[1]) g <- g + 
            geom_line(data=d2,aes(y=(Value-sd),colour=Variable),linetype='dotted',alpha=.4) + 
            geom_line(data=d2,aes(y=(Value+sd),colour=Variable),linetype='dotted',alpha=.4)
      } else {
        g <- g+ 
          geom_ribbon(data=d2,aes(ymin=(Value-sd),x=Time,
            ymax=(Value+sd),fill=(Subject)),inherit.aes = FALSE,
            alpha=polygonalpha/polygonsteps,linetype=0)
        if(si== polysteps[1]) g <- g + 
            geom_line(data=d2,aes(y=(Value-sd),colour=Subject),linetype='dotted',alpha=.7) + 
            geom_line(data=d2,aes(y=(Value+sd),colour=Subject),linetype='dotted',alpha=.7)
      }
    } 
  }
      g <- g + 
        geom_line()+
        geom_point()+
        theme_minimal()
    
  if(plot) suppressWarnings(print(g))
  return(invisible(g))
  
}


