#' ctStanFit
#'
#' Fits a ctsem model specified via \code{\link{ctModel}} with type either 'stanct' or 'standt', using Bayseian inference software
#' Stan. 
#' 
#' @param datalong long format data containing columns for subject id (numeric values, 1 to max subjects), manifest variables, 
#' any time dependent (i.e. varying within subject) predictors, 
#' and any time independent (not varying within subject) predictors.
#' @param ctstanmodel model object as generated by \code{\link{ctModel}} with type='stanct' or 'standt', for continuous or discrete time
#' models respectively.
#' @param stanmodeltext already specified Stan model character string, generally leave NA unless modifying Stan model directly.
#' (Possible after modification of output from fit=FALSE)
#' @param intoverstates logical indicating whether or not to integrate over latent states using a Kalman filter. 
#' Generally recommended to set TRUE unless using non-gaussian measurement model. 
#' @param binomial Deprecated. Logical indicating the use of binary rather than Gaussian data, as with IRT analyses.
#' This now sets \code{intoverstates = FALSE} and the \code{manifesttype} of every indicator to 1, for binary.
#' @param fit If TRUE, fit specified model using Stan, if FALSE, return stan model object without fitting.
#' @param intoverpop if TRUE, integrates over population distribution of parameters rather than full sampling.
#' Allows for optimization of non-linearities and random effects.
#' @param plot if TRUE, a Shiny program is launched upon fitting to interactively plot samples. 
#' May struggle with many (e.g., > 5000) parameters, and may leave sample files in working directory if sampling is terminated.
#' @param derrind vector of integers denoting which latent variables are involved in dynamic error calculations.
#' latents involved only in deterministic trends or input effects can be removed from matrices (ie, that
#' obtain no additional stochastic inputs after first observation), speeding up calculations. 
#' If unsure, leave default of 'all' ! Ignored if intoverstates=FALSE.
#' @param optimize if TRUE, use \code{\link{optimstan}} function for maximum a posteriori estimates. 
#' @param optimcontrol list of parameters sent to \code{\link{optimstan}} governing optimization / importance sampling.
#' @param nopriors logical. If TRUE, any priors are disabled -- sometimes desirable for optimization. 
#' @param iter number of iterations, half of which will be devoted to warmup by default when sampling.
#' When optimizing, this is the maximum number of iterations to allow -- convergence hopefully occurs before this!
#' @param inits vector of parameter start values, as returned by the rstan function \code{rstan::unconstrain_pars} for instance. 
#' @param chains number of chains to sample, during HMC or post-optimization importance sampling.
#' @param cores number of cpu cores to use. Either 'maxneeded' to use as many as available,
#' up to the number of chains, or a positive integer.
#' @param control List of arguments sent to \code{\link[rstan]{stan}} control argument, 
#' regarding warmup / sampling behaviour. Unless specified, values used are:
#' list(adapt_delta = .8, adapt_window=2, max_treedepth=10, adapt_init_buffer=2, stepsize = .001)
#' @param nlcontrol List of non-linear control parameters. 
#' \code{nldynamics} defaults to "auto", but may also be a logical. Set to FALSE to use estimator that assumes linear dynamics, 
#' TRUE to use non-linear estimator. "auto" selects linear when the model is obviously linear, 
#' otherwise nonlinear -- nonlinear is slower.
#' \code{nlmeasurement} defaults to "auto", but may also be a logical. Set to TRUE to use non linear measurement model estimator, 
#' FALSE to use linear model. "auto" selects linear if appropriate, otherwise nonlinear. Non-linear methods are slower but applicable to both linear
#' and non linear cases.
#' \code{ukffull} may be TRUE or FALSE. If FALSE, nonlinear filtering via the unscented filter uses a minimal number of sigma points,
#' that does not capture skew in the resulting distribution. 
#' \code{maxtimestep} must be a positive numeric,  specifying the largest time
#' span covered by the numerical integration. The large default ensures that for each observation time interval, 
#' only a single step of exponential integration is used. When \code{maxtimestep} is smaller than the observation time interval, 
#' the integration is nested within an Euler like loop. 
#' Smaller values may offer greater accuracy, but are slower and not always necessary. Given the exponential integration,
#' linear model elements are fit exactly with only a single step. 
#' \code{ukfspread} should be a small positive numeric value, indicating what fraction of a standard deviation to 
#' use for unscented sigma points. Values between 1e-4 and 2 have tended to be reasonable, in our experience. 
#' In general, larger values may not make sense when using the default of \code{ukffull=FALSE}.
#' @param verbose Integer from 0 to 2. Higher values print more information during model fit -- for debugging.
#' @param stationary Logical. If TRUE, T0VAR and T0MEANS input matrices are ignored, 
#' the parameters are instead fixed to long run expectations. More control over this can be achieved
#' by instead setting parameter names of T0MEANS and T0VAR matrices in the input model to 'stationary', for
#' elements that should be fixed to stationarity.
#' @param forcerecompile logical. For development purposes. 
#' If TRUE, stan model is recompiled, regardless of apparent need for compilation.
#' @param savescores Logical. If TRUE, output from the Kalman filter is saved in output. For datasets with many variables
#' or time points, will increase file size substantially.
#' @param gendata Logical -- generate random data sets per sample? For later plotting / model checking.
#' For datasets with many manifest variables or time points, may increase file size substantially. Needs to be set FALSE for win32.
#' @param ... additional arguments to pass to \code{\link[rstan]{stan}} function.
#' @importFrom Rcpp evalCpp
#' @export
#' @examples
#' \dontrun{
#' #test data with 2 manifest indicators measuring 1 latent process each, 
#' # 1 time dependent predictor, 3 time independent predictors
#' head(ctstantestdat) 
#' 
#' #generate a ctStanModel
#' model<-ctModel(type='stanct',
#' n.latent=2, latentNames=c('eta1','eta2'),
#' n.manifest=2, manifestNames=c('Y1','Y2'),
#' n.TDpred=1, TDpredNames='TD1', 
#' n.TIpred=3, TIpredNames=c('TI1','TI2','TI3'),
#' LAMBDA=diag(2))
#' 
#' #set all parameters except manifest means to be fixed across subjects
#' model$pars$indvarying[-c(19,20)] <- FALSE
#' 
#' #fit model to data (takes a few minutes - but insufficient 
#' # iterations and max_treedepth for inference!)
#' fit<-ctStanFit(ctstantestdat, model, iter=200, chains=2, 
#' control=list(max_treedepth=6))
#' 
#' #output functions
#' summary(fit) 
#' 
#' plot(fit)
#' 
#' 
#' 
#' ###### EXTENDED EXAMPLES #######
#' 
#' 
#' 
#' library(ctsem)
#' set.seed(3)
#' 
#' #recommended to adjust these to appropriate number of cores on machine / 
#' # chains desired. (min 3 chains recommended, but not necessary here)
#' setcores <- 3
#' setchains <- 3
#' 
#' #### Data generation (this section needs to be run, but not necessary to understand!)
#' Tpoints <- 20
#' nmanifest <- 4
#' nlatent <- 2
#' nsubjects<-20
#' 
#' #random effects
#' age <- rnorm(nsubjects) #standardised
#' cint1<-rnorm(nsubjects,2,.3)+age*.5
#' cint2 <- cint1*.5+rnorm(nsubjects,1,.2)+age*.5
#' tdpredeffect <- rnorm(nsubjects,5,.3)+age*.5
#' 
#' for(i in 1:nsubjects){
#'   #generating model
#'   gm<-ctModel(Tpoints=Tpoints,n.manifest = nmanifest,n.latent = nlatent,n.TDpred = 1,
#'     LAMBDA = matrix(c(1,0,0,0, 0,1,.8,1.3),nrow=nmanifest,ncol=nlatent),
#'     DRIFT=matrix(c(-.3, .1, 0, -.5),nlatent,nlatent),
#'     TDPREDEFFECT=matrix(c(tdpredeffect[i],0),nrow=nlatent),
#'     TDPREDMEANS=matrix(c(rep(0,Tpoints-10),1,rep(0,9)),ncol=1),
#'     DIFFUSION = matrix(c(.5, 0, 0, .5),2,2),
#'     CINT = matrix(c(cint1[i],cint2[i]),ncol=1),
#'     T0VAR=diag(2,nlatent,nlatent),
#'     MANIFESTVAR = diag(.5, nmanifest))
#'   
#'   #generate data
#'   newdat <- ctGenerate(ctmodelobj = gm,n.subjects = 1,burnin = 2,
#'     dtmat<-rbind(c(rep(.5,8),3,rep(.5,Tpoints-9))),
#'       wide = FALSE)
#'   newdat[,'id'] <- i #set id for each subject
#'   newdat <- cbind(newdat,age[i]) #include time independent predictor
#'   if(i ==1) {
#'     dat <- newdat[1:(Tpoints-10),] #pre intervention data
#'     dat2 <- newdat #including post intervention data
#'   }
#'   if(i > 1) {
#'     dat <- rbind(dat, newdat[1:(Tpoints-10),])
#'     dat2 <- rbind(dat2,newdat)
#'   }
#' }
#' colnames(dat)[ncol(dat)] <- 'age'
#' colnames(dat2)[ncol(dat)] <- 'age'
#' 
#' 
#' #plot generated data for sanity
#' plot(age)
#' matplot(dat[,gm$manifestNames],type='l',pch=1)
#' plotvar <- 'Y1'
#' plot(dat[dat[,'id']==1,'time'],dat[dat[,'id']==1,plotvar],type='l',
#'   ylim=range(dat[,plotvar],na.rm=TRUE))
#' for(i in 2:nsubjects){
#'   points(dat[dat[,'id']==i,'time'],dat[dat[,'id']==i,plotvar],type='l',col=i)
#' }
#' 
#' 
#' 
#' 
#' 
#' #### Model fitting (from here it is good to understand!)
#' 
#' #Specify univariate linear growth curve
#' #page 5 of https://cran.r-project.org/web/packages/ctsem/vignettes/hierarchical.pdf 
#' # documents these arguments (or use ?ctModel )
#' 
#' m1 <- ctModel(n.manifest = 1,n.latent = 1,n.TIpred = 1, type = 'stanct',
#'   manifestNames = c('Y1'), latentNames=c('L1'),TIpredNames = 'age',
#'   DRIFT=matrix(-1e-5,nrow=1,ncol=1),
#'   DIFFUSION=matrix(0,nrow=1,ncol=1),
#'   CINT=matrix(c('cint1'),ncol=1),
#'   T0MEANS=matrix(c('t0m1'),ncol=1),
#'   T0VAR=matrix(0,nrow=1,ncol=1),
#'   LAMBDA = diag(1),
#'   MANIFESTMEANS=matrix(0,ncol=1),
#'   MANIFESTVAR=matrix(c('merror1'),nrow=1,ncol=1))
#' 
#' #modify between subject aspects -- alternatively, run: edit(m1$pars)
#' m1$pars$indvarying[-which(m1$pars$matrix %in% c('T0MEANS','CINT'))] <- FALSE
#' m1$pars$age_effect[-which(m1$pars$matrix %in% c('T0MEANS','CINT'))] <- FALSE
#' 
#' 
#' plot(m1) #plot prior distributions
#' 
#' #fit
#' f1 <- ctStanFit(datalong = dat, ctstanmodel = m1, 
#'   cores = setcores,chains = setchains,plot=TRUE,
#'   control=list(max_treedepth=7),iter=150)
#' 
#' summary(f1)
#' 
#' #plots of individual subject models v data
#' ctKalman(f1,timestep=.01,plot=TRUE,subjects=1:4,kalmanvec=c('y','etasmooth'))
#' ctKalman(f1,timestep=.01,plot=TRUE,subjects=1,kalmanvec=c('y','ysmooth'))
#' 
#' ctStanPlotPost(f1) #compare prior to posterior distributions 
#' ctStanPlotPost(f1, priorwidth = FALSE) #rescale to width of posterior 
#' 
#' ctStanPostPredict(f1) #compare randomly generated data from posterior to observed data
#' 
#' cf<-ctCheckFit(f1) #compare covariance of randomly generated data to observed cov
#' plot(cf)
#' 
#' #accessing the stan object directly 
#' library(rstan)
#' postsamples <- extract(f1$stanfit,pars='Ygen') #extract data generated from posterior
#' plot( f1$data$time, 
#'   postsamples$Ygen[1,1, ,1]) #iteration 1 (already shuffled), chain 1, all occasions, var 1.
#' points(f1$data$time, f1$data$Y[,1],col='red') #1st manifest variable
#' 
#' 
#' 
#' 
#' 
#' #Specify model including dynamics
#' m2 <- ctModel(n.manifest = 1,n.latent = 1,n.TIpred = 1, type = 'stanct',
#'   manifestNames = c('Y1'), latentNames=c('L1'),TIpredNames = 'age',
#'   DRIFT=matrix('drift11',nrow=1,ncol=1),
#'   DIFFUSION=matrix('diffusion11',nrow=1,ncol=1),
#'   CINT=matrix(c('cint1'),ncol=1),
#'   T0MEANS=matrix(c('t0m1'),ncol=1),
#'   T0VAR=matrix('t0var11',nrow=1,ncol=1),
#'   LAMBDA = diag(1),
#'   MANIFESTMEANS=matrix(0,ncol=1),
#'   MANIFESTVAR=matrix(c('merror1'),nrow=1,ncol=1))
#' 
#' m2$pars$indvarying[-which(m2$pars$matrix %in% c('T0MEANS','CINT'))] <- FALSE
#' m2$pars$age_effect[-which(m2$pars$matrix %in% c('T0MEANS','CINT'))] <- FALSE
#' 
#' 
#' f2 <- ctStanFit(datalong = dat, ctstanmodel = m2, cores = setcores,
#'   chains = setchains,plot=TRUE,
#'   control=list(max_treedepth=7),iter=150)
#' 
#' summary(f2,parmatrices=TRUE,timeinterval=1)
#' 
#' ctKalman(f2,timestep=.01,plot=TRUE,subjects=1,kalmanvec=c('y','etaprior'))
#' ctKalman(f2,timestep=.01,plot=TRUE,subjects=1:4,kalmanvec=c('y','etasmooth'))
#' ctKalman(f2,timestep=.01,plot=TRUE,subjects=1:2,kalmanvec=c('y','ysmooth'))
#' 
#' ctStanPlotPost(f2)
#' 
#' ctStanPostPredict(f2)
#' 
#' 
#' 
#' 
#' 
#' #Include intervention
#' m3 <- ctModel(n.manifest = 1,n.latent = 1,n.TIpred = 1, type = 'stanct',
#'   manifestNames = c('Y1'), latentNames=c('L1'),TIpredNames = 'age',
#'   n.TDpred=1,TDpredNames = 'TD1', #this line includes the intervention
#'   TDPREDEFFECT=matrix(c('tdpredeffect'),nrow=1,ncol=1), #intervention effect
#'   DRIFT=matrix('drift11',nrow=1,ncol=1),
#'   DIFFUSION=matrix('diffusion11',nrow=1,ncol=1),
#'   CINT=matrix(c('cint1'),ncol=1),
#'   T0MEANS=matrix(c('t0m1'),ncol=1),
#'   T0VAR=matrix('t0var11',nrow=1,ncol=1),
#'   LAMBDA = diag(1),
#'   MANIFESTMEANS=matrix(0,ncol=1),
#'   MANIFESTVAR=matrix(c('merror1'),nrow=1,ncol=1))
#' 
#' m3$pars$indvarying[-which(m3$pars$matrix %in% 
#'   c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#'   
#' m3$pars$age_effect[-which(m3$pars$matrix %in% 
#'   c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#' 
#' f3 <- ctStanFit(datalong = dat2, ctstanmodel = m3, cores = setcores,
#'   chains = setchains,plot=TRUE,
#'   control=list(max_treedepth=7),iter=150)
#' 
#' summary(f3,parmatrices=TRUE)
#' 
#' ctKalman(f3,timestep=.01,plot=TRUE,subjects=1,kalmanvec=c('y','etaprior'))
#' ctKalman(f3,timestep=.01,plot=TRUE,subjects=1:4,kalmanvec=c('y','etasmooth'))
#' ctKalman(f3,timestep=.01,plot=TRUE,subjects=1:2,kalmanvec=c('y','ysmooth'))
#' 
#' ctStanPlotPost(f3)
#' 
#' ctStanPostPredict(f3, datarows=0:100)
#' 
#' 
#' 
#' 
#' 
#' 
#' #include 2nd latent process
#' 
#' #use either full explicit specification
#' m4 <- ctModel(n.manifest = 2,n.latent = 2,n.TIpred = 1, type = 'stanct', #no of vars updated
#'   manifestNames = c('Y1','Y2'), latentNames=c('L1','L2'),TIpredNames = 'age', 
#'   n.TDpred=1,TDpredNames = 'TD1', 
#'   TDPREDEFFECT=matrix(c('tdpredeffect1','tdpredeffect2'),nrow=2,ncol=1), 
#'   DRIFT=matrix(c('drift11','drift21','drift12','drift22'),nrow=2,ncol=2),
#'   DIFFUSION=matrix(c('diffusion11','diffusion21',0,'diffusion22'),nrow=2,ncol=2),
#'   CINT=matrix(c('cint1','cint2'),nrow=2,ncol=1),
#'   T0MEANS=matrix(c('t0m1','t0m2'),nrow=2,ncol=1),
#'   T0VAR=matrix(c('t0var11','t0var21',0,'t0var22'),nrow=2,ncol=2),
#'   LAMBDA = matrix(c(1,0,0,1),nrow=2,ncol=2),
#'   MANIFESTMEANS=matrix(c(0,0),nrow=2,ncol=1),
#'   MANIFESTVAR=matrix(c('merror1',0,0,'merror2'),nrow=2,ncol=2))
#' 
#' #restrict between subjects variation / covariate effects
#' m4$pars$indvarying[-which(m4$pars$matrix %in% c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#' m4$pars$age_effect[-which(m4$pars$matrix %in% c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#' 
#' #or rely on defaults (MANIFESTMEANS now free instead of CINT -- 
#' #  no substantive difference for one indicator factors)
#' m4 <- ctModel(n.manifest = 2,n.latent = 2,n.TIpred = 1, type = 'stanct', 
#'   manifestNames = c('Y1','Y2'), latentNames=c('L1','L2'),TIpredNames = 'age',
#'   n.TDpred=1,TDpredNames = 'TD1',
#'   LAMBDA = matrix(c(1,0,0,1),nrow=2,ncol=2))
#' 
#' #restrict between subjects variation / covariate effects
#' m4$pars$indvarying[-which(m4$pars$matrix %in% c('T0MEANS','MANIFESTMEANS','TDPREDEFFECT'))] <- FALSE
#' m4$pars$age_effect[-which(m4$pars$matrix %in% c('T0MEANS','MANIFESTMEANS','TDPREDEFFECT'))] <- FALSE
#' 
#' f4 <- ctStanFit(datalong = dat2, ctstanmodel = m4, cores = setcores,chains = setchains,plot=TRUE,
#'   optimize=T,verbose=1,
#'   control=list(max_treedepth=7),iter=150)
#' 
#' summary(f4,parmatrices=TRUE)
#' 
#' ctStanDiscretePars(f4,plot=TRUE) #auto and cross regressive plots over time
#' 
#' ctKalman(f4,timestep=.01,plot=TRUE,subjects=1,kalmanvec=c('y','etaprior'))
#' ctKalman(f4,timestep=.01,plot=TRUE,subjects=1:2,kalmanvec=c('y','etasmooth'))
#' ctKalman(f4,timestep=.01,plot=TRUE,subjects=1:2,kalmanvec=c('y','ysmooth'))
#' 
#' ctStanPlotPost(f4)
#' 
#' ctStanPostPredict(f4,wait=F)
#' 
#' 
#' 
#' #non-linear dedpendencies - based on m3 model (including intervention)
#' #specify intervention as dependent on extra parameter in PARS matrix, and latent process 1
#' 
#' m3nl <- ctModel(n.manifest = 1,n.latent = 1,n.TIpred = 1, type = 'stanct',
#'   manifestNames = c('Y1'), latentNames=c('L1'),TIpredNames = 'age',
#'   n.TDpred=1,TDpredNames = 'TD1', 
#'   TDPREDEFFECT=matrix(c('PARS[1,1] * state[1]'),nrow=1,ncol=1), 
#'   PARS=matrix(c('tdpredeffect'),1,1),
#'   DRIFT=matrix('drift11',nrow=1,ncol=1),
#'   DIFFUSION=matrix('diffusion11',nrow=1,ncol=1),
#'   CINT=matrix(c('cint1'),ncol=1),
#'   T0MEANS=matrix(c('t0m1'),ncol=1),
#'   T0VAR=matrix('t0var11',nrow=1,ncol=1),
#'   LAMBDA = diag(1),
#'   MANIFESTMEANS=matrix(0,ncol=1),
#'   MANIFESTVAR=matrix(c('merror1'),nrow=1,ncol=1))
#' 
#' m3nl$pars$indvarying[-which(m3nl$pars$matrix %in% 
#'   c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#'   
#' m3nl$pars$age_effect[-which(m3nl$pars$matrix %in% 
#'   c('T0MEANS','CINT','TDPREDEFFECT'))] <- FALSE
#' 
#' #here fit using optimization instead of sampling -- not appropriate in all cases!
#' f3nl <- ctStanFit(datalong = dat2, ctstanmodel = m3nl, 
#'   cores = setcores, chains = setchains,
#'   optimize=TRUE)
#' 
#' summary(f3nl)
#' 
#' #plot functions need updating for non-linearities! (as of ctsem v 2.7.3)
#' #extract.ctStanFit can be used to extract samples and create own plots.
#' #last index of kalaman subobject denotes element of Kalman output.
#' # 1 = ll on std scale, 2= ll scale, 3=error, 4=prediction, 
#' # 5= eta prior, 6= eta upd
#' 
#' ctStanPostPredict(f3nl, datarows=1:100)
#' 
#' e=extract.ctStanFit(f3nl)
#' subindex = which(f3nl$data$subject ==3) #specify subject
#'  
#'  matplot(f3nl$data$time[subindex], # Y predictions given earlier Y
#'    t(e$kalman[,subindex,4]), 
#'    type='l',lty=1,col=rgb(0,0,0,.1))
#'  
#'  points(f3nl$data$time[subindex], #actual Y
#'    f3nl$data$Y[subindex,1],type='p',col='red')
#'    
#'   matplot(f3nl$data$time[subindex], add = TRUE, #Generated Y from model
#'     t(e$Ygen[,subindex,1]), 
#'     type='l',lty=1,col=rgb(0,0,1,.05))
#' }

ctStanFit<-function(datalong, ctstanmodel, stanmodeltext=NA, iter=1000, intoverstates=TRUE, binomial=FALSE,
   fit=TRUE, intoverpop=FALSE, stationary=FALSE,plot=FALSE,  derrind='all',
  optimize=FALSE,  optimcontrol=list(),
  nlcontrol = list(), nopriors=FALSE, chains=1,cores='maxneeded', inits=NULL,
  forcerecompile=FALSE,savescores=TRUE,gendata=TRUE,
  control=list(),verbose=0,...){
  
  if(class(ctstanmodel) != 'ctStanModel') stop('not a ctStanModel object')
  
  if(is.null(nlcontrol$ukfspread)) nlcontrol$ukfspread = 1e-1
  if(is.null(nlcontrol$maxtimestep)) nlcontrol$maxtimestep = 999999
  if(is.null(nlcontrol$nldynamics)) nlcontrol$nldynamics = 'auto'
  if(is.null(nlcontrol$ukffull)) nlcontrol$ukffull = FALSE
  if(is.null(nlcontrol$nlmeasurement)) nlcontrol$nlmeasurement = 'auto'
  nldynamics <- nlcontrol$nldynamics
  
  args=match.call()
  
  if(gendata && .Machine$sizeof.pointer == 4){
    message('Win32 machine -- setting gendata=FALSE to avoid out of memory problems')
    gendata <- FALSE
  }

 
  idName<-ctstanmodel$subjectIDname
  timeName<-ctstanmodel$timeName
  continuoustime<-ctstanmodel$continuoustime
  
  #extract any calcs from model
  calcindices <- grep('\\]|\\[',ctstanmodel$pars$param)
  if(length(calcindices) > 0){
    for(ci in calcindices){
      ctstanmodel$calcs <- c(ctstanmodel$calcs,paste0(ctstanmodel$pars$matrix[ci],
        '[',ctstanmodel$pars$row[ci], ', ', ctstanmodel$pars$col[ci],'] = ',
        ctstanmodel$pars$param[ci]))
    }
    ctstanmodel$pars$value[calcindices] <- 99999
    ctstanmodel$pars$param[calcindices] <- NA
  }
  
  if(length(unique(datalong[,idName]))==1 & any(ctstanmodel$pars$indvarying[is.na(ctstanmodel$pars$value)]==TRUE)) {
    ctstanmodel$pars$indvarying <- FALSE
    message('Individual variation not possible as only 1 subject! indvarying set to FALSE on all parameters')
  }
  
  if(binomial){
    message('Binomial argument deprecated -- in future set manifesttype in the model object to 1 for binary indicators')
    intoverstates <- FALSE
    ctstanmodel$manifesttype[] <- 1
  }
  
  ukf=FALSE
  recompile <- FALSE
  if(optimize && !intoverstates) stop('intoverstates=TRUE required for optimization!')
  if(optimize && !intoverpop && any(ctstanmodel$pars$indvarying[is.na(ctstanmodel$pars$value)]==TRUE && nldynamics != FALSE)){
    intoverpop <- TRUE
    message('Setting intoverpop=TRUE to enable optimization of random effects...')
  }
  
  if(intoverpop & any(ctstanmodel$pars$indvarying[is.na(ctstanmodel$pars$value)])) ukf=TRUE else {
    if(intoverpop==TRUE) message('No individual variation -- disabling intoverpop switch'); intoverpop <- FALSE
    }
  
  
  if(naf(!is.na(ctstanmodel$rawpopsdbaselowerbound))) recompile <- TRUE
  if(ctstanmodel$rawpopsdbase != 'normal(0,1)') recompile <- TRUE
  if(ctstanmodel$rawpopsdtransform != 'exp(2*rawpopsdbase-1)') recompile <- TRUE
  
  
  if(cores=='maxneeded') cores=min(c(chains,parallel::detectCores()))
  
  checkvarying<-function(matrixnames,yesoutput,nooutput=''){#checks if a matrix is set to individually vary in ctspec
    check<-0
    out<-nooutput
    if('T0VAR' %in% matrixnames & nt0varstationary > 0) matrixnames <- c(matrixnames,'DRIFT','DIFFUSION')
    if('T0MEANS' %in% matrixnames & nrow(t0meansstationary) > 0) matrixnames <- c(matrixnames,'DRIFT','CINT')
    for( matname in matrixnames){
      if(any(c(ctspec$indvarying)[c(ctspec$matrix) %in% matrixnames])) check<-c(check,1)
    }
    if(sum(check)==length(matrixnames))  out<-yesoutput
    return(out)
  }
  
  dynamicmatrices <- c('DRIFT','DIFFUSION','CINT','PARS')
  tdpredmatrices <- 'TDPREDEFFECT'
  measurementmatrices <- c('LAMBDA','MANIFESTMEANS','MANIFESTVAR','PARS')
  t0matrices <- c('T0MEANS','T0VAR')
  basematrices <- unique(c(ctstanmodel$pars$matrix,'TDPREDEFFECT','PARS')) #c(dynamicmatrices,measurementmatrices,t0matrices)
  
  
  
  
  #read in ctmodel values
  ctspec<-ctstanmodel$pars
  
  if(!all(ctspec$transform[!is.na(suppressWarnings(as.integer(ctspec$transform)))] %in% c(0,1,2,3,4))) stop('Unknown transform specified -- integers should be 0 to 4')
  
  # if(binomial) {
  #   ctspec<-ctspec[ctspec$matrix != 'MANIFESTVAR',]
  #   message(paste0('MANIFESTVAR matrix is ignored when binomial=TRUE'))
  # }
  
    
  if(stationary) {
    ctspec$param[ctspec$matrix %in% c('T0VAR','T0MEANS')] <- 'stationary'
    ctspec$value[ctspec$matrix %in% c('T0VAR','T0MEANS')] <- NA
    ctspec$indvarying[ctspec$matrix %in% c('T0VAR','T0MEANS')] <- FALSE
  }
  
  manifesttype=ctstanmodel$manifesttype
  
  #fix binary manifestvariance
  if(any(manifesttype==1)){ #if any non continuous variables, (with free parameters)...
    if(any(is.na(as.numeric(c(ctspec$value[ctspec$matrix=='MANIFESTVAR'][ctspec$row[ctspec$matrix=='MANIFESTVAR'] %in% which(manifesttype==1)],
      ctspec$value[ctspec$matrix=='MANIFESTVAR'][ctspec$col[ctspec$matrix=='MANIFESTVAR'] %in% which(manifesttype!=0)]))))){
    message('Fixing any free MANIFESTVAR parameters for binary indicators to deterministic calculation')
    ctspec$value[ctspec$matrix=='MANIFESTVAR'][ctspec$row[ctspec$matrix=='MANIFESTVAR'] %in% which(manifesttype==1)] <- 0
    ctspec$value[ctspec$matrix=='MANIFESTVAR'][ctspec$col[ctspec$matrix=='MANIFESTVAR'] %in% which(manifesttype==1)] <- 0
    ctspec$value[ctspec$matrix=='MANIFESTVAR' & ctspec$row %in% which(manifesttype==1) & ctspec$row == ctspec$col] <- 1e-5
  }}
  
  #clean ctspec structure
  found=FALSE
  ctspec$indvarying=as.logical(ctspec$indvarying)
  ctspec$value=as.numeric(ctspec$value)
  ctspec$transform=as.character(ctspec$transform)
  ctspec$param=as.character(ctspec$param)
  comparison=c(NA,NA,FALSE)
  replacement=c(NA,NA,FALSE)
  # names(comparison)=c('param','transform','indvarying')
  for(rowi in 1:nrow(ctspec)){
    if( !is.na(ctspec$value[rowi])) {
      if(any(c(!is.na(ctspec[rowi,'param']),!is.na(ctspec[rowi,'transform']),ctspec[rowi,'indvarying']))){
        found<-TRUE
        ctspec[rowi,c('param','transform','indvarying')]=replacement
      }
    }
  }
  if(found) message('Minor inconsistencies in model found - removing param name, transform and indvarying from any parameters with a value specified')


  
  #adjust transforms for optimization

  if(1==99 && optimize) {
    message('Adapting standard deviation transforms for optimization')
    # ctstanmodel$rawpopsdtransform <- 'log(1+exp(rawpopsdbase))*10'
    indices <- ctspec$matrix %in% c('DIFFUSION','MANIFESTVAR','T0VAR') & is.na(ctspec$value) & ctspec$row == ctspec$col
    ctspec$transform[indices] <- 1
    ctspec$multiplier[indices] <- 1
    ctspec$meanscale[indices] <- 10
    ctspec$offset[indices] <- 0
  }
  
  ctstanmodel$pars <- ctspec #updating because we save the model later
  

  
  #collect individual stationary elements and update ctspec
  t0varstationary <- as.matrix(rbind(ctspec[which(ctspec$param %in% 'stationary' & ctspec$matrix %in% 'T0VAR'),c('row','col')]))
  if(nrow(t0varstationary) > 0){ #ensure upper tri is consistent with lower
    for(i in 1:nrow(t0varstationary)){
      if(t0varstationary[i,1] != t0varstationary[i,2]) t0varstationary <- rbind(t0varstationary,t0varstationary[i,c(2,1)])
    }}
  t0varstationary = unique(t0varstationary) #remove any duplicated rows
  

  t0meansstationary <- as.matrix(rbind(ctspec[which(ctspec$param[ctspec$matrix %in% 'T0MEANS'] %in% 'stationary'),c('row','col')]))
  ctspec$value[ctspec$param %in% 'stationary'] <- 0
  ctspec$indvarying[ctspec$param %in% 'stationary'] <- FALSE
  ctspec$transform[ctspec$param %in% 'stationary'] <- NA
  ctspec$param[ctspec$param %in% 'stationary'] <- NA
  
  nt0varstationary <- nrow(t0varstationary)
  nt0meansstationary <- nrow(t0meansstationary)
  # if(nt0meansstationary ==0) t0meansstationary <- matrix(-99,ncol=2)
  
  
  nsubjects <- length(unique(datalong[, idName])) 
  
  #create random effects indices for each matrix
  for(mati in basematrices){
    if( (!intoverpop && any(ctspec$indvarying[ctspec$matrix==mati])) || any(unlist(ctspec[ctspec$matrix==mati,paste0(ctstanmodel$TIpredNames,'_effect')]))) subindex <- 1:nsubjects else subindex <- rep(1,nsubjects)
    assign(paste0(mati,'subindex'), subindex)
  }
  if(stationary || nt0varstationary > 0) T0VARsubindex <- rep(1:max(c(T0VARsubindex,DRIFTsubindex,DIFFUSIONsubindex)), ifelse(max(c(T0VARsubindex,DRIFTsubindex,DIFFUSIONsubindex)) > 1, 1, nsubjects))
  if(stationary || nt0meansstationary > 0) T0MEANSsubindex <- rep(1:max(c(T0MEANSsubindex,DRIFTsubindex,CINTsubindex)), ifelse(max(c(T0MEANSsubindex,DRIFTsubindex,CINTsubindex)) > 1, 1, nsubjects))
  asymCINTsubindex <- rep(1:max(c(CINTsubindex,DRIFTsubindex)), ifelse(max(c(CINTsubindex,DRIFTsubindex)) > 1, 1, nsubjects))
  asymDIFFUSIONsubindex <- rep(1:max(c(DIFFUSIONsubindex,DRIFTsubindex)), ifelse(max(c(DIFFUSIONsubindex,DRIFTsubindex)) > 1, 1, nsubjects))

  # if(ukf) asymDIFFUSIONsubindex <- rep(1,nsubjects)
  
  #simply exponential?
  driftdiagonly <- ifelse(all(!is.na(ctspec$value[ctspec$matrix == 'DRIFT' & ctspec$row != ctspec$col]) &
     all(ctspec$value[ctspec$matrix == 'DRIFT' & ctspec$row != ctspec$col] == 0) ), 1, 0)
  
  # #split ctspec into unique and non-unique components
  # ctspecduplicates <- ctspec[duplicated(ctspec$param)&!is.na(ctspec$param),]
  # popmeanduplicates<-c()
  # if(any(duplicated(ctspec$param)&!is.na(ctspec$param))){
  #   for(i in 1:nrow(ctspecduplicates)){
  #     popmeanduplicates[i] = paste0('rawpopmeans[',match(ctspecduplicates$param[i], unique(ctspec$param[!is.na(ctspec$param)])),']')
  #   }
  # }
  # ctspec <- ctspec[!duplicated(ctspec$param) | is.na(ctspec$param),]
  # ctspecduplicates=cbind(ctspecduplicates,popmeanduplicates)
  
  n.latent<-ctstanmodel$n.latent
  n.manifest<-ctstanmodel$n.manifest
  n.TDpred<-ctstanmodel$n.TDpred
  n.TIpred<-ctstanmodel$n.TIpred
  
  manifestNames<-ctstanmodel$manifestNames
  latentNames<-ctstanmodel$latentNames
  TDpredNames<-ctstanmodel$TDpredNames
  TIpredNames<-ctstanmodel$TIpredNames
  # indvarying<-ctspec$indvarying #c(ctspec$indvarying,ctspecduplicates$indvarying)
  # nindvarying<-sum(ctspec$indvarying)
  # nparams<-length(unique(ctspec$param[!is.na(ctspec$param)]))
  
  
  ###data checks
  if (!(idName %in% colnames(datalong))) stop(paste('id column', omxQuotes(idName), "not found in data"))
  
  #scale check
  if(naf(any(abs(colMeans(datalong[,c(ctstanmodel$manifestNames,ctstanmodel$TDpredNames),drop=FALSE],na.rm=TRUE)) > 5))){
    message('Uncentered data noted -- default priors *may* not be appropriate')
  }
  
  if(ctstanmodel$n.TIpred > 1 && any(abs(colMeans(datalong[,c(ctstanmodel$TIpredNames),drop=FALSE],na.rm=TRUE)) > .3)){
    message('Uncentered TI predictors noted -- default priors may not be appropriate')
  }
  
  if(naf(any(abs(apply(datalong[,c(ctstanmodel$manifestNames,ctstanmodel$TDpredNames,ctstanmodel$TIpredNames),drop=FALSE],2,sd,na.rm=TRUE)) > 3))){
    message('Unscaled data noted -- default priors may not be appropriate')
  }

  #fit spec checks
  # if(binomial & any(intoverstates)) stop('Binomial only possible with intoverstates=FALSE')
  
  #id mapping
  original <- unique(datalong[,idName])
  datalong <- makeNumericIDs(datalong,idName,timeName)
  new <- unique(datalong[,idName])
  idmap <- cbind(original, new)

  
  #t0 index
  T0check<-rep(1,nrow(datalong))
  for(i in 2:nrow(datalong)){
    T0check[i]<- ifelse(datalong[i,idName] != datalong[i-1,idName], 1, 0)
  }
  
  if (!(timeName %in% colnames(datalong))) stop(paste('time column', omxQuotes(timeName), "not found in data"))
  if(any(is.na(datalong[,timeName]))) stop('Missing "time" column!')
  
  
   #configure user specified calculations
  if(ctstanmodel$gradient != 'gradient = DRIFT * state + CINT[,1];') {
    recompile <- TRUE
    nldynamics <- TRUE
  }
  gradient <- ctstanmodel$gradient
  
  dynamiccalcs <- ctstanmodel$calcs[
    unlist(lapply(
      lapply(ctstanmodel$calcs, function(x) unlist(lapply(dynamicmatrices, function(y) grepl(y,x)))),
      function(z) any(z)))
    ]
  
  tdpredcalcs <- ctstanmodel$calcs[
    unlist(lapply(
      lapply(ctstanmodel$calcs, function(x) unlist(lapply('TDPREDEFFECT', function(y) grepl(y,x)))),
      function(z) any(z)))
    ]
  
  measurementcalcs <- ctstanmodel$calcs[
    unlist(lapply(
      lapply(ctstanmodel$calcs, function(x) unlist(lapply(measurementmatrices, function(y) grepl(y,x)))),
      function(z) any(z)))
    ]
  
    t0calcs <- ctstanmodel$calcs[
    unlist(lapply(
      lapply(ctstanmodel$calcs, function(x) unlist(lapply(t0matrices, function(y) grepl(y,x)))),
      function(z) any(z)))
      ]
    
    for(mati in c('DRIFT','DIFFUSION','CINT','TDPREDEFFECT','LAMBDA','MANIFESTMEANS','MANIFESTVAR','T0MEANS','T0VAR','PARS')){
      dynamiccalcs <- gsub(mati,paste0('s',mati),dynamiccalcs)
      measurementcalcs <- gsub(mati,paste0('s',mati),measurementcalcs)
      t0calcs <- gsub(mati,paste0('s',mati),t0calcs)
      tdpredcalcs <- gsub(mati,paste0('s',mati),tdpredcalcs)
      gradient <- gsub(mati,paste0('s',mati),gradient)
    }
    

    if(length(c(dynamiccalcs,measurementcalcs,t0calcs,tdpredcalcs)) > 0) recompile <- TRUE
    if( (nt0varstationary + nt0meansstationary) >0 && 
      length(dynamiccalcs > 0)) message('Stationarity assumptions based on initial states when using non-linear dynamics')
    
    if(
      length(c(dynamiccalcs,tdpredcalcs)) > 0 | 
        intoverpop | nldynamics==TRUE
    ) { message('Using nonlinear Kalman filter for dynamics'); ukf <- TRUE}

    nlmeasurement <- nlcontrol$nlmeasurement
    if(nlmeasurement == 'auto') {
      nlmeasurement <- FALSE
      if(length(measurementcalcs) > 0 || intoverpop && any(ctstanmodel$pars$indvarying[ctstanmodel$pars$matrix %in% measurementmatrices])) { 
        nlmeasurement <- TRUE
      }
    }
    if(nlmeasurement) message('Using nonlinear Kalman filter for measurement update');
    if(!nlmeasurement) message('Using linear Kalman filter for measurement update');
    
    
    
    if(ukf==FALSE && nldynamics=="auto") {
      message('Linear model specified -- setting nldynamics = FALSE'); 
      nldynamics <- FALSE
    }
    if(ukf==TRUE && nldynamics=="auto") {
      message('Non-linear model specified -- setting nldynamics = TRUE'); 
      nldynamics <- TRUE
    }
  
  if(nldynamics && !intoverstates) stop('intoverstates must be TRUE for nonlinear dynamics')
  if(!nldynamics) message('Using linear Kalman filter for dynamics')
    
    #intoverpop calcs setup
    intoverpopdynamiccalcs <- paste0('
    if(intoverpop==1){
',paste0(
      unlist(lapply(c(dynamicmatrices), function(m){
      paste0('
    for(ri in 1:size(',m,'setup)){ //for each row of matrix setup
      if(',m,'setup[ ri,5] > 0 && ( statei == 0 || statei == nlatent + ',m,'setup[ ri,5])){ // if individually varying
        s',m,'[',m,'setup[ ri,1], ',m,'setup[ri,2]] = 
          ','tform(state[nlatent +',m,'setup[ri,5] ], ',m,'setup[ri,4], ',m,'values[ri,2], ',m,'values[ri,3], ',m,'values[ri,4] ); 
      }
    }')
    })),collapse='\n'),'}',collapse='\n')
    
#    //if(statei==2 || (statei > 2 && ukfstates[nlatent +',m,'setup[ri,5], statei ] != ukfstates[nlatent +',m,'setup[ri,5], statei-1 ])){ //only recalculate if state changed

    
    measurementcalcs <- paste0(measurementcalcs,';
      if(intoverpop==1){
        ',paste0(unlist(lapply(measurementmatrices, function(m){
      paste0('
    for(ri in 1:size(',m,'setup)){
      if(',m,'setup[ ri,5] > 0){ 
         s',m,'[',m,'setup[ ri,1], ',m,'setup[ri,2]] = ',
         'tform(ukfstates[nlatent +',m,'setup[ri,5], statei ], ',m,'setup[ri,4], ',m,'values[ri,2], ',m,'values[ri,3], ',m,'values[ri,4] ); 
      }
    }')
    })),collapse='\n'),'}',collapse='\n')
    
    t0calcs <- paste0(t0calcs,';
      if(intoverpop==1){
        ',paste0(unlist(lapply(t0matrices, function(m){
      paste0('
    for(ri in 1:size(',m,'setup)){
      if(',m,'setup[ ri,5] > 0){ 
         s',m,'[',m,'setup[ ri,1], ',m,'setup[ri,2]] = ',
          'tform(state[nlatent +',m,'setup[ri,5] ], ',m,'setup[ri,4], ',m,'values[ri,2], ',m,'values[ri,3], ',m,'values[ri,4] ); 
      }
    }')
    })),collapse='\n'),'}',collapse='\n')
    
    #end intoverpop setup
  
  #check id and calculate intervals, discrete matrix indices
  driftindex<-rep(0,nrow(datalong))
  diffusionindex<-driftindex
  cintindex<-driftindex
  oldsubi<-0
  dT<-rep(-1,length(datalong[,timeName]))
  
  for(rowi in 1:length(datalong[,timeName])) {
    subi<-datalong[rowi,idName]
    if(rowi==1 && subi!=1) stop('subject id column must ascend from 1 to total subjects without gaps')
    if(oldsubi!=subi && subi-oldsubi!=1) stop('subject id column must ascend from 1 to total subjects without gaps')
    if(subi - oldsubi == 1) {
      dT[rowi]<-0
      subistartrow<-rowi
    }
    if(subi - oldsubi == 0) {
      if(continuoustime) dT[rowi]<-round(datalong[rowi,timeName] - datalong[rowi-1,timeName],8)
      if(!continuoustime) dT[rowi]<-1
      if(dT[rowi] <=0) stop(paste0('A time interval of ', dT[rowi],' was found at row ',rowi))
      # if(subi!=oldsubi) intervalChange[rowi] <-  0
      # if(subi==oldsubi && dT[rowi] != dT[rowi-1]) intervalChange[rowi] <- 1
      # if(subi==oldsubi && dT[rowi] == dT[rowi-1]) intervalChange[rowi] <- 0
      
      if(!ukf){
      if(dT[rowi] %in% dT[1:(rowi-1)]) dTinwhole<-TRUE else dTinwhole<-FALSE
      if(dT[rowi] %in% dT[subistartrow:(rowi-1)]) dTinsub<-TRUE else dTinsub<-FALSE
      
      if(checkvarying('DRIFT',1,0)==0 & dTinwhole==FALSE) driftindex[rowi] <- max(driftindex)+1
      if(checkvarying('DRIFT',1,0)==1 & dTinsub==FALSE) driftindex[rowi] <- max(driftindex)+1
      if(checkvarying('DRIFT',1,0)==0 & dTinwhole==TRUE) driftindex[rowi] <- driftindex[match(dT[rowi],dT)]
      if(checkvarying('DRIFT',1,0)==1 & dTinsub==TRUE) driftindex[rowi] <- driftindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
      
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==0 & dTinwhole==FALSE) diffusionindex[rowi] <- max(diffusionindex)+1
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==1 & dTinsub==FALSE) diffusionindex[rowi] <- max(diffusionindex)+1
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==0 & dTinwhole==TRUE) diffusionindex[rowi] <- diffusionindex[match(dT[rowi],dT)]
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==1 & dTinsub==TRUE) diffusionindex[rowi] <- diffusionindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
      
      if(checkvarying(c('CINT','DRIFT'),1,0)==0 & dTinwhole==FALSE) cintindex[rowi] <- max(cintindex)+1
      if(checkvarying(c('CINT','DRIFT'),1,0)==1 & dTinsub==FALSE) cintindex[rowi] <- max(cintindex)+1
      if(checkvarying(c('CINT','DRIFT'),1,0)==0 & dTinwhole==TRUE) cintindex[rowi] <- cintindex[match(dT[rowi],dT)]
      if(checkvarying(c('CINT','DRIFT'),1,0)==1 & dTinsub==TRUE) cintindex[rowi] <- cintindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
    }}
    oldsubi<-subi
  }
  # if(!ukf){
  # message('Unique discreteDRIFT calculations per step required = ', length(unique(driftindex))-1)
  # message('Unique discreteCINT calculations per step required = ', length(unique(cintindex))-1)
  # message('Unique discreteDIFFUSION calculations per step required = ', length(unique(diffusionindex))-1)
  # }
  # datalong[sort(c(which(dT > 5),which(dT > 5)+1,which(dT > 5)-1)),1:2]
  
  if(mean(dT) > 3) message('Average time interval greater than 3 -- if using default priors, consider rescaling time data...')
  
  
  if(n.TDpred > 0) {
    tdpreds <- datalong[,TDpredNames,drop=FALSE]
    if(any(is.na(tdpreds))) message('Missingness in TDpreds! Replaced by zeroes...')
    tdpreds[is.na(tdpreds)] <-0 ## rough fix for missingness
  }
  if(n.TIpred > 0) {
    tipreds <- datalong[match(unique(datalong[,idName]),datalong[,idName]),TIpredNames,drop=FALSE]
    if(any(is.na(tipreds))) {
      message(paste0('Missingness in TIpreds - sampling ', sum(is.na(tipreds)),' values'))
      tipreds[is.na(tipreds)] = 99999
    }
  }
  
  datalong[,c(manifestNames,TIpredNames)][is.na(datalong[,c(manifestNames,TIpredNames)])]<-99999 #missing data
  
  
  #check diffusion indices input by user - which latents are involved in covariance
  if(intoverstates==FALSE || all(derrind=='all') ) derrind = 1:n.latent
  # if(all(derrind=='all')) derrind = sort(unique(ctstanmodel$pars$col[
  #   ctstanmodel$pars$matrix=='DIFFUSION' & (!is.na(ctstanmodel$pars$param) | ctstanmodel$pars$value!=0)]))
  derrind = as.integer(derrind)
  if(any(derrind > n.latent)) stop('derrind > than n.latent found!')
  if(length(derrind) > n.latent) stop('derrind vector cannot be longer than n.latent!')
  if(length(unique(derrind)) < length(derrind)) stop('derrind vector cannot contain duplicates or!')
  ndiffusion=length(derrind)
  # message(paste(ndiffusion ,'/',n.latent,'latent variables needed for covariance calculations'))
  
#integration steps
integrationsteps <- sapply(dT,function(x)  ceiling(x / nlcontrol$maxtimestep));
dTsmall <- dT / integrationsteps
dTsmall[is.na(dTsmall)] = 0


ukfilterfunc<-function(ppchecking){
  out<-paste0('
  int si;
  int counter = 0;
  vector[nlatentpop] eta; //latent states
  matrix[nlatentpop, nlatentpop] etacov; //covariance of latent states

  //measurement 
  vector[nmanifest] err;
  vector[nmanifest] ypred;
  vector[ukf ? nmanifest : 0] ystate;
  matrix[nmanifest, nmanifest] ypredcov;
  matrix[nlatentpop, nmanifest] K; // kalman gain
  matrix[nmanifest, nmanifest] ypredcov_sqrt; 

  vector[nmanifest+nmanifest+ (savescores ? nmanifest*2+nlatentpop*2 : 0)] kout[ndatapoints];

  matrix[ukf ? nlatentpop :0,ukf ? nlatentpop :0] sigpoints;

  //linear continuous time calcs
  matrix[nlatent+1,nlatent+1] discreteDRIFT;
  matrix[nlatent,nlatent] discreteDIFFUSION;

  //dynamic system matrices
  ',subjectparaminit(pop=FALSE,smats=TRUE),'

  if(nldynamics==0) discreteDIFFUSION = rep_matrix(0,nlatent,nlatent); //in case some elements remain zero due to derrind

  if(savescores) kout = rep_array(rep_vector(99999,rows(kout[1])),ndatapoints);

  for(rowi in 1:ndatapoints){

    matrix[ukf ? nlatentpop : 0, ukffull ? 2*nlatentpop +2 : nlatentpop + 2 ] ukfstates; //sampled states relevant for dynamics
    matrix[ukf ? nmanifest : 0 , ukffull ? 2*nlatentpop +2 : nlatentpop + 2] ukfmeasures; // expected measures based on sampled states
    si=subject[rowi];
    
    if(T0check[rowi] == 1) { // calculate initial matrices if this is first row for si

    ',subjectparscalc2(pop=FALSE,smats=TRUE),'

      if(ukf==1){
        eta = rep_vector(0,nlatentpop); // because some values stay zero
        sigpoints = rep_matrix(0, nlatentpop,nlatentpop);
        if(intoverpop==1) {
          if(ntipred ==0) eta[ (nlatent+1):(nlatentpop)] = rawpopmeans[indvaryingindex];
          if(ntipred >0) eta[ (nlatent+1):(nlatentpop)] = rawpopmeans[indvaryingindex] + TIPREDEFFECT[indvaryingindex] * tipreds[si]\';
        }
      }

      if(ukf==0){
        eta = sT0MEANS[,1]; //prior for initial latent state
        if(ntdpred > 0) eta += sTDPREDEFFECT * tdpreds[rowi];
        etacov =  sT0VAR;
      }

    } //end T0 matrices

    if(nldynamics==0 && ukf==0 && T0check[rowi]==0){ //linear kf time update
    
      if(continuoustime ==1){
        int dtchange = 0;
        if(si==1 && T0check[rowi -1] == 1) {
          dtchange = 1;
        } else if(T0check[rowi-1] == 1 && dT[rowi-2] != dT[rowi]){
          dtchange = 1;
        } else if(T0check[rowi-1] == 0 && dT[rowi-1] != dT[rowi]) dtchange = 1;
        
        if(dtchange==1 || (T0check[rowi-1]==1 && (si <= DRIFTsubindex[nsubjects] || si <= CINTsubindex[nsubjects]))){
          discreteDRIFT = expm2(append_row(append_col(sDRIFT,sCINT),rep_matrix(0,1,nlatent+1)) * dT[rowi]);
        }
    
        if(dtchange==1 || (T0check[rowi-1]==1 && (si <= DIFFUSIONsubindex[nsubjects]|| si <= DRIFTsubindex[nsubjects]))){
          discreteDIFFUSION[derrind, derrind] = sasymDIFFUSION[derrind, derrind] - 
            quad_form( sasymDIFFUSION[derrind, derrind], discreteDRIFT[derrind, derrind]\' );
          if(intoverstates==0) discreteDIFFUSION = cholesky_decompose(makesym(discreteDIFFUSION,verbose,1));
        }
      }
  
      if(continuoustime==0 && T0check[rowi-1] == 1){
        discreteDRIFT=append_row(append_col(sDRIFT,sCINT),rep_matrix(0,1,nlatent+1));
        discreteDRIFT[nlatent+1,nlatent+1] = 1;
        discreteDIFFUSION=sDIFFUSION;
        if(intoverstates==0) discreteDIFFUSION = cholesky_decompose(makesym(discreteDIFFUSION,verbose,1));
      }

      eta = (discreteDRIFT * append_row(eta,1.0))[1:nlatent];
      if(ntdpred > 0) eta += sTDPREDEFFECT * tdpreds[rowi];
      if(intoverstates==1) {
        etacov = quad_form(etacov, discreteDRIFT[1:nlatent,1:nlatent]\');
        if(ndiffusion > 0) etacov += discreteDIFFUSION;
      }
    }//end linear time update


    if(ukf==1){ //ukf time update
      vector[nlatentpop] state;
      if(T0check[rowi]==0){
        matrix[nlatentpop,nlatentpop] J;
        vector[nlatent] base;
        J = rep_matrix(0,nlatentpop,nlatentpop); //dont necessarily need to loop over tdpreds here...
        if(continuoustime==1){
          matrix[nlatentpop,nlatentpop] Je;
          matrix[nlatent*2,nlatent*2] dQi;
          for(stepi in 1:integrationsteps[rowi]){
            for(statei in 0:nlatentpop){
              if(statei>0){
                J[statei,statei] = 1e-6;
                state = eta + J[,statei];
              } else {
                state = eta;
              }
              ',paste0(dynamiccalcs,';\n',collapse=' '),';\n', intoverpopdynamiccalcs,' //remove diffusion calcs and do elsewhere
              if(statei== 0) {
                base = sDRIFT * state[1:nlatent] + sCINT[,1];
              }
              if(statei > 0) J[1:nlatent,statei] = (( sDRIFT * state[1:nlatent] + sCINT[,1]) - base)/1e-6;
            }
            ',paste0(dynamiccalcs,';\n',collapse=' '),' //find a way to remove this repeat
            Je= expm2(J * dTsmall[rowi]) ;
            discreteDRIFT = expm2(append_row(append_col(sDRIFT,sCINT),rep_vector(0,nlatent+1)\') * dTsmall[rowi]);
            sasymDIFFUSION = to_matrix(  -kronsum(J[1:nlatent,1:nlatent]) \\ to_vector(tcrossprod(sDIFFUSION)), nlatent,nlatent);
            discreteDIFFUSION =  sasymDIFFUSION - quad_form( sasymDIFFUSION, Je[1:nlatent,1:nlatent]\' );
            etacov = quad_form(etacov, Je\');
            etacov[1:nlatent,1:nlatent] += discreteDIFFUSION;
            eta[1:nlatent] = (discreteDRIFT * append_row(eta[1:nlatent],1.0))[1:nlatent];
          }
        }

        if(continuoustime==0){ //need covariance in here
            for(statei in 0:nlatentpop){
              if(statei>0){
                J[statei,statei] = 1e-6;
                state = eta + J[,statei];
              } else {
                state = eta;
              }
              ',paste0(dynamiccalcs,';\n',collapse=' '),';\n', intoverpopdynamiccalcs,' //remove diffusion calcs and do elsewhere
              if(statei== 0) {
                base = sDRIFT * state[1:nlatent] + sCINT[,1];
              }
              if(statei > 0) J[1:nlatent,statei] = (( sDRIFT * state[1:nlatent] + sCINT[,1]) - base)/1e-6;
            }
          ',paste0(dynamiccalcs,';\n',collapse=' '),'
          discreteDRIFT=append_row(append_col(sDRIFT,sCINT),rep_matrix(0,1,nlatent+1));
          discreteDRIFT[nlatent+1,nlatent+1] = 1;
          etacov = quad_form(etacov, J\');
          etacov[1:nlatent,1:nlatent] += sDIFFUSION;
          if(intoverstates==0) discreteDIFFUSION = cholesky_decompose(makesym(discreteDIFFUSION,verbose,1));
          eta[1:nlatent] = (discreteDRIFT * append_row(eta[1:nlatent],1.0))[1:nlatent];
        }
      } // end of non t0 time update
  
  
    if(nlmeasurement==1 || ntdpred > 0 || T0check[rowi]==1){ //ukf time update
  
      if(T0check[rowi]==1) {
        if(intoverpop==1) sigpoints[(nlatent+1):(nlatentpop), (nlatent+1):(nlatentpop)] = rawpopcovsqrt * sqrtukfadjust;
        sigpoints[1:nlatent,1:nlatent] = sT0VAR * sqrtukfadjust;
      }
      
      if(T0check[rowi]==0)  sigpoints = cholesky_decompose(makesym(etacov,verbose,1)) * sqrtukfadjust;
    
      //configure ukf states
      for(statei in 2:cols(ukfstates) ){ //for each ukf state sample
  
          state = eta; 
          if(statei > (2+nlatentpop)){
            state += -sigpoints[,statei-(2+nlatentpop)];
          } else
          if(statei > 2) state += sigpoints[,statei-2]; 

        if(T0check[rowi]==1){
          ',paste0(t0calcs,';',collapse=' '),'
          state[1:nlatent] += sT0MEANS[,1];
        } 
        ',paste0(tdpredcalcs,';',collapse=' '),'
        if(ntdpred > 0) state[1:nlatent] +=   (sTDPREDEFFECT * tdpreds[rowi]); //tdpred effect only influences at observed time point','
        ukfstates[, statei] = state; //now contains time updated state
        if(statei==2 && ukffull==1) ukfstates[, 1] = state; //mean goes in twice for weighting
      }
  
      if(ukffull == 1) {
        eta = colMeans(ukfstates\');
        etacov = cov_of_matrix(ukfstates\') / asquared;
      }
      if(ukffull == 0){
        eta = ukfstates[,2];
        etacov = tcrossprod(ukfstates[,3:(nlatentpop+2)] - rep_matrix(ukfstates[,2],nlatentpop)) /asquared / (nlatentpop+.5);
      }
    } //end ukf if necessary time update
  } // end non linear time update


  if(savescores==1) kout[rowi,(nmanifest*4+1):(nmanifest*4+nlatentpop)] = eta;
if(verbose > 1) print("etaprior = ", eta, " etapriorcov = ",etacov);

    if(intoverstates==0 && nldynamics == 0) {
      if(T0check[rowi]==1) eta += cholesky_decompose(sT0VAR) * etaupdbasestates[(1+(rowi-1)*nlatent):(rowi*nlatent)];
      if(T0check[rowi]==0) eta +=  discreteDIFFUSION * etaupdbasestates[(1+(rowi-1)*nlatent):(rowi*nlatent)];
    }

    if (nobs_y[rowi] > 0) {  // if some observations create right size matrices for missingness and calculate...
    
      int o[nobs_y[rowi]]= whichobs_y[rowi,1:nobs_y[rowi]]; //which obs are not missing in this row
      int o1[nbinary_y[rowi]]= whichbinary_y[rowi,1:nbinary_y[rowi]];
      int o0[ncont_y[rowi]]= whichcont_y[rowi,1:ncont_y[rowi]];

      if(nlmeasurement==0){ //non ukf measurement
        if(intoverstates==1) { //classic kalman
          ypred[o] = sMANIFESTMEANS[o,1] + sLAMBDA[o,] * eta[1:nlatent];
          if(nbinary_y[rowi] > 0) ypred[o1] = to_vector(inv_logit(to_array_1d(sMANIFESTMEANS[o1,1] +sLAMBDA[o1,] * eta[1:nlatent])));
          ypredcov[o,o] = quad_form(etacov[1:nlatent,1:nlatent], sLAMBDA[o,]\') + sMANIFESTVAR[o,o];
          for(wi in 1:nmanifest){ 
            if(manifesttype[wi]==1 && Y[rowi,wi] != 99999) ypredcov[wi,wi] += fabs((ypred[wi] - 1) .* (ypred[wi]));
            if(manifesttype[wi]==2 && Y[rowi,wi] != 99999) ypredcov[wi,wi] += square(fabs((ypred[wi] - round(ypred[wi])))); 
          }
          K[,o] = mdivide_right(etacov * append_row(sLAMBDA[o,]\',rep_matrix(0,nlatentpop-nlatent,nmanifest)[,o]), ypredcov[o,o]); 
          etacov += -K[,o] * append_col(sLAMBDA[o,],rep_matrix(0,nmanifest,nlatentpop-nlatent)[o,]) * etacov;
        }
        if(intoverstates==0) { //sampled states
          if(ncont_y[rowi] > 0) {
            ypred[o0] = sMANIFESTMEANS[o0,1] + sLAMBDA[o0,] * eta[1:nlatent];
            ypredcov_sqrt[o0,o0] = sMANIFESTVAR[o0,o0];
          }
          if(nbinary_y[rowi] > 0) ypred[o1] = to_vector(inv_logit(to_array_1d(sMANIFESTMEANS[o1,1] +sLAMBDA[o1,] * eta[1:nlatent])));
        }
      } 
  

      if(nlmeasurement==1){ //ukf measurement
        vector[nlatentpop] state; //dynamic portion of current states
        matrix[nmanifest,cols(ukfmeasures)] merrorstates;

        for(statei in 2:cols(ukfmeasures)){
          state = ukfstates[, statei];
          ',paste0(measurementcalcs,';\n',collapse=''),'
  
          ukfmeasures[, statei] = sMANIFESTMEANS[,1] + sLAMBDA * state[1:nlatent];
          if(nbinary_y[rowi] > 0) {
            ukfmeasures[o1 , statei] = to_vector(inv_logit(to_array_1d(ukfmeasures[o1 , statei])));
          }
        
        merrorstates[,statei] = diagonal(sMANIFESTVAR);
        for(wi in 1:nmanifest){ 
          if(manifesttype[wi]==1 && Y[rowi,wi] != 99999) merrorstates[wi,statei] = sMANIFESTVAR[wi,wi] + fabs((ukfmeasures[wi,statei] - 1) .* (ukfmeasures[wi,statei])); //sMANIFESTVAR[wi,wi] + (merror[wi] / cols(ukfmeasures) +1e-8);
          if(manifesttype[wi]==2 && Y[rowi,wi] != 99999) merrorstates[wi,statei] = sMANIFESTVAR[wi,wi] + square(fabs((ukfmeasures[wi,statei]- round(ukfmeasures[wi,statei])))); 
        }
          
          if(statei==2 && ukffull == 1) { 
            merrorstates[,1] = merrorstates[,2];
            ukfmeasures[ , 1] = ukfmeasures [,2];
          }
        } 
        if(ukffull == 1) {
          ypred[o] = colMeans(ukfmeasures[o,]\'); 
          ypredcov[o,o] = cov_of_matrix(ukfmeasures[o,]\') /asquared + diag_matrix(colMeans(merrorstates[o,]\')); //
          K[,o] = mdivide_right(crosscov(ukfstates\', ukfmeasures[o,]\') /asquared, ypredcov[o,o]); 
        }
        if(ukffull == 0){
          ypred[o] = ukfmeasures[o,2];
          for(ci in 3:cols(ukfmeasures)) ukfmeasures[o,ci] += -ukfmeasures[o,2];
          for(ci in 3:cols(ukfstates)) ukfstates[,ci] += -ukfstates[,2];
          ypredcov[o,o] = tcrossprod(ukfmeasures[o,3:(nlatentpop+2)]) /asquared / (nlatentpop +.5) + diag_matrix(merrorstates[o,2]);
          K[,o] = mdivide_right(ukfstates[,3:cols(ukfstates)] * ukfmeasures[o,3:cols(ukfmeasures)]\' /asquared / (nlatentpop+.5), ypredcov[o,o]); 
        }
        etacov +=  - quad_form(ypredcov[o,o],  K[,o]\');
      } //end ukf measurement


'    
,if(ppchecking) paste0('
{
int skipupd = 0;
        for(vi in 1:nobs_y[rowi]){
          if(fabs(ypred[o[vi]]) > 1e10 || is_nan(ypred[o[vi]]) || is_inf(ypred[o[vi]])) {
            skipupd = 1; 
            ypred[o[vi]] =99999;
if(verbose > 1) print("pp ypred problem! row ", rowi);
          }
        }
        if(skipupd==0){ 
          if(ncont_y[rowi] > 0) ypredcov_sqrt[o0,o0]=cholesky_decompose(makesym(ypredcov[o0, o0],verbose,1)); 
          if(ncont_y[rowi] > 0) Ygen[ rowi, o0] = ypred[o0] + ypredcov_sqrt[o0,o0] * Ygenbase[rowi,o0]; 
          if(nbinary_y[rowi] > 0) for(obsi in 1:size(o1)) Ygen[rowi, o1[obsi]] = ypred[o1[obsi]] > Ygenbase[rowi,o1[obsi]] ? 1 : 0; 
          for(vi in 1:nobs_y[rowi]) if(is_nan(Ygen[rowi,o[vi]])) {
            Ygen[rowi,o[vi]] = 99999;
print("pp ygen problem! row ", rowi);
          }
        err[o] = Ygen[rowi,o] - ypred[o]; // prediction error
        }
if(verbose > 1) {
print("rowi ",rowi, "  si ", si, 
          "  eta ",eta,"  etacov ",etacov,"  ypred ",ypred,"  ypredcov ",ypredcov, "  K ",K,
          "  sDRIFT ", sDRIFT, " sDIFFUSION ", sDIFFUSION, " sCINT ", sCINT, "  sMANIFESTVAR ", diagonal(sMANIFESTVAR), "  sMANIFESTMEANS ", sMANIFESTMEANS, 
          "  sT0VAR", sT0VAR, " sT0MEANS ", sT0MEANS,
          "  rawpopsd ", rawpopsd, "  rawpopsdbase ", rawpopsdbase, "  rawpopmeans ", rawpopmeans );
        print("discreteDRIFT ",discreteDRIFT,  "  discreteDIFFUSION ", discreteDIFFUSION)
}
if(verbose > 2) print("ukfstates ", ukfstates, "  ukfmeasures ", ukfmeasures);
}
      '),
  
      if(!ppchecking) 'err[o] = Y[rowi,o] - ypred[o]; // prediction error','
    
      if(savescores==1) {
        int tmpindex[nobs_y[rowi]] = o;
        for(oi in 1:ncont_y[rowi]) tmpindex[oi] +=  nmanifest*2;
        kout[rowi,tmpindex] = err[o];
        for(oi in 1:ncont_y[rowi]) tmpindex[oi] +=  nmanifest;
        kout[rowi,tmpindex] = ypred[o];
      }
      if(intoverstates==1) eta +=  (K[,o] * err[o]);
  
      ',if(!ppchecking){
        'if(nbinary_y[rowi] > 0) kout[rowi,o1] =  Y[rowi,o1] .* (ypred[o1]) + (1-Y[rowi,o1]) .* (1-ypred[o1]); //intoverstates==0 && 
  
        if(verbose > 1) {
          print("rowi ",rowi, "  si ", si, "  eta ",eta,"  etacov ",etacov,
            "  ypred ",ypred,"  ypredcov ",ypredcov, "  K ",K,
            "  sDRIFT ", sDRIFT, " sDIFFUSION ", sDIFFUSION, " sCINT ", sCINT, "  sMANIFESTVAR ", diagonal(sMANIFESTVAR), "  sMANIFESTMEANS ", sMANIFESTMEANS, 
            "  sT0VAR", sT0VAR,  " sT0MEANS ", sT0MEANS,
            "discreteDRIFT ", discreteDRIFT, "  discreteDIFFUSION ", discreteDIFFUSION, "  sasymDIFFUSION ", sasymDIFFUSION, 
            "  rawpopsd ", rawpopsd,  "  rawpopsdbase ", rawpopsdbase, "  rawpopmeans ", rawpopmeans );
        }
        if(verbose > 2) print("ukfstates ", ukfstates, "  ukfmeasures ", ukfmeasures);
  
        if(size(o0) > 0){
          int tmpindex[ncont_y[rowi]] = o0;
          for(oi in 1:ncont_y[rowi]) tmpindex[oi] +=  nmanifest;
           if(intoverstates==1) ypredcov_sqrt[o0,o0]=cholesky_decompose(makesym(ypredcov[o0,o0],verbose,1));
           kout[rowi,o0] = mdivide_left_tri_low(ypredcov_sqrt[o0,o0], err[o0]); //transform pred errors to standard normal dist and collect
           kout[rowi,tmpindex] = log(diagonal(ypredcov_sqrt[o0,o0])); //account for transformation of scale in loglik
        }
      '},'
    }//end nobs > 0 section
  if(savescores==1) kout[rowi,(nmanifest*4+nlatentpop+1):(nmanifest*4+nlatentpop+nlatentpop)] = eta;','
}//end rowi
')}

  kalmanll <- function(){'
  if(sum(nbinary_y) > 0) {
    vector[sum(nbinary_y)] binaryll;
    counter = 1;
    for(ri in 1:ndatapoints){
      int o1[nbinary_y[ri]] = whichbinary_y[ri,1:nbinary_y[ri]]; //which indicators are observed and binary
      binaryll[counter:(counter + nbinary_y[ri]-1)] = kout[ri,o1];
      counter+= nbinary_y[ri];
    }
    ll+= sum(log(binaryll));
  }

  if(( sum(ncont_y) > 0)) {
    vector[sum(ncont_y)] errtrans[2];
    counter = 1;
    for(ri in 1:ndatapoints){
      int o0[ncont_y[ri]] = whichcont_y[ri,1:ncont_y[ri]]; //which indicators are observed and continuous
      errtrans[1,counter:(counter + ncont_y[ri]-1)] = kout[ri, o0];
      for(oi in 1:ncont_y[ri]) o0[oi] +=  nmanifest; //modify o0 index
      errtrans[2,counter:(counter + ncont_y[ri]-1)] = kout[ri, o0];
      counter+= ncont_y[ri];
    }
    ll += normal_lpdf(errtrans[1]|0,1) - sum(errtrans[2]);
    }
'}


subjectparaminit<- function(pop=FALSE,smats=TRUE){
  if(smats && pop) stop('smats and pop cannot both be TRUE!')
  paste0(
   paste0( unlist(lapply(basematrices,function(m){
      paste0('matrix[ ',m,'setup_rowcount ? max(',m,'setup[,1]) : 0, ',m,'setup_rowcount ? max(',m,'setup[,2]) : 0 ] ',
        ifelse(smats,'s',''),ifelse(pop,'pop_',''),m,if(!smats && !pop) paste0('[',m,'subindex[nsubjects]]'),';',collapse='\n')
    })),collapse='\n'),'

  matrix[nlatent,nlatent] ',ifelse(smats,'s',''),ifelse(pop,'pop_',''),'asymDIFFUSION',if(!smats && !pop) '[asymDIFFUSIONsubindex[nsubjects]]','; //stationary latent process variance
  vector[nt0meansstationary ? nlatent : 0] ',ifelse(smats,'s',''),ifelse(pop,'pop_',''),'asymCINT',if(!smats && !pop) '[asymCINTsubindex[nsubjects]]','; // latent process asymptotic level
  ',collapse='\n')
}


subjectparscalc2 <- function(pop=FALSE,smats=TRUE){
  out <- paste0(
 '{
  vector[nparams] rawindparams;
  vector[nparams] tipredaddition;
  vector[nparams] indvaraddition;
  int output = ',as.integer(smats==FALSE && pop==TRUE),';
  
  if(si < 2 || (si > 1 && (nindvarying >0 || ntipred > 0))){ //si of 0 used
    tipredaddition = rep_vector(0,nparams);
    indvaraddition = rep_vector(0,nparams);

    if(si > 0 && nindvarying > 0 && intoverpop==0) indvaraddition[indvaryingindex] = rawpopcovsqrt * baseindparams[(1+(si-1)*nindvarying):(si*nindvarying)];
  
    if(si > 0 &&  ntipred > 0) tipredaddition = TIPREDEFFECT * tipreds[si]\';
  
    rawindparams = rawpopmeans + tipredaddition + indvaraddition;
  }
',

    paste0(unlist(lapply(basematrices, function(m) {
      paste0('
  if(si <= ',m,'subindex[nsubjects]){
    for(ri in 1:size(',m,'setup)){
      if(',m,'setup[ ri,5] > 0 || ',m,'setup[ ri,6] > 0 || si < 2){
        s',m,'[',m,'setup[ ri,1], ',m,'setup[ri,2]] = ',
          m,'setup[ri,3] ? ', 
            'tform(rawindparams[ ',m,'setup[ri,3] ], ',m,'setup[ri,4], ',m,'values[ri,2], ',m,'values[ri,3], ',m,'values[ri,4] ) : ',
            m,'values[ri,1]; //either transformed, scaled and offset free par, or fixed value
      }
    }
  }
      ',collapse='\n')
    })),collapse='\n'),'

  // perform any whole matrix transformations 
    

  if(si <= DIFFUSIONsubindex[nsubjects] &&(output==1|| ukf==0)) sDIFFUSION = sdcovsqrt2cov(sDIFFUSION,ukf);
  if(ukf==0 || output ==1 || nt0varstationary > 0){
    if(si <= asymDIFFUSIONsubindex[nsubjects]) {
      if(ndiffusion < nlatent) sasymDIFFUSION = to_matrix(rep_vector(0,nlatent * nlatent),nlatent,nlatent);

      if(continuoustime==1) sasymDIFFUSION[ derrind, derrind] = to_matrix( 
      -kronsum(sDRIFT[ derrind, derrind ]) \\  to_vector( 
          ( (output==0 && ukf==1) ? tcrossprod(sDIFFUSION[ derrind, derrind ]) : sDIFFUSION[ derrind, derrind ] ) + 
          IIlatent[ derrind, derrind ] * 1e-5), ndiffusion,ndiffusion);

      if(continuoustime==0) sasymDIFFUSION = to_matrix( (IIlatent2 - 
        sqkron_prod(sDRIFT, sDRIFT)) *  to_vector(
          (output==0 && ukf==1) ? tcrossprod(sDIFFUSION[ derrind, derrind ]) : sDIFFUSION[ derrind, derrind ])
        ,ndiffusion, ndiffusion);
    } //end asymdiffusion loops
  }
          
    if(nt0meansstationary > 0){
      if(si <= asymCINTsubindex[nsubjects]){
        if(continuoustime==1) sasymCINT =  -sDRIFT \\ sCINT[ ,1 ];
        if(continuoustime==0) sasymCINT =  (IIlatent - sDRIFT) \\ sCINT[,1 ];
      }
    }

          
    if(binomial==0){
      if(si <= MANIFESTVARsubindex[nsubjects]) {
         for(ri in 1:nmanifest) sMANIFESTVAR[ri,ri] = square(sMANIFESTVAR[ri,ri]);
      }
    }
          
          
    if(si <= T0VARsubindex[nsubjects]) {
      if(ukf==0 || output==1) sT0VAR = sdcovsqrt2cov(sT0VAR,ukf);
      if(nt0varstationary > 0) {
        if(ukf==1 && output ==0) sasymDIFFUSION = cholesky_decompose(makesym(sasymDIFFUSION,verbose,0));
          for(ri in 1:nt0varstationary){ // in case of negative variances
          sT0VAR[t0varstationary[ri,1],t0varstationary[ri,2] ] = 
           t0varstationary[ri,1] == t0varstationary[ri,2] ? 
            sqrt(square(sasymDIFFUSION[t0varstationary[ri,1],t0varstationary[ri,2] ])) : 
            sasymDIFFUSION[t0varstationary[ri,1],t0varstationary[ri,2] ];
        }
      }
    }
    
    if(nt0meansstationary > 0){
      if(si <= T0MEANSsubindex[nsubjects]) {
        for(ri in 1:nt0meansstationary){
          sT0MEANS[t0meansstationary[ri,1] , 1] = 
            sasymCINT[t0meansstationary[ri,1] ];
        }
      }
    }
  ',if(!smats) paste0('
  if(si > 0){
',collectsubmats(pop=FALSE),'
  }',collapse=''),'
',if(pop) paste0('
  if(si == 0){
',collectsubmats(pop=TRUE),'
  }
',collapse=''),'
}
  ',collapse='\n')
  
  return(out)
}

collectsubmats <- function(pop=FALSE,matrices=c(basematrices,'asymDIFFUSION','asymCINT')){
  out<-''
  for(m in matrices){
    if(!pop) out <- paste0(out, m,'[',m,'subindex[si]] = s',m,'; \n')
    if(pop) out <- paste0(out, 'pop_',m,' = s',m,'; \n')
  }
  return(out)
}
    
matsetup <-list()
matvalues <-list()
freepar <- 0
freeparcounter <- 0
indvaryingindex <-array(0,dim=c(0))
indvaryingcounter <- 0
TIPREDEFFECTsetup <- matrix(0,0,n.TIpred)
tipredcounter <- 1
indvar <- 0
extratformcounter <- 0
extratforms <- c()
for(m in basematrices){
  mdat<-matrix(0,0,6)
  mval<-matrix(0,0,5)
  for(i in 1:nrow(ctspec)){ 
    if(ctspec$matrix[i] == m) {
      
      if(!is.na(ctspec$param[i]) & !grepl('[',ctspec$param[i],fixed=TRUE)){ #if a free parameter,
        if(i > 1 && any(ctspec$param[1:(i-1)] %in% ctspec$param[i])){ #and after row 1, check for duplication
          freepar <- mdat[,'param'][ match(ctspec$param[i], rownames(mdat)) ] #find which freepar corresponds to duplicate
          indvar <- ifelse(ctspec$indvarying[i],  mdat[,'indvarying'][ match(ctspec$param[i], rownames(mdat)) ],0)#and which indvar corresponds to duplicate
        } else { #if not duplicated
          freeparcounter <- freeparcounter + 1
          TIPREDEFFECTsetup <- rbind(TIPREDEFFECTsetup,rep(0,ncol(TIPREDEFFECTsetup))) #add an extra row...
          if(ctspec$indvarying[i]) {
            indvaryingcounter <- indvaryingcounter + 1
            indvar <- indvaryingcounter
          }
          if(!ctspec$indvarying[i]) indvar <- 0
          freepar <- freeparcounter
          if(is.na(suppressWarnings(as.integer(ctspec$transform[i])))) { #extra tform needed
            extratformcounter <- extratformcounter + 1
            extratforms <- paste0(extratforms,'if(transform==',-10-extratformcounter,') out = ',
              ctspec$offset[i],' + ',ctspec$multiplier[i],' * (',
              gsub('param', paste0('param * ',ctspec$meanscale[i]),ctspec$transform[i]),');')
            ctspec$transform[i] <- -10-extratformcounter
          }
          if(n.TIpred > 0) {
            TIPREDEFFECTsetup[freepar,][ ctspec[i,paste0(TIpredNames,'_effect')]==TRUE ] <- 
              tipredcounter: (tipredcounter + sum(as.integer(suppressWarnings(ctspec[i,paste0(TIpredNames,'_effect')]))) -1)
            tipredcounter<- tipredcounter + sum(as.integer(suppressWarnings(ctspec[i,paste0(TIpredNames,'_effect')])))
          }
        }
        # if(intoverpop && ctspec$indvarying[i]) {
        #   mdynadd <- 
        # }
      }
      
      mdatnew <- matrix(c(
        ctspec$row[i],
        ctspec$col[i],
        ifelse(!is.na(ctspec$param[i]),freepar, 0),
        ifelse(is.na(as.integer(ctspec$transform[i])), -1, as.integer(ctspec$transform[i])),
        ifelse(!is.na(ctspec$param[i]),indvar,0),
        ifelse(any(TIPREDEFFECTsetup[freepar,] > 0), 1, 0)
        ),nrow=1)
      rownames(mdatnew) <- ctspec$param[i]
      
      mdat<-rbind(mdat,mdatnew)
      
      # if(ctspec$indvarying[i]) indvaryingindex <- c(indvaryingindex, freepar)
      
      mval<-rbind(mval, matrix(c(ctspec$value[i], ctspec$multiplier[i], ctspec$meanscale[i],ctspec$offset[i], ctspec$sdscale[i]),ncol=5))
      colnames(mdat) <- c('row','col','param','transform', 'indvarying','tipred')
      colnames(mval) <- c('value','multiplier','meanscale','offset','sdscale')
    }
  }
  if(!is.null(mval)) mval[is.na(mval)] <- 99999 else mval<-array(0,dim=c(0,4))

  matsetup[[m]] = mdat
  matvalues[[m]] <- mval
}

  popsetup <- do.call(rbind,matsetup) 
  popvalues <- do.call(rbind,matvalues) 
  popvalues <- popvalues[popsetup[,'param'] !=0,,drop=FALSE]
  popsetup <- popsetup[popsetup[,'param'] !=0,,drop=FALSE]
  
  parname <-rownames(popsetup)
  rownames(popsetup) <- NULL
  rownames(popvalues) <- NULL

  popsetup <- data.frame(parname,popsetup,stringsAsFactors = FALSE)
  popvalues <- data.frame(parname,popvalues,stringsAsFactors = FALSE)

  nindvarying <- max(popsetup$indvarying)
  nparams <- max(popsetup$param)

  indvaryingindex <- popsetup$param[which(popsetup$indvarying > 0)]
  indvaryingindex <- array(indvaryingindex[!duplicated(indvaryingindex)])
  sdscale <- array(popvalues$sdscale[match(indvaryingindex,popsetup$param)])


  if(any(popsetup[,'transform'] < -10)) recompile <- TRUE #if custom transforms needed

  writemodel<-function(){
    paste0('
functions{

   matrix covsqrt2corsqrt(matrix mat, int invert){ //converts from unconstrained lower tri matrix to cor
    matrix[rows(mat),cols(mat)] o;
    vector[rows(mat)] s;
  
    for(i in 1:rows(o)){ //set upper tri to lower
      for(j in min(i+1,rows(mat)):rows(mat)){
        o[j,i] = inv_logit(mat[j,i])*2-1;  // can change cor prior here
        o[i,j] = o[j,i];
      }
      o[i,i]=1; // change to adjust prior for correlations
    }
  
    if(invert==1) o = inverse(o);
  
    for(i in 1:rows(o)){
      s[i] = inv_sqrt(o[i,] * o[,i]);
      if(is_inf(s[i])) s[i]=0;
    }
    return diag_pre_multiply(s,o);
  }

  int[] checkoffdiagzero(matrix M){
    int z[rows(M)];
    for(i in 1:rows(M)){
      z[i] = 0;
      for(j in 1:cols(M)){
        if(i!=j){
          if(M[i,j] != 0){
            z[i] = 1;
            break;
          }
        }
      }
      if(z[i]==0){ //check rows
        for(j in 1:rows(M)){
          if(i!=j){
            if(M[j,i] != 0){
              z[i] = 1;
              break;
            }
          }
        }
      }
    }
  return z;
  }

  matrix expm2(matrix M){
    matrix[rows(M),rows(M)] out;
    int z[rows(out)] = checkoffdiagzero(M);
    int z1[sum(z)];
    int z0[rows(M)-sum(z)];
    int cz1 = 1;
    int cz0 = 1;
    for(i in 1:rows(M)){
      if(z[i] == 1){
        z1[cz1] = i;
        cz1 += 1;
      }
      if(z[i] == 0){
        z0[cz0] = i;
        cz0 += 1;
      }
    }
    if(size(z1) > 0) out[z1,z1] = matrix_exp(M[z1,z1]);
    out[z0,] = rep_matrix(0,size(z0),rows(M));
    out[,z0] = rep_matrix(0,rows(M),size(z0));
    for(i in 1:size(z0)) out[z0[i],z0[i]] = exp(M[z0[i],z0[i]]);
    return out;
  }
      
  matrix sdcovsqrt2cov(matrix mat, int cholbasis){ //converts from lower partial sd and diag sd to cov or cholesky cov
    if(cholbasis==0)  {
      return(tcrossprod(diag_pre_multiply(diagonal(mat),covsqrt2corsqrt(mat, 0))));
    } else return(tcrossprod(mat));
  }

  matrix sqkron_prod(matrix mata, matrix matb){
    int m=rows(mata);
    int p=rows(matb);
    int n=cols(mata);
    int q=cols(matb);
    matrix[rows(mata)*rows(matb),cols(mata)*cols(matb)] out;
    for (k in 1:p){
      for (l in 1:q){
        for (i in 1:m){
          for (j in 1:n){
            out[ p*(i-1)+k, q*(j-1)+l ] = mata[i, j] * matb[k, l];
          }
        }
      }
    }
    return out;
  }

  matrix kronsum(matrix mata){
    matrix[rows(mata),rows(mata)] II = diag_matrix(rep_vector(1,rows(mata)));
    return sqkron_prod(mata, II) + sqkron_prod(II, mata );
  }

  vector colMeans(matrix mat){
    vector[cols(mat)] out;
    for(i in 1:cols(mat)){
      out[i] = mean(mat[,i]);
    }
    return out;
  }

  matrix cov_of_matrix(matrix mat){
    vector[cols(mat)] means = colMeans(mat);
    matrix[rows(mat), cols(mat)] centered;
    matrix[cols(mat), cols(mat)] covm;
    for (coli in 1:cols(mat)){
      for (ri in 1:rows(mat)){
        centered[ri,coli] = mat[ri,coli] - means[coli];
      }
    }
    covm = crossprod(centered) / (rows(mat)-1);
    return covm; 
  }

  matrix crosscov(matrix a, matrix b){
    matrix[rows(a),cols(a)] da;
    matrix[rows(b),cols(b)] db;
    matrix[cols(a),cols(b)] out;
  
    da = a - rep_matrix( (colMeans(a))\',rows(a));
    db = b - rep_matrix( (colMeans(b))\',rows(b));
    out = da\' * db ./ (rows(a)-1.0);
    return out;
  }

  matrix makesym(matrix mat, int verbose, int pd){
    matrix[rows(mat),cols(mat)] out;
    for(coli in 1:cols(mat)){
      if(pd ==1 && mat[coli,coli] < 1e-5){
        if(verbose > 0) print("diagonal too low (",mat[coli,coli],") during makesym row ", coli, " col ", coli);
        out[coli,coli] = 1e-5;
      } else out[coli,coli] = mat[coli,coli]; 
      for(rowi in coli:rows(mat)){
        if(rowi > coli) {
          out[rowi,coli] = mat[rowi,coli]; //(mat[coli,rowi] + ) *.5;
          out[coli,rowi] = mat[rowi,coli];
        }
        if(is_nan(out[rowi,coli])){
          if(verbose > 0) print("nan during makesym row ", rowi, " col ", coli);
          if(rowi==coli) out[rowi,coli] = 99999;
          if(rowi!=coli) {
            out[rowi,coli] = 0;
            out[coli,rowi] = 0;
          }
        }
      }
    }
    return out;
  }

  real tform(real param, int transform, data real multiplier, data real meanscale, data real offset){
    real out;
  
    ',tformshapes(),
  if(extratformcounter > 0) extratforms,'

    return out;
  }

}
data {
  int<lower=0> ndatapoints;
  int<lower=1> nmanifest;
  int<lower=1> nlatent;
  int<lower=1> nsubjects;
  int<lower=0> ntipred; // number of time independent covariates
  int<lower=0> ntdpred; // number of time dependent covariates

  int T0check[ndatapoints]; // logical indicating which rows are the first for each subject
  matrix[ntipred ? nsubjects : 0, ntipred ? ntipred : 0] tipredsdata;
  int nmissingtipreds;
  int ntipredeffects;
  real<lower=0> tipredsimputedscale;
  real<lower=0> tipredeffectscale;

  vector[nmanifest] Y[ndatapoints];
  int nopriors;
  int nldynamics;
  vector[ntdpred] tdpreds[ntdpred ? ndatapoints : 0];
  
  real dT[ndatapoints]; // time intervals
  real dTsmall[ndatapoints];
  int binomial; //binary data only
  int integrationsteps[ndatapoints] ; // time steps needed between time intervals for integration
  int subject[ndatapoints];
  int<lower=0> nparams;
  int continuoustime; // logical indicating whether to incorporate timing information
  int nindvarying; // number of subject level parameters that are varying across subjects
  int nindvaryingoffdiagonals; //number of off diagonal parameters needed for popcov matrix
  vector[nindvarying] sdscale;
  int indvaryingindex[nindvarying];

  int nt0varstationary;
  int nt0meansstationary;
  int t0varstationary [nt0varstationary, 2];
  int t0meansstationary [nt0meansstationary, 2];

  int nobs_y[ndatapoints];  // number of observed variables per observation
  int whichobs_y[ndatapoints, nmanifest]; // index of which variables are observed per observation
  int ndiffusion; //number of latents involved in covariance calcs
  int derrind[ndiffusion]; //index of which latent variables are involved in covariance calculations

  int manifesttype[nmanifest];
  int nbinary_y[ndatapoints];  // number of observed binary variables per observation
  int whichbinary_y[ndatapoints, nmanifest]; // index of which variables are observed and binary per observation
  int ncont_y[ndatapoints];  // number of observed continuous variables per observation
  int whichcont_y[ndatapoints, nmanifest]; // index of which variables are observed and continuous per observation
  
  int intoverpop;
  int ukf;
  real ukfspread;
  int ukffull;
  int nlmeasurement;
  int intoverstates;
  int verbose; //level of printing during model fit

  ',paste0(unlist(lapply(c(basematrices,'asymCINT','asymDIFFUSION'),function(mati) paste0('int ',mati,'subindex[nsubjects];',collapse='\n'))),collapse='\n'),'
  ',paste0(unlist(lapply(basematrices,function(m) paste0('int ',m,'setup_rowcount;',collapse='\n'))),collapse='\n'),'
  ',paste0(unlist(lapply(basematrices,function(m) paste0('int ',m,'setup[',m,'setup_rowcount,6 ];',collapse='\n'))),collapse='\n'),'
  ',paste0(unlist(lapply(basematrices,function(m) paste0('matrix[',m,'setup_rowcount, 5] ',m,'values;'))),collapse='\n'),'
  int TIPREDEFFECTsetup[nparams, ntipred];
  int nmatrixslots;
  int popsetup[nmatrixslots,6];
  real popvalues[nmatrixslots,5];
  int savescores;
  int gendata;
}
      
transformed data{
  matrix[nlatent,nlatent] IIlatent;
  matrix[nlatent*nlatent,nlatent*nlatent] IIlatent2;
  int nlatentpop;
  real asquared;
  real sqrtukfadjust;
  nlatentpop = intoverpop ? nlatent + nindvarying : nlatent;
  IIlatent = diag_matrix(rep_vector(1,nlatent));
  IIlatent2 = diag_matrix(rep_vector(1,nlatent*nlatent));

  //ukf approximation parameters
  asquared =  square(2.0/sqrt(0.0+nlatentpop) * ukfspread); 
  sqrtukfadjust = sqrt(0.0+nlatentpop +( asquared * (nlatentpop  + 0.5) - (nlatentpop) ) );
}
      
parameters {
  vector[nparams] rawpopmeans; // population level means \n','
  vector',if(!is.na(ctstanmodel$rawpopsdbaselowerbound)) paste0('<lower=',ctstanmodel$rawpopsdbaselowerbound[1],'>'),'[nindvarying ? 1 : 0] rawpopsdbase; //population level std dev
  simplex[nindvarying] rawpopsdprops[nindvarying ? 1 : 0];
  vector[nindvaryingoffdiagonals] sqrtpcov; // unconstrained basis of correlation parameters
  vector[intoverpop ? 0 : nindvarying*nsubjects] baseindparams; //vector of subject level deviations, on the raw scale
  
  vector[ntipredeffects] tipredeffectparams; // effects of time independent covariates
  vector[nmissingtipreds] tipredsimputed;
  
  vector[intoverstates ? 0 : nlatent*ndatapoints] etaupdbasestates; //sampled latent states posterior
}
      
transformed parameters{
  vector[nindvarying] rawpopsd; //population level std dev
  matrix[nindvarying,nindvarying] rawpopcovsqrt; 
  real ll;
  vector[nmanifest+nmanifest+ (savescores ? nmanifest*2+nlatentpop*2 : 0)] kalman[savescores ? ndatapoints : 0];

  matrix[ntipred ? nsubjects : 0, ntipred ? ntipred : 0] tipreds; //tipred values to fill from data and, when needed, imputation vector
  matrix[nparams, ntipred] TIPREDEFFECT; //design matrix of individual time independent predictor effects

  if(ntipred > 0){ 
    int counter = 0;
    for(coli in 1:cols(tipreds)){ //insert missing ti predictors
      for(rowi in 1:rows(tipreds)){
        if(tipredsdata[rowi,coli]==99999) {
          counter += 1;
          tipreds[rowi,coli] = tipredsimputed[counter];
        } else tipreds[rowi,coli] = tipredsdata[rowi,coli];
      }
    }
    for(ci in 1:ntipred){ //configure design matrix
      for(ri in 1:nparams){
        if(TIPREDEFFECTsetup[ri,ci] > 0) {
          TIPREDEFFECT[ri,ci] = tipredeffectparams[TIPREDEFFECTsetup[ri,ci]];
        } else {
          TIPREDEFFECT[ri,ci] = 0;
        }
      }
    }
  }

  if(nindvarying > 0){
    int counter =0;
    rawpopsd = sqrt(rawpopsdprops[1] * square(',ctstanmodel$rawpopsdtransform, ')[1]) .*sdscale; // sqrts of proportions of total variance
    for(j in 1:nindvarying){
      rawpopcovsqrt[j,j] = 1;
      for(i in 1:nindvarying){
        if(i > j){
          counter += 1;
          rawpopcovsqrt[i,j]=sqrtpcov[counter];
          rawpopcovsqrt[j,i]=sqrtpcov[counter];
        }
      }
    }
    rawpopcovsqrt = diag_pre_multiply(rawpopsd, 
      covsqrt2corsqrt(rawpopcovsqrt,0)); 
  }//end indvarying par setup

  ll=0;
{',
ukfilterfunc(ppchecking=FALSE),
 kalmanll(),'
if(savescores) kalman = kout;
}

}
      
model{

  if(nopriors==0){
    target += normal_lpdf(rawpopmeans|0,1);
  
    if(ntipred > 0){ 
      tipredeffectparams ~ normal(0,tipredeffectscale);
      tipredsimputed ~ normal(0,tipredsimputedscale);
    }
    
    if(nindvarying > 0){
      if(nindvarying >1) sqrtpcov ~ normal(0,1);
      if(intoverpop==0) baseindparams ~ normal(0,1);
      rawpopsdbase ~ ',ctstanmodel$rawpopsdbase,';
    }
  } //end pop priors section
  
  if(intoverstates==0) etaupdbasestates ~ normal(0,1);
  
  target += ll;
  if(verbose > 0) print("lp = ", target());
}
generated quantities{
  vector[nparams] popmeans;
  vector[nparams] popsd;
  matrix[nindvarying,nindvarying] rawpopcov;
  matrix[nindvarying,nindvarying] rawpopcorr;
  matrix[nparams,ntipred] linearTIPREDEFFECT;
  vector[nmanifest] Ygen[gendata ? ndatapoints : 0];

  ',subjectparaminit(pop=TRUE,smats=FALSE),'

  ',subjectparaminit(pop=FALSE,smats=FALSE),'

  {
  ',subjectparaminit(pop=FALSE,smats=TRUE),'

  for(si in 0:subject[ndatapoints]){
    ',subjectparscalc2(pop=TRUE,smats=FALSE),'
    }
  }

rawpopcov = tcrossprod(rawpopcovsqrt);
rawpopcorr = quad_form_diag(rawpopcov,inv_sqrt(diagonal(rawpopcov)));

popsd = rep_vector(0,nparams);
{
vector[nparams] rawpopsdfull;
rawpopsdfull[indvaryingindex] = rawpopsd; //base for calculations

    for(ri in 1:dims(popsetup)[1]){
      if(popsetup[ri,3] !=0) {

        popmeans[popsetup[ ri,3]] = tform(rawpopmeans[popsetup[ri,3] ], popsetup[ri,4], popvalues[ri,2], popvalues[ri,3], popvalues[ri,4] ); 

        popsd[popsetup[ ri,3]] = popsetup[ ri,5] ? 
          fabs(tform(
            rawpopmeans[popsetup[ri,3] ]  + rawpopsdfull[popsetup[ ri,3]], popsetup[ri,4], popvalues[ri,2], popvalues[ri,3], popvalues[ri,4]) -
           tform(
            rawpopmeans[popsetup[ri,3] ]  - rawpopsdfull[popsetup[ ri,3]], popsetup[ri,4], popvalues[ri,2], popvalues[ri,3], popvalues[ri,4]) ) /2 : 
          0; 

        if(ntipred > 0){
          for(tij in 1:ntipred){
            if(TIPREDEFFECTsetup[popsetup[ri,3],tij] ==0) {
              linearTIPREDEFFECT[popsetup[ri,3],tij] = 0;
            } else {
            linearTIPREDEFFECT[popsetup[ri,3],tij] = (
              tform(rawpopmeans[popsetup[ri,3] ] + TIPREDEFFECT[popsetup[ri,3],tij] * .01, popsetup[ri,4], popvalues[ri,2], popvalues[ri,3], popvalues[ri,4] ) -
              tform(rawpopmeans[popsetup[ri,3] ] - TIPREDEFFECT[popsetup[ri,3],tij] * .01, popsetup[ri,4], popvalues[ri,2], popvalues[ri,3], popvalues[ri,4] )
              ) /2 * 100;
            }
         }
        }
      }
    }
}

',if(gendata) paste0('
  if(gendata > 0){
  vector[nmanifest] Ygenbase[ndatapoints];
  Ygen = rep_array(rep_vector(99999,nmanifest),ndatapoints);
  for(mi in 1:nmanifest){
    if(manifesttype[mi]==0 || manifesttype[mi]==2) {
      Ygenbase[1:ndatapoints,mi] = normal_rng(rep_vector(0,gendata*ndatapoints),rep_vector(1,gendata*ndatapoints));
    }
    if(manifesttype[mi]==1){
      Ygenbase[1:ndatapoints,mi] =  uniform_rng(rep_vector(0,gendata*ndatapoints),rep_vector(1,gendata*ndatapoints));
    }
  }
{
',ukfilterfunc(ppchecking=TRUE),'

}}
',collapse=";\n"),'

}
')
}


  
  if(is.na(stanmodeltext)) stanmodeltext<-writemodel() else recompile<-TRUE

  # out<-list(stanmodeltext=stanmodeltext)
  
  standata<-list(
    Y=cbind(as.matrix(datalong[,manifestNames])),
    subject=as.integer(datalong[,idName]),
    time=datalong[,ctstanmodel$timeName], #not used in model but used elsewhere
    nsubjects=as.integer(nsubjects),
    nmanifest=as.integer(n.manifest),
    nldynamics=as.integer(nldynamics),
    integrationsteps=as.integer(integrationsteps),
    intoverstates=as.integer(intoverstates),
    verbose=as.integer(verbose),
    indvaryingindex=array(as.integer(indvaryingindex)),
    continuoustime=as.integer(sum(continuoustime)),
    nlatent=as.integer(n.latent),
    ntipred=as.integer(n.TIpred),
    ntdpred=as.integer(n.TDpred),
    binomial=as.integer(binomial),
    nparams=as.integer(nparams),
    gendata=as.integer(gendata),
    nindvarying=as.integer(nindvarying),
    nindvaryingoffdiagonals=as.integer((nindvarying^2-nindvarying)/2),
    ndatapoints=as.integer(nrow(datalong)),
    dT=dT,
    T0check=as.integer(T0check),
    dTsmall=dTsmall,
    nt0varstationary=as.integer(nt0varstationary),
    nt0meansstationary=as.integer(nt0meansstationary),
    t0varstationary=matrix(as.integer(t0varstationary),ncol=2),
    t0meansstationary=matrix(as.integer(t0meansstationary),ncol=2),
    derrind=array(as.integer(derrind),dim=ndiffusion),
    ndiffusion=as.integer(ndiffusion),
    driftdiagonly = as.integer(driftdiagonly),
    intoverpop=as.integer(intoverpop),
    ukf=as.integer(ukf),
    ukfspread = nlcontrol$ukfspread,
    ukffull = as.integer(nlcontrol$ukffull),
    nlmeasurement=as.integer(nlmeasurement),
    nopriors=as.integer(nopriors),
    savescores=as.integer(savescores),
    manifesttype=array(as.integer(manifesttype),dim=length(manifesttype)),
    nobs_y=array(as.integer(apply(datalong[,manifestNames,drop=FALSE],1,function(x) length(x[x!=99999]))),dim=nrow(datalong)),
    whichobs_y=matrix(as.integer(t(apply(datalong[,manifestNames,drop=FALSE],1,function(x) {
      out<-as.numeric(which(x!=99999))
      if(length(out)==0) out<-rep(0,n.manifest)
      if(length(out)<n.manifest) out<-c(out,rep(0,n.manifest-length(out)))
      out
    }) )),nrow=c(nrow(datalong),ncol=n.manifest)),
    nbinary_y=array(as.integer(apply(datalong[,manifestNames,drop=FALSE],1,function(x) 
       length(x[manifesttype==1 & x!=99999]))),dim=nrow(datalong)),
    whichbinary_y=matrix(as.integer(t(apply(datalong[,manifestNames,drop=FALSE],1,function(x) {
      out<-as.numeric(which(manifesttype==1 & x!=99999)) #conditional on whichobs_y
      # if(ukf==TRUE || intoverstates==FALSE) out<- as.numeric(which(manifesttype==1)) else out<- rep(0,n.manifest) #not conditional on whichobs_y
      if(length(out)==0) out<-rep(0,n.manifest)
      if(length(out)<n.manifest) out<-c(out,rep(0,n.manifest-length(out)))
      out
    }) )),nrow=c(nrow(datalong),ncol=n.manifest)),
    ncont_y=array(as.integer(apply(datalong[,manifestNames,drop=FALSE],1,function(x) 
      length(x[(manifesttype==0 || manifesttype==2) & x!=99999]))),dim=nrow(datalong)),
    whichcont_y=matrix(as.integer(t(apply(datalong[,manifestNames,drop=FALSE],1,function(x) {
      out<-as.numeric(which( (manifesttype==0 | manifesttype==2) & x!=99999)) #conditional on whichobs_y
      # if(ukf==TRUE || intoverstates==FALSE) out<- as.numeric(which(manifesttype==0 | manifesttype==2)) else out<-  rep(1,n.manifest) #not conditional on whichobs_y
      if(length(out)==0) out<-rep(0,n.manifest)
      if(length(out)<n.manifest) out<-c(out,rep(0,n.manifest-length(out)))
      out
    }) )),nrow=c(nrow(datalong),ncol=n.manifest))
    )

  if(n.TIpred == 0) tipreds <- array(0,c(0,0))
  standata$tipredsdata <- as.matrix(tipreds)
  standata$nmissingtipreds <- as.integer(length(tipreds[tipreds== 99999]))
  standata$ntipredeffects <- as.integer(ifelse(n.TIpred > 0, as.integer(max(TIPREDEFFECTsetup)), 0))
  standata$TIPREDEFFECTsetup <- apply(TIPREDEFFECTsetup,c(1,2),as.integer,.drop=FALSE)
  standata$tipredsimputedscale <- ctstanmodel$tipredsimputedscale
  standata$tipredeffectscale <- ctstanmodel$tipredeffectscale
  
  if(n.TDpred ==0) tdpreds <- matrix(0,0,0)
  standata$tdpreds=array(as.matrix(tdpreds),dim=c(nrow(tdpreds),ncol(tdpreds)))
  
  #add subject variability indices to data
  for(mati in c(basematrices,'asymCINT','asymDIFFUSION')){
    sname <- paste0(mati,'subindex')
    standata[[sname]] <- array(as.integer((get(sname))),dim=nsubjects)
  }

  for(m in basematrices){
    standata[[paste0(m,'setup_rowcount')]] <- as.integer(nrow(matsetup[[m]]))
    standata[[paste0(m,'setup')]] <- apply(matsetup[[m]],c(1,2),as.integer,.drop=FALSE)
    standata[[paste0(m,'values')]] <- matvalues[[m]]
  }

  standata$popsetup <- apply(popsetup[,-1],c(1,2),as.integer,.drop=FALSE) #with parname column removed
  standata$popvalues <- apply(popvalues[,-1],c(1,2),as.numeric)
  standata$nmatrixslots <- as.integer(nrow(popsetup))
  
  standata$sdscale <- sdscale

  if(fit){

    if(recompile || forcerecompile) {
     message('Compiling model...') 
      sm <- stan_model(model_name='ctsem', model_code = c(stanmodeltext))
    }
    if(!recompile && !forcerecompile) sm <- stanmodels$ctsm

    if(!is.null(inits)&& inits!=0){
      sf <- stan_reinitsf(sm,standata)
      inits <- constrain_pars(sf,inits)
      staninits=list()
      if(chains > 0){
        for(i in 1:chains){
          staninits[[i]]<-inits
        }
      }
    }
    
    if(!is.null(inits)&& inits==0) staninits=0
    
    
    if(is.null(inits)){
      staninits=list()
      if(chains > 0){
        for(i in 1:(chains)){
          staninits[[i]]=list(
            rawpopmeans=array(rnorm(nparams,0,.1)),
            rawpopsdbase=array(rnorm(ifelse(nindvarying >0,1,0),0,.1)),
            sqrtpcov=array(rnorm((nindvarying^2-nindvarying)/2,0,.1)),
            baseindparams=array(rnorm(ifelse(intoverpop,0,nsubjects*nindvarying),0,.1)),
            eta=array(stats::rnorm(nrow(datalong)*n.latent,0,.1),dim=c(nrow(datalong),n.latent)),
            tipredeffectparams=array(rnorm(standata$ntipredeffects,0,.1))
            )
          if(!is.na(ctstanmodel$rawpopsdbaselowerbound)) staninits[[i]]$rawpopsdbase=exp(staninits[[i]]$rawpopsdbase)
        }
      }
    }
    
    if(!optimize){
      
      #control arguments for rstan
      # if(is.null(control$adapt_term_buffer)) control$adapt_term_buffer <- min(c(iter/10,max(iter-20,75)))
      if(is.null(control$adapt_delta)) control$adapt_delta <- .8
      if(is.null(control$adapt_window)) control$adapt_window <- 2
      if(is.null(control$max_treedepth)) control$max_treedepth <- 10
      if(is.null(control$adapt_init_buffer)) control$adapt_init_buffer=2
      if(is.null(control$stepsize)) control$stepsize=.001
      
      message('Sampling...')
      
      stanargs <- list(object = sm, 
        # enable_random_init=TRUE,
        init_r=.1,
        init=staninits,
        refresh=20,
        iter=iter,
        data = standata, chains = chains, control=control,
        cores=cores,
        ...) 
      if(plot==TRUE) stanfit <- do.call(stanWplot,stanargs) else stanfit <- do.call(sampling,stanargs)
    }
    
    if(optimize==TRUE) {
      opargs <- c(list(standata = standata,sm = sm,init = staninits[[1]], cores=cores, verbose=verbose,nopriors=as.logical(nopriors)),optimcontrol)
      stanfit <- do.call(optimstan,opargs)
    }
    
  } # end if fit==TRUE
  #convert missings back to NA's for data output
  standataout<-unlist(standata)
  standataout[standataout==99999] <- NA
  standataout <- utils::relist(standataout,skeleton=standata)
  
  if(fit) {
    out <- list(args=args,
    setup=list(recompile=recompile,popsetup=popsetup,idmap=idmap,popsetup=popsetup,popvalues=popvalues,basematrices=basematrices,extratforms=extratforms), 
    stanmodeltext=stanmodeltext, data=standataout, standata=standata, ctstanmodel=ctstanmodel,stanmodel=sm, stanfit=stanfit)
  class(out) <- 'ctStanFit'
  }
  
  # matrixsetup <- list(matsetup,basematrices,dynamicmatrices,measurementmatrices,t0matrices)
  # names(matrixsetup) <- c('matsetup','basematrices','dynamicmatrices','measurementmatrices','t0matrices')
  if(!fit) out=list(args=args,setup=list(recompile=recompile,idmap=idmap,popsetup=popsetup,popvalues=popvalues,basematrices=basematrices,extratforms=extratforms),
    stanmodeltext=stanmodeltext,data=standataout, standata=standata, ctstanmodel=ctstanmodel)
  
  
  return(out)
}

