#' Compute a Tab Book
#'
#' This function allows you to generate a tab book from a multitable and data.
#' As with other functions, you can select the rows and columns you want to
#' work with by subsetting the \code{dataset} you pass into the function.
#'
#' By specifying a "json" \code{format}, instead of generating an Excel
#' workbook, you'll get a TabBookResult object, containing nested CrunchCube
#' results. You can then further format these and construct custom tab reports.
#' @param multitable a \code{Multitable} object
#' @param dataset CrunchDataset, which may have been subsetted with a filter
#' expression on the rows and a selection of variables on the columns.
#' @param weight a CrunchVariable that has been designated as a potential
#' weight variable for \code{dataset}, or \code{NULL} for unweighted results.
#' Default is the currently applied weight, \code{\link{weight}(dataset)}.
#' @param format character export format: currently supported values are "json"
#' (default) and "xlsx".
#' @param file character local filename to write to. A default filename will be
#' generated from the \code{multitable}'s name if one is not supplied and the
#' "xlsx" format is requested. Not required for "json" format export.
#' @param ... Additional "options" passed to the tab book POST request.
#' @return If "json" format is requested, the function returns an object of
#' class \code{TabBookResult}, containing a list of \code{MultitableResult}
#' objects, which themselves contain \code{CrunchCube}s. If "xlsx" is requested,
#' the function invisibly returns the filename (\code{file} if specified, else
#' the autogenerated file name). If you request "json" and wish to access the
#' JSON data underlying the \code{TabBookResult}, pass in a path for \code{file}
#' and you will get a JSON file written there as well.
#' @examples
#' \dontrun{
#' m <- newMultitable(~ gender + age4 + marstat, data=ds)
#' tabBook(m, ds[ds$income > 1000000,], format="xlsx", file="wealthy-tab-book.xlsx")
#' book <- tabBook(m, ds) # Returns a TabBookResult
#' tables <- prop.table(book, 2)
#' }
#' @importFrom jsonlite fromJSON
#' @export
tabBook <- function (multitable, dataset, weight=crunch::weight(dataset),
                    format=c("json", "xlsx"), file, ...) {

    f <- match.arg(format)
    accept <- list(
        json="application/json",
        xlsx="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
    )[[f]]
    if (missing(file)) {
        if (f == "json") {
            ## We don't need a file.
            file <- NULL
        } else {
            ## Generate a reasonable filename in the current working dir
            file <- paste(name(multitable), f, sep=".")
        }
    }

    if (!is.null(weight)) {
        weight <- self(weight)
    }
    body <- list(
        filter=zcl(activeFilter(dataset)),
        weight=weight,
        options=list(...)
    )
    ## Add this after so that if it is NULL, the "where" key isn't present
    body$where <- variablesFilter(dataset)

    tabbook_url <- shojiURL(multitable, "views", "tabbook")
    ## POST the query, which (after progress polling) returns a URL to download
    result <- crPOST(tabbook_url, config=add_headers(`Accept`=accept),
        body=toJSON(body))
    if (is.null(file)) {
        ## Read in the tab book content and turn it into useful objects
        out <- retry(crGET(result))
        if (is.raw(out)) {
            ## TODO: fix the content-type header from the server
            ## See https://www.pivotaltracker.com/story/show/148554039
            out <- fromJSON(rawToChar(out), simplifyVector=FALSE)
        }
        return(TabBookResult(out))
    } else {
        file <- crDownload(result, file)
        ## (invisibly) return the filename
        invisible(file)
    }
}

#' TabBookResult and MultitableResult methods
#'
#' @param x a TabBookResult or MultitableResult
#' @param i an index into \code{x}
#' @param j an index into \code{x}, ignored
#' @param ... also ignored
#' @return Returns what you'd expect.
#' @name tabbook-methods
NULL

setMethod("initialize", "TabBookResult", function (.Object, ...) {
    .Object <- callNextMethod(.Object, ...)
    .Object$sheets <- lapply(.Object$sheets, MultitableResult)
    return(.Object)
})

#' @rdname tabbook-methods
#' @export
setMethod("length", "TabBookResult", function (x) length(x$sheets))
#' @rdname tabbook-methods
#' @export
setMethod("[[", c("TabBookResult", "numeric"), function (x, i, ...) {
    x$sheets[[i]]
})
#' @rdname tabbook-methods
#' @export
setMethod("[[", c("TabBookResult", "character"), function (x, i, ...) {
    x$sheets[[match(i, names(x))]]
})

#' @rdname tabbook-methods
#' @export
setMethod("dim", "TabBookResult", function (x) {
    nrows <- length(x)
    ncols <- ifelse(nrows, length(x[[1]]), 0L)
    return(c(nrows, ncols))
})
#' @rdname tabbook-methods
#' @export
setMethod("names", "TabBookResult", function (x) {
    unlist(lapply(x$meta$sheets, function (sheet) sheet$name))
})
#' @rdname tabbook-methods
#' @export
setMethod("aliases", "TabBookResult", function (x) {
    unlist(lapply(x, function (mt) aliases(mt[[1]])[1]), use.names=FALSE)
})
#' @rdname tabbook-methods
#' @export
setMethod("descriptions", "TabBookResult", function (x) {
    unlist(lapply(x, function (mt) descriptions(mt[[1]])[1]), use.names=FALSE)
})

setMethod("lapply", "TabBookResult", function (X, FUN, ...) {
    lapply(X$sheets, FUN, ...)
})

setMethod("initialize", "MultitableResult", function (.Object, ...) {
    .Object <- callNextMethod(.Object, ...)
    ## The first cube in the results list is the "total" column. It's a 1-D
    ## cube, add a second "dimension" so that it appears to be 2-D, n x 1
    .Object$result[[1]]$result$dimensions <- c(.Object$result[[1]]$result$dimensions,
        list(list(
            type=list(
                class="enum",
                elements=list(
                    list(id=0, value="", missing=FALSE)
                )
            ),
            references=list(
                alias="total",
                name="Total"
            )
        )))
    .Object$result <- lapply(.Object$result, function (cube) {
        cube <- CrunchCube(cube)
        ## If cubes are 3D (categorical array x multitable), aperm the cubes so
        ## that column is multitable var (3 -> 2), row is category of
        ## array (2 -> 1), subvar is "tab" (1 -> 3)
        if (length(dim(cube)) == 3L) {
            ## TODO: CubeDims should get a [ method
            ## TODO: CrunchCube should get a dimensions<-
            cube@dims <- CubeDims(dimensions(cube)[c(2, 3, 1)],
                references=dimensions(cube)@references[c(2, 3, 1)])
            cube@arrays <- lapply(cube@arrays, aperm, perm=c(2, 3, 1))
        }
        return(cube)
    })
    return(.Object)
})
#' @rdname tabbook-methods
#' @export
setMethod("length", "MultitableResult", function (x) length(x$result))
#' @rdname tabbook-methods
#' @export
setMethod("[[", "MultitableResult", function (x, i, ...) {
    x$result[[i]]
})
setMethod("lapply", "MultitableResult", function (X, FUN, ...) {
    lapply(X$result, FUN, ...)
})
#' @rdname tabbook-methods
#' @export
setMethod("names", "MultitableResult", function (x) {
    unlist(lapply(x, function (cube) names(cube)[2]), use.names=FALSE)
})
#' @rdname tabbook-methods
#' @export
setMethod("aliases", "MultitableResult", function (x) {
    unlist(lapply(x, function (cube) aliases(cube)[2]), use.names=FALSE)
})
#' @rdname tabbook-methods
#' @export
setMethod("descriptions", "MultitableResult", function (x) {
    unlist(lapply(x, function (cube) descriptions(cube)[2]), use.names=FALSE)
})

#' @rdname show-crunch
#' @export
setMethod("show", "MultitableResult", function (object) {
    show(do.call("cbind", lapply(object, cubeToArray)))
})

#' @export
as.array.TabBookResult <- function (x, ...) lapply(x, as.array)

#' @export
as.array.MultitableResult <- function (x, ...) lapply(x, as.array)

#' @rdname cube-computing
#' @export
setMethod("prop.table", "MultitableResult", function (x, margin=NULL) {
    lapply(x, prop.table, margin=margin)
})

#' @rdname cube-computing
#' @export
setMethod("prop.table", "TabBookResult", function (x, margin=NULL) {
    lapply(x, prop.table, margin=margin)
})

#' @rdname cube-computing
#' @export
setMethod("bases", "TabBookResult", function (x, margin=NULL) {
    lapply(x, bases, margin=margin)
})

#' @rdname cube-computing
#' @export
setMethod("bases", "MultitableResult", function (x, margin=NULL) {
    lapply(x, bases, margin=margin)
})
