% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm.R
\name{cpss.lm}
\alias{cpss.lm}
\title{Detecting changes in linear models}
\usage{
cpss.lm(
  formula,
  data = NULL,
  algorithm = "BS",
  dist_min = floor(log(n)),
  ncps_max = ceiling(n^0.4),
  pelt_pen_val = NULL,
  pelt_K = 0,
  wbs_nintervals = 500,
  criterion = "CV",
  times = 2
)
}
\arguments{
\item{formula}{a \code{formula} object specifying the GLM with change-points.}

\item{data}{an optional data frame containing the variables in the model.}

\item{algorithm}{a character string specifying the change-point searching algorithm, one of the following choices: "SN" (segment neighborhood), "BS" (binary segmentation), "WBS" (wild binary segmentation) and "PELT" (pruned exact linear time) algorithms.}

\item{dist_min}{an integer specifying minimum searching distance (length of feasible segments).}

\item{ncps_max}{an integer specifying an upper bound of the number of true change-points.}

\item{pelt_pen_val}{a numeric vector specifying candidate values of the penalty only if \code{algorithm = "PELT"}.}

\item{pelt_K}{a numeric value for pruning adjustment only if \code{algorithm = "PELT"}. It is usually taken to be 0 if the negative log-likelihood is used as a cost, see Killick et al. (2012).}

\item{wbs_nintervals}{an integer specifying the number of random intervals drawn only if \code{algorithm = "WBS"}, see Fryzlewicz (2014).}

\item{criterion}{a character string specifying the model selection criterion, "CV" ("cross-validation") or "MS" ("multiple-splitting").}

\item{times}{an integer specifying how many times of sample-splitting should be performed; It should be 2 if \code{criterion = "CV"}.}
}
\value{
\code{cpss.lm} returns an object of an \proglang{S4} class, called "\code{cpss}", which collects data and information required for further change-point analyses and summaries. See \code{\link{cpss.custom}}.
}
\description{
Detecting changes in linear models
}
\examples{
library("cpss")
set.seed(666)
n <- 400
tau <- c(80, 200, 300)
tau_ext <- c(0, tau, n)
be <- list(c(0, 1), c(1, 0.5), c(0, 1), c(-1, 0.5))
seg_len <- diff(c(0, tau, n))
x <- rnorm(n)
mu <- lapply(seq(1, length(tau) + 1), function(k) {
  be[[k]][1] + be[[k]][2] * x[(tau_ext[k] + 1):tau_ext[k + 1]]
})
mu <- do.call(c, mu)
sig <- unlist(lapply(seq(1, length(tau) + 1), function(k) {
  rep(be[[k]][2], seg_len[k])
}))
y <- rnorm(n, mu, sig)
res <- cpss.lm(
  formula = y ~ x,
  algorithm = "BS", ncps_max = 10
)
summary(res)
# 80  202  291
coef(res)
# $coef
#             [,1]      [,2]        [,3]       [,4]
# [1,] -0.00188792 1.0457718 -0.03963209 -0.9444813
# [2,]  0.91061557 0.6291965  1.20694409  0.4410036
#
# $sigma
# [1] 0.8732233 0.4753216 0.9566516 0.4782329
}
\references{
Killick, R., Fearnhead, P., and Eckley, I. A. (2012). Optimal Detection of Changepoints With a Linear Computational Cost. Journal of the American Statistical Association, 107(500):1590–1598.

Fryzlewicz, P. (2014). Wild binary segmentation for multiple change-point detection. The Annals of Statistics, 42(6): 2243–2281.
}
\seealso{
\code{\link{cpss.glm}}
}
