% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/conleyreg.R
\name{conleyreg}
\alias{conleyreg}
\title{Conley standard error estimations}
\usage{
conleyreg(
  formula,
  data,
  dist_cutoff,
  model = c("ols", "logit", "probit"),
  unit = NULL,
  time = NULL,
  lat = NULL,
  lon = NULL,
  kernel = c("bartlett", "uniform"),
  lag_cutoff = 0,
  intercept = TRUE,
  verbose = TRUE,
  ncores = NULL,
  dist_comp = c("precise", "fast"),
  sparse = FALSE,
  batch = FALSE
)
}
\arguments{
\item{formula}{regression equation as formula or character string}

\item{data}{input data, either in non-spatial data frame format (includes tibbles and data tables) with columns denoting coordinates or in sf format with a spatial
points geometry. When using a non-spatial data frame format, the coordinates must be longlat. sf objects can use any projection. Note that the projection can influence
the computed distances, which is a general phenomenon in GIS software and not specific to \code{conleyreg}.}

\item{dist_cutoff}{the distance cutoff in km}

\item{model}{the applied model. Either \code{ols} (default), \code{logit}, or \code{probit}. \code{logit} and \code{probit} are currently restricted to
cross-sectional applications.}

\item{unit}{the variable identifying the cross-sectional dimension. Only needs to be specified, if data is not cross-sectional. Assumes that units do not change their
location over time.}

\item{time}{the variable identifying the time dimension}

\item{lat}{the variable specifying the latitude in longlat format}

\item{lon}{the variable specifying the longitude in longlat format}

\item{kernel}{the kernel applied within the radius. Either \code{bartlett} (default) or \code{uniform}.}

\item{lag_cutoff}{the cutoff along the time dimension. Defaults to 0, meaning that standard errors are only adjusted cross-sectionally.}

\item{intercept}{boolean specifying whether to include an intercept. Defaults to \code{TRUE}. Fixed effects models omit the intercept automatically.}

\item{verbose}{boolean specifying whether to print messages on intermediate estimation steps. Defaults to \code{TRUE}.}

\item{ncores}{the number of CPU cores to use in the estimations. Defaults to the machine's number of CPUs. Does not affect cross-sectional applications.}

\item{dist_comp}{choice between \code{precise} (default) and \code{fast} distance computations when data is longlat. Even when choosing \code{precise}, you can still
tweak the performance by setting the library that the \code{sf} package uses in distance computations. \code{sf::sf_use_s2(TRUE)} makes it rely on s2 which should be
faster than the alternative choice of GEOS with \code{sf::sf_use_s2(FALSE)}. With \code{precise}, distances are great circle distances, with \code{fast} they are
haversine distances. Non-longlat data is not affected by this parameter and always uses Euclidean distances.}

\item{sparse}{boolean specifying whether to use sparse rather than dense (regular) matrices in distance computations. Defaults to \code{FALSE}. Only has an effect when
\code{dist_comp = "fast"}. Sparse matrices are more efficient than dense matrices, when the distance matrix has a lot of zeros arising from points located outside the
respective \code{dist_cutoff}. It is recommended to keep the default unless the machine is unable to allocate enough memory.}

\item{batch}{boolean specifying whether distances are inserted into a sparse matrix element by element (\code{FALSE}) or all at once as a batch (\code{TRUE}). Defaults
to \code{FALSE}. This argument only has an effect when \code{dist_comp = "fast"} and \code{sparse = TRUE}. Batch insertion is faster than element-wise insertion, but
requires more memory.}
}
\value{
Returns a \code{lmtest::coeftest} matrix of coefficient estimates and standard errors. \insertNoCite{*}{conleyreg}
}
\description{
This function estimates ols, logit, and probit models with Conley standard errors.
}
\details{
This code is an extension and modification of earlier Conley standard error implementations by (i) Richard Bluhm, (ii) Luis Calderon and Leander Heldring,
(iii) Darin Christensen and Thiemo Fetzer, and (iv) Timothy Conley. Results vary across implementations because of different distance functions and buffer shapes.
}
\examples{
# Generate cross-sectional example data
data <- data.frame(y = sample(c(0, 1), 100, replace = TRUE),
  x1 = stats::runif(100, -50, 50),
  lat = runif(100, -90, 90),
  lon = runif(100, -180, 180))

# Estimate ols model with Conley standard errors using a 1000 km radius
conleyreg(y ~ x1, data, 1000, lat = "lat", lon = "lon")

# Estimate same model with an sf object as input
conleyreg(y ~ x1, sf::st_as_sf(data, coords = c("lon", "lat"), crs = 4326), 1000)

# Estimate same model with an sf object of another projection as input
conleyreg(y ~ x1, sf::st_transform(sf::st_as_sf(data, coords = c("lon", "lat"), crs = 4326),
  crs = "+proj=aeqd"), 1000)

# Estimate logit model
conleyreg(y ~ x1, data, 1000, "logit", lat = "lat", lon = "lon")

# Add variable
data$x2 <- sample(1:5, 100, replace = TRUE)

# Estimate ols model with fixed effects
conleyreg(y ~ x1 | x2, data, 1000, lat = "lat", lon = "lon")

# Estimate probit model with fixed effects
conleyreg(y ~ x1 | x2, data, 1000, "probit", lat = "lat", lon = "lon")

# Add panel variables
data$time <- rep(1:10, each = 10)
data$unit <- rep(1:10, times = 10)
\donttest{
# Estimate ols model using panel data
conleyreg(y ~ x1, data, 1000, unit = "unit", time = "time", lat = "lat", lon = "lon")
}

}
\references{
\insertAllCited{}
}
