#' Generate Seasons
#' 
#' Identify the hours, days or months in a (monthly, daily or hourly) temperature dataset that belong to a particular season.
#' Seasons are defined according to the `mrange` argument, which specifies the start and end month of
#' the season. The `years` argument specifies the year, in which the dormancy season of interest ends.
#' 
#' @param temps list. generated by `chillR`
#' @param mrange numeric. vector with two entries for the range of months (start month and end month)
#' @param years numeric. vector of years to be considered (with each entry specifying the year, in which the season **ends**)
#' @export
genSeason <- function(temps, mrange=c(8,6), years) {
  stopifnot(length(mrange) == 2)
  N <- length(years)
  stopifnot(N >= 1)
  Season <- list()
  if(inherits(temps, "list"))  temperatures<-temps[[1]] else temperatures<-temps
  for(j in c(1:N)) {
    ## season starts in November of the preceding year
    ## the winter season from August to June
    ## we add the Year as the last element to ship it to sapply
    Season[[j]] <- c(which(((temperatures$Month %in% c(mrange[1]:12)) &
                            (temperatures$Year==years[j]-1))),
                     which(((temperatures$Month %in% c(1:mrange[2])) &
                            (temperatures$Year==years[j]))))
  }
  return(invisible(Season))
}

#' genSeasonList
#' 
#' Generates a list with data.frame elements for each season. 
#'
#' @param temps data.frame. Must have columns `Temp` containing the temperatures, `JDay`
#' the JDays, `Month` the months and `Year` the years. This
#' kind of data frame is for instance generated by \link{stack_hourly_temps}, but can also
#' be generated by hand or using a different routine.
#' @param mrange numeric. vector of length two for the range of months the season
#' should span. E.g. `mrange=c(8,6)` would span a season from August to next June.
#' There must not be any overlap in months, i.e. mrange[1] must be larger mrange[2].
#' @param years numeric. vector of years to be considered
#'
#' @return
#' Returns a list of data frames. Each element of the list corresponds to one season. The
#' `data.frame` for each year has named columns `Temp`, `JDay` and `Year`. 
#' @export
genSeasonList  <- function(temps, mrange=c(8,6), years) {
  stopifnot(length(mrange) == 2)
  stopifnot(!missing(years))
  stopifnot(!missing(temps))
  stopifnot(mrange[1] > mrange[2])
  N <- length(years)
  stopifnot(N >= 1)
  SeasonList <- list()
  for(j in c(1:N)) {
    ii <- c(which(((temps$Month %in% c(mrange[1]:12)) &
                   (temps$Year==years[j]-1))),
            which(((temps$Month %in% c(1:mrange[2])) &
                   (temps$Year==years[j]))))
    SeasonList[[j]] <- data.frame(Temp=temps$Temp[ii],
                                  JDay=temps$JDay[ii],
                                  Year=temps$Year[ii])
  }
  return(invisible(SeasonList))
}
