% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adjust_loglik.R
\name{adjust_loglik}
\alias{adjust_loglik}
\title{Loglikelihood adjustment using the sandwich estimator}
\usage{
adjust_loglik(loglik = NULL, ..., cluster = NULL, p = NULL,
  init = NULL, par_names = NULL, fixed_pars = NULL, fixed_at = 0,
  name = NULL, larger = NULL, alg_deriv = NULL, alg_hess = NULL,
  mle = NULL, H = NULL, V = NULL)
}
\arguments{
\item{loglik}{A named function.  Returns a vector of the
loglikelihood contributions of individual observations.  The first
argument must be the vector of model parameter(s). If any of the model
parameters are out-of-bounds then \code{loglik} should return either
\code{-Inf} or a vector with at least one element equal to \code{-Inf}.
The number of parameters in the \strong{full} model must be specified
using (at least) one of the arguments \code{p}, \code{init} or
\code{par_names}.}

\item{...}{Further arguments to be passed either to \code{loglik}
(and to \code{alg_deriv} and \code{alg_hess} if these are supplied) or
to \code{\link[stats]{optim}}.  The latter may include \code{gr},
\code{method}, \code{lower}, \code{upper} or \code{control}.
In the call to \code{\link[stats]{optim}}, \code{hessian = TRUE}
will be used regardless of any value supplied.
The function \code{loglik} must \emph{not} have arguments with names
that match any of these arguments to \code{\link[stats]{optim}}.}

\item{cluster}{A vector or factor indicating from which cluster the
respective loglikelihood contributions from \code{loglik} originate.
Must have the same length as the vector returned by \code{loglik}.
If \code{cluster} is not supplied then it is set inside
\code{adjust_loglik} under the assumption that each observation forms
its own cluster.}

\item{p}{A numeric scalar.  The dimension of the \strong{full} parameter
vector, i.e. the number of parameters in the full model.  Must be
consistent with the lengths of \code{init} and \code{par_names},
if these are also supplied.}

\item{init}{A numeric vector of initial values.  Must have length equal
to the number of parameters in the \strong{full} model.  If \code{init}
is supplied then \code{p} is set to \code{length(init)}, provided that
this is consistent with the the value given by \code{p} or implied
by \code{length(par_names)}.
If \code{fixed_pars} is not \code{NULL} then \code{init[-fixed_pars]}
is used in the search for the MLE.
If \code{init} is not supplied then \code{rep(0.1, p)} is used.}

\item{par_names}{A character vector.  Names of the \code{p} parameters
in the \strong{full} model.  Must be consistent with the lengths of
\code{init} and \code{p}, if these are also supplied.}

\item{fixed_pars}{A vector specifying which parameters are to be restricted
to be equal to the value(s) in \code{fixed_at}.  Can be either a numeric
vector, specifying indices of the components of the \strong{full} parameter
vector, or a character vector of parameter names, which must be a subset
of those supplied in \code{par_names} or stored in the object
\code{larger}.}

\item{fixed_at}{A numeric vector of length 1 or \code{length(fixed_pars)}.
If \code{length(fixed_at) = 1} then the components \code{fixed_pars}
of the parameter vector are all fixed at \code{fixed_at}.
If \code{length(fixed_at) = length(fixed_pars)} then the component
\code{fixed_pars[i]} is fixed at \code{fixed_at[i]} for each \code{i}.}

\item{name}{A character scalar.  A name for the model that gives rise
to \code{loglik}.  If this is not supplied then the name in
\code{larger} is used, if this has been supplied, and the name of
the function \code{loglik} otherwise.}

\item{larger}{Only relevant if \code{fixed_pars} is not \code{NULL}.
  If \code{larger} is supplied but \code{fixed_pars} is not then an error
  will result.  if \code{larger} is supplied then information about the
  model in \code{larger}, e.g. about \code{p} and \code{par_names} will
  override any attempt to set these arguments in the call to
  \code{adjust_loglik}.

  An object of class \code{"chandwich"} returned by \code{adjust_loglik},
  corresponding to a model in which the smaller model implied by
  \code{fixed_pars} is nested.  If \code{larger} is supplied then
  all the arguments to \code{adjust_loglik} apart from
  \code{fixed_pars} and \code{fixed_at} are extracted from \code{larger}.
  If \code{init} is not supplied in the current call to
  \code{adjust_loglik} then \code{init} is set to
  \code{attr(larger, "MLE")}, with the elements in \code{fixed_pars}
  set to \code{fixed_at}.}

\item{alg_deriv}{A function with the vector of model parameter(s) as its
first argument.  Returns a \code{length(cluster)} by \code{p} numeric
matrix. Column i contains the derivatives of each of the loglikelihood
contributions in \code{loglik} with respect to model parameter i.}

\item{alg_hess}{A function with the vector of model parameter(s) as its
  first argument.  Returns a \code{p} by \code{p} numeric matrix equal to
  the Hessian of \code{loglik}, i.e. the matrix of second derivatives of
  the function \code{loglik}.

  Supplying both \code{V} and \code{alg_deriv} or both \code{H} and
  \code{alg_hess} will produce an error.}

\item{mle}{A numeric vector.  Can only be used if \code{fixed_pars = NULL}.
Provides the maximum likelihood estimate of the model parameters,
that is, the value of the parameter vector
at which the independence loglikelihood \code{loglik} is maximized.
Must have length equal to the number of parameters in the
\strong{full} model.  If \code{mle} is supplied then \code{p} is set
to \code{length(mle)}, provided that this is consistent with the the
value given by \code{p} or implied by \code{length(par_names)}.
If \code{mle} is supplied then it overrides \code{init}.}

\item{H, V}{p by p numeric matrices.  Only used if \code{mle} is supplied.
  Provide estimates of the Hessian of the
  independence loglikelihood (H) and the variance of the vector
  of cluster-specific contributions to the score vector (first
  derivatives with respect to the parameters) of the independence
  loglikelihood, each evaluated at the MLE \code{mle}.  See the
  \emph{Introducing chandwich} vignette and/or Chandler and Bate (2007).

  Supplying both \code{V} and \code{alg_deriv} or both \code{H} and
  \code{alg_hess} will produce an error.}
}
\value{
A function of class \code{"chandwich"} to evaluate an adjusted
  loglikelihood, or the independence loglikelihood, at one or more sets
  of model parameters, with arguments
  \item{x}{A numeric vector or matrix giving values of the \code{p_current}
    (see below) parameters in the model to which the returned adjusted
    loglikelihood applies.
    If \code{p_current = 1} this may be a numeric vector or a matrix
    with 1 column.
    If \code{p_current > 1} this may be a numeric vector of length \code{p}
    (one set of model parameters) or a numeric matrix with \code{p}
    columns (\code{nrow(x)} sets of model parameters), one set in each row
    of \code{x}.}
  \item{type}{A character scalar.  The type of adjustment to use.
    One of \code{"vertical"}, \code{"cholesky"}, \code{"spectral"} or
    \code{"none"}.}  The latter results in the evaluation of the
    (unadjusted) independence loglikelihood.
  The function has (additional) attributes
  \item{p_full, p_current}{The number of parameters in the full model and
    current models, respectively.}
  \item{free_pars}{A numeric vector giving the indices of the free
    parameters in the current model, with names inferred from
    \code{par_names} if this was supplied.}
  \item{MLE, res_MLE}{Numeric vectors, with names inferred from
    \code{par_names} if this was supplied.  Maximum likelihood estimates
    of free parameters under the current model (\code{mle}) and all
    parameters in the full model, including any parameters with fixed
    values (\code{res_MLE}).}
  \item{SE, adjSE}{The unadjusted and adjusted estimated standard errors,
    of the free parameters, respectively.}
  \item{VC, adjVC}{The unadjusted and adjusted estimated
    variance-covariance matrix of the free parameters, respectively.}
  \item{HI, HA}{The Hessians of the independence and adjusted loglikelihood,
    respectively.}
  \item{C_cholesky, C_spectral}{The matrix C in equation (14) of Chandler and
    Bate (2007), calculated using Cholesky decomposition and spectral
    decomposition, respectively.}
  \item{full_par_names, par_names}{The names of the parameters in the full
    and current models, respectively, if these were supplied in
    this call or a previous call.}
  \item{max_loglik}{The common maximised value of the independence and
    adjusted loglikelihoods.}
  \item{loglik, cluster}{The arguments \code{loglik} and \code{cluster}
    supplied in this call, or a previous call.}
  \item{loglik_args}{A list containing the further arguments passed to
    \code{loglik} via ... in this call, or a previous call.}
  \item{name}{The argument \code{name}, or the name of the function
    \code{loglik} if \code{name} isn't supplied.}
  If \code{fixed_pars} is not \code{NULL} then there are further attributes
  \item{fixed_pars}{The argument \code{fixed_pars}, with names inferred from
    \code{par_names} if this was supplied.}
  \item{fixed_at}{The argument \code{fixed_at}, with names inferred from
    \code{par_names} if this was supplied.}
  If \code{alg_deriv} and/or \code{alg_hess} were supplied then these are
  returned as further attributes.

  To view an individual attribute use \code{attr(x, "name")}
  or \code{attributes(x)$name}.
}
\description{
Performs adjustments of a user-supplied independence loglikelihood for the
presence of cluster dependence, following
\href{http://dx.doi.org/10.1093/biomet/asm015}{Chandler and Bate (2007)}.
The user provides a function that returns a vector of observation-specific
loglikelihood contributions and a vector that indicates cluster membership.
The loglikelihood of a sub-model can be adjusted by fixing a set of
parameters at particular values.
}
\details{
Three adjustments to the independence loglikelihood described in
  Chandler and Bate (2007) are available.  The vertical adjustment is
  described in Section 6 and two horizontal adjustments are described
  in Sections 3.2 to 3.4.  See the descriptions of \code{type} and, for the
  horizontal adjustments, the descriptions of \code{C_cholesky} and
  \code{C_spectral}, in \strong{Value}.

  The adjustments involve first and second derivatives of the loglikelihood
  with respect to the model parameters.  These are estimated using
  \code{\link[numDeriv]{jacobian}} and \code{\link[stats:optim]{optimHess}}
  unless \code{alg_deriv} and/or \code{alg_hess} are supplied.
}
\examples{
# ------------------------- Binomial model, rats data ----------------------

# Contributions to the independence loglikelihood
binom_loglik <- function(prob, data) {
  if (prob < 0 || prob > 1) {
    return(-Inf)
  }
  return(dbinom(data[, "y"], data[, "n"], prob, log = TRUE))
}
rat_res <- adjust_loglik(loglik = binom_loglik, data = rats, par_names = "p")

# Plot the loglikelihoods
plot(rat_res, type = 1:4, legend_pos = "bottom", lwd = 2, col = 1:4)
# MLE, SEs and adjusted SEs
summary(rat_res)

# -------------------------- GEV model, owtemps data -----------------------
# ------------ following Section 5.2 of Chandler and Bate (2007) -----------

# Contributions to the independence loglikelihood
gev_loglik <- function(pars, data) {
  o_pars <- pars[c(1, 3, 5)] + pars[c(2, 4, 6)]
  w_pars <- pars[c(1, 3, 5)] - pars[c(2, 4, 6)]
  if (o_pars[2] <= 0 | w_pars[2] <= 0) return(-Inf)
  o_data <- data[, "Oxford"]
  w_data <- data[, "Worthing"]
  check <- 1 + o_pars[3] * (o_data - o_pars[1]) / o_pars[2]
  if (any(check <= 0)) return(-Inf)
  check <- 1 + w_pars[3] * (w_data - w_pars[1]) / w_pars[2]
  if (any(check <= 0)) return(-Inf)
  o_loglik <- log_gev(o_data, o_pars[1], o_pars[2], o_pars[3])
  w_loglik <- log_gev(w_data, w_pars[1], w_pars[2], w_pars[3])
  return(o_loglik + w_loglik)
}

# Initial estimates (method of moments for the Gumbel case)
sigma <- as.numeric(sqrt(6 * diag(var(owtemps))) / pi)
mu <- as.numeric(colMeans(owtemps) - 0.57722 * sigma)
init <- c(mean(mu), -diff(mu) / 2, mean(sigma), -diff(sigma) / 2, 0, 0)

# Loglikelihood adjustment for the full model
par_names <- c("mu[0]", "mu[1]", "sigma[0]", "sigma[1]", "xi[0]", "xi[1]")
large <- adjust_loglik(gev_loglik, data = owtemps, init = init,
                       par_names = par_names)
# Rows 1, 3 and 4 of Table 2 of Chandler and Bate (2007)
t(summary(large))

# Loglikelihood adjustment of some smaller models: xi[1] = 0 etc

# Starting from a larger model
medium <- adjust_loglik(larger = large, fixed_pars = "xi[1]")
small <- adjust_loglik(larger = large, fixed_pars = c("sigma[1]", "xi[1]"))
small <- adjust_loglik(larger = medium, fixed_pars = c("sigma[1]", "xi[1]"))

# Starting from scratch
medium <- adjust_loglik(gev_loglik, data = owtemps, init = init,
          par_names = par_names, fixed_pars = "xi[1]")
small <- adjust_loglik(gev_loglik, data = owtemps, init = init,
         par_names = par_names, fixed_pars = c("sigma[1]", "xi[1]"))

# --------- Misspecified Poisson model for negative binomial data ----------

# ... following Section 5.1 of the "Object-Oriented Computation of Sandwich
# Estimators" vignette of the sandwich package
# https://cran.r-project.org/web/packages/sandwich/vignettes/sandwich-OOP.pdf

# Simulate data
set.seed(123)
x <- rnorm(250)
y <- rnbinom(250, mu = exp(1 + x), size = 1)
# Fit misspecified Poisson model
fm_pois <- glm(y ~ x + I(x^2), family = poisson)
summary(fm_pois)$coefficients

# Contributions to the independence loglikelihood
pois_glm_loglik <- function(pars, y, x) {
  log_mu <- pars[1] + pars[2] * x + pars[3] * x ^ 2
  return(dpois(y, lambda = exp(log_mu), log = TRUE))
}
pars <- c("alpha", "beta", "gamma")
pois_quad <- adjust_loglik(pois_glm_loglik, y = y, x = x, par_names = pars)
summary(pois_quad)

# Providing algebraic derivatives and Hessian
pois_alg_deriv <- function(pars, y, x) {
  mu <- exp(pars[1] + pars[2] * x + pars[3] * x ^ 2)
  return(cbind(y - mu, x * (y - mu), x ^2 * (y - mu)))
}
pois_alg_hess <- function(pars, y, x) {
  mu <- exp(pars[1] + pars[2] * x + pars[3] * x ^ 2)
  alg_hess <- matrix(0, 3, 3)
  alg_hess[1, ] <- -c(sum(mu), sum(x * mu), sum(x ^ 2 * mu))
  alg_hess[2, ] <- -c(sum(x * mu), sum(x ^ 2 * mu), sum(x ^ 3 * mu))
  alg_hess[3, ] <- -c(sum(x ^ 2 * mu), sum(x ^ 3 * mu), sum(x ^ 4 * mu))
  return(alg_hess)
}
pois_quad <- adjust_loglik(pois_glm_loglik, y = y, x = x, p = 3,
                           alg_deriv = pois_alg_deriv, alg_hess = pois_alg_hess)
summary(pois_quad)

got_sandwich <- requireNamespace("sandwich", quietly = TRUE)
if (got_sandwich) {
  # Providing MLE, H and V
  # H and V calculated using bread() and meat() from sandwich package
  n_obs <- stats::nobs(fm_pois)
  pois_quad <- adjust_loglik(pois_glm_loglik, y = y, x = x, p = 3,
                             mle = fm_pois$coefficients,
                             H = -solve(sandwich::bread(fm_pois) / n_obs),
                             V = sandwich::meat(fm_pois) * n_obs)
}
}
\references{
Chandler, R. E. and Bate, S. (2007). Inference for clustered
  data using the independence loglikelihood. \emph{Biometrika},
  \strong{94}(1), 167-183. \url{http://dx.doi.org/10.1093/biomet/asm015}
}
\seealso{
\code{\link{summary.chandwich}} for maximum likelihood estimates
  and unadjusted and adjusted standard errors.

\code{\link{plot.chandwich}} for plots of one-dimensional adjusted
  loglikelihoods.

\code{\link{conf_intervals}} for confidence intervals for
  individual parameters.

\code{\link{conf_region}} for a confidence region for
  a pair of parameters.

\code{\link{compare_models}} to compare nested models using an
  (adjusted) likelihood ratio test.
}
