\name{model_adequacy_hbd}
\alias{model_adequacy_hbd}
\title{
Check if a birth-death model adequately explains a timetree.
}
\description{
Given a rooted ultrametric timetree and a homogenous birth-death model, check if the model adequately explains various aspects of the tree, such as the branch length and node age distributions and other test statistics. The function uses bootstrapping to simulate multiple hypothetical trees according to the model and then compares the distribution of those trees to the original tree. This function may be used to quantify the "goodness of fit" of a birth-death model to a timetree.
}
\usage{
model_adequacy_hbd( tree,
                    models,
                    splines_degree  = 1,
                    extrapolate     = FALSE,
                    Nbootstraps     = 1000,
                    Nthreads        = 1)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
\item{tree}{
A rooted ultrametric timetree of class "phylo".
}
\item{models}{
\itemize{Either a single HBD model or a list of HBD models, specifying the pool of models from which to randomly draw bootstraps. Every model should itself be a named list with some or all of the following elements: 
\item \code{ages}: Numeric vector, specifying discrete ages (times before present) in ascending order, on which the pulled speciation rate will be specified. Age increases from tips to root; the youngest tip in the input tree has age 0. The age grid must cover the present-day (age 0) and the root
\item \code{PSR}: Numeric vector of size NG, listing the pulled speciation rate (PSR) of the HBD model at the corresponding \code{ages}. Between grid points, the PSR is assumed to either be constant (if \code{splines_degree=0}), or linearly (if \code{splines_degree=1}) or quadratically (if \code{splines_degree=2}) or cubically (if \code{splines_degree=3}). To calculate the PSR of an HBD model based on the speciation and extinction rate, see \code{\link{simulate_deterministic_hbd}}.
}
}
\item{splines_degree}{
Integer, one of 0, 1, 2 or 3, specifying the polynomial degree of the PSR between age-grid points. For example, \code{splines_degree=0} means piecewise constant, \code{splines_degree=1} means piecewise linear and so on.
}
\item{extrapolate}{
Logical, specifying whether to extrapolate the model variables \eqn{\lambda}, \eqn{\mu}, \eqn{\psi} and \eqn{\kappa} (as constants) beyond the provided age grid all the way to \code{stem_age} and \code{end_age} if needed.
}
\item{Nbootstraps}{
Integer, the number of bootstraps (simulations) to perform for calculating statistical significances. A larger number will increase the accuracy of estimated statistical significances.
}
\item{Nthreads}{
Integer, number of parallel threads to use for bootstrapping. Note that on Windows machines this option is ignored.
}
}


\details{
In addition to model selection, the adequacy of any chosen model should also be assessed in absolute terms, i.e. not just relative to other competing models (after all, all considered models might be bad). This function essentially determines how probable it is for hypothetical trees generated by a candidate model to resemble the tree at hand, in terms of various test statistics (such as the historically popular "gamma" statistic, or the Colless tree imbalance). In particular, the function uses a Kolmogorov-Smirnov test to check whether the probability distributions of edge lengths and node ages in the tree resemble those expected under the model. All statistical significances are calculated using bootstrapping, i.e. by simulating trees from the provided model with the same number of tips and the same root age as the original tree.

Note that even if an HBD model appears to adequately explain a given timetree, this does not mean that the model even approximately resembles the true diversification history (i.e., the true speciation and extinction rates) that generated the tree (Louca and Pennell 2020). Hence, it is generally more appropriate to say that a given model "congruence class" (or PSR) rather than a specific model (speciation rate, extinction rate, and sampling fraction) explains the tree.

This function requires that the HBD model (or more precisely, its congruence class) be defined in terms of the PSR. If your model is defined in terms of speciation/extinction rates and a sampling fraction, or if your model's congruence class is defined in terms of the pulled diversification rate (PDR) and the product \eqn{\rho\lambda_o}, then you can use \code{\link{simulate_deterministic_hbd}} to first calculate the corresponding PSR.
}

\value{
A named list with the following elements:
\item{success}{
Logical, indicating whether the model evaluation was successful. If \code{FALSE}, then an additional return variable, \code{error}, will contain a description of the error; in that case all other return variables may be undefined. Note that \code{success} does not say whether the model explains the tree, but rather whether the computation was performed without errors.
}
\item{Nbootstraps}{
Integer, the number of bootstraps used.
}
\item{tree_gamma}{
Numeric, gamma statistic (Pybus and Harvey 2000) of the original tree.
}
\item{bootstrap_mean_gamma}{
Numeric, mean gamma statistic across all bootstrap trees.
}
\item{Pgamma}{
Numeric, two-sided statistical significance of the tree's gamma statistic under the provided null model, i.e. the probability that \code{abs(bootstrap_mean_gamma-tree_gamma)} would be as large as observed.
}
\item{tree_Colless}{
Numeric, Colless imbalance statistic (Shao and Sokal, 1990) of the original tree.
}
\item{bootstrap_mean_Colless}{
Numeric, mean Colless statistic across all bootstrap trees.
}
\item{PColless}{
Numeric, two-sided statistical significance of the tree's Colless statistic under the provided null model, i.e. the probability that \code{abs(bootstrap_mean_Colless-tree_Colless)} would be as large as observed.
}
\item{tree_Sackin}{
Numeric, Sackin statistic (Sackin, 1972) of the original tree.
}
\item{bootstrap_mean_Sackin}{
Numeric, median Sackin statistic across all bootstrap trees.
}
\item{PSackin}{
Numeric, two-sided statistical significance of the tree's Sackin statistic under the provided null model, i.e. the probability that \code{abs(bootstrap_mean_Sackin-tree_Sackin)} would be as large as observed.
}
\item{tree_Blum}{
Numeric, Blum statistic (Blum and Francois, 2006) of the original tree.
}
\item{bootstrap_mean_Blum}{
Numeric, median Blum statistic across all bootstrap trees.
}
\item{PBlum}{
Numeric, two-sided statistical significance of the tree's Blum statistic under the provided null model, i.e. the probability that \code{abs(bootstrap_mean_Blum-tree_Blum)} would be as large as observed.
}
\item{tree_edgeKS}{
Numeric, Kolmogorov-Smirnov (KS) statistic of the original tree's edge lengths, i.e. the estimated maximum difference between the tree's and the model's (estimated) cumulative distribution function of edge lengths.
}
\item{bootstrap_mean_edgeKS}{
Numeric, mean KS statistic of the bootstrap trees' edge lengths.
}
\item{PedgeKS}{
Numeric, the one-sided statistical significance of the tree's edge-length KS statistic, i.e. the probability that the KS statistic of any tree generated by the model would be larger than the original tree's KS statistic. A low value means that the probability distribution of edge lengths in the original tree differs strongly from that expected based on the model.
}
\item{tree_nodeKS}{
Numeric, Kolmogorov-Smirnov (KS) statistic of the original tree's node ages (divergence times), i.e. the estimated maximum difference between the tree's and the model's (estimated) cumulative distribution function of node ages.
}
\item{bootstrap_mean_nodeKS}{
Numeric, mean KS statistic of the bootstrap trees' node ages.
}
\item{PnodeKS}{
Numeric, the one-sided statistical significance of the tree's node-age KS statistic, i.e. the probability that the KS statistic of any tree generated by the model would be larger than the original tree's KS statistic. A low value means that the probability distribution of node ages in the original tree differs strongly from that expected based on the model.
}
\item{tree_sizeKS}{
Numeric, Kolmogorov-Smirnov (KS) statistic of the original tree's node sizes (number of descending tips per node), i.e. the estimated maximum difference between the tree's and the model's (estimated) cumulative distribution function of node sizes.
}
\item{bootstrap_mean_sizeKS}{
Numeric, mean KS statistic of the bootstrap trees' node sizes.
}
\item{PsizeKS}{
Numeric, the one-sided statistical significance of the tree's node-size KS statistic, i.e. the probability that the KS statistic of any tree generated by the model would be larger than the original tree's KS statistic. A low value means that the probability distribution of node sizes in the original tree differs strongly from that expected based on the model.
}
\item{statistical_tests}{
Data frame, listing the above statistical test results in a more compact format (one test statistic per row).
}
\item{LTT_ages}{
Numeric vector, listing ages (time before present) on which the tree's LTT will be defined.
}
\item{tree_LTT}{
Numeric vector of the same length as \code{LTT_ages}, listing the number of lineages in the tree at every age in \code{LTT_ages}.
}
\item{bootstrap_LTT_CI}{
Named list containing the elements \code{means}, \code{medians}, \code{CI50lower}, \code{CI50upper}, \code{CI95lower} and \code{CI95upper}. Each of these elements is a numeric vector of length equal to \code{LTT_ages}, listing the mean or a specific percentile of LTTs of bootstrap trees at every age in \code{LTT_ages}. For example, \code{bootstrap_LTT_CI$CI95lower[10]} and \code{bootstrap_LTT_CI$CI95upper[10]} define the lower and upper bound, respectively, of the 95\% confidence interval of LTTs generated by the model at age \code{LTT_ages[10]}.
}
\item{fraction_LTT_in_CI95}{
Numeric, fraction of the tree's LTT contained within the equal-tailed 95\%-confidence interval of the distribution of LTT values predicted by the model. For example, a value of 0.5 means that at half of the time points between the present-day and the root, the tree's LTT is contained with the 95\%-CI of predicted LTTs.
}
}


\author{Stilianos Louca}

\references{
S. Louca and M. W. Pennell (2020). Extant timetrees are consistent with a myriad of diversification histories. Nature. 580:502-505.

O. G. Pybus and P. H. Harvey (2000). Testing macro-evolutionary models using incomplete molecular phylogenies. Proceedings of the Royal Society of London. Series B: Biological Sciences. 267:2267-2272.

M. J. Sackin (1972). "Good" and "Bad" Phenograms. Systematic Biology. 21:225-226.

K.T. Shao, R. R. Sokal (1990). Tree Balance. Systematic Biology. 39:266-276.

M. G. B. Blum and O. Francois (2006). Which random processes describe the Tree of Life? A large-scale study of phylogenetic tree imbalance. Systematic Biology. 55:685-691.
}

\seealso{
\code{\link{simulate_deterministic_hbd}},
\code{\link{model_adequacy_hbds}}
}

\examples{
# generate a tree
tree = castor::generate_tree_hbd_reverse(Ntips  = 50,
                                         lambda = 1,
                                         mu     = 0.5,
                                         rho    = 1)$trees[[1]]
root_age = castor::get_tree_span(tree)$max_distance

# define & simulate a somewhat different BD model
model = simulate_deterministic_hbd(LTT0         = 50,
                                   oldest_age   = root_age,
                                   lambda       = 1.5,
                                   mu           = 0.5,
                                   rho0         = 1)

# compare the tree to the model
adequacy = model_adequacy_hbd(tree, 
                              models        = model,
                              Nbootstraps   = 100,
                              Nthreads      = 2)
if(!adequacy$success){
    cat(sprintf("Adequacy test failed: \%s\n",adequacy$error))
}else{
    print(adequacy$statistical_tests)
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ Pruning }
