% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/csem_model.R
\name{parseModel}
\alias{parseModel}
\title{Parse lavaan model}
\usage{
parseModel(
  .model        = NULL, 
  .instruments  = NULL, 
  .check_errors = TRUE
  )
}
\arguments{
\item{.model}{A model in \link[lavaan:model.syntax]{lavaan model syntax}
or a \link{cSEMModel} list.}

\item{.instruments}{A named list of vectors of instruments. The names
of the list elements are the names of the dependent (LHS) constructs of the structural
equation whose explanatory variables are endogenous. The vectors
contain the names of the instruments corresponding to each equation. Note
that exogenous variables of a given equation \strong{must} be supplied as
instruments for themselves. Defaults to \code{NULL}.}

\item{.check_errors}{Logical. Should the model to parse be checked for correctness
in a sense that all necessary components to estimate the model are given?
Defaults to \code{TRUE}.}
}
\value{
An object of class \link{cSEMModel} is a standardized list containing the
following components. J stands for the number of constructs and K for the number
of indicators.
\describe{
\item{\verb{$structural}}{A matrix mimicking the structural relationship between
constructs. If constructs are only linearly related, \code{structural} is
of dimension (J x J) with row- and column names equal to the construct
names. If the structural model contains nonlinear relationships
\code{structural} is (J x (J + J*)) where J* is the number of
nonlinear terms. Rows are ordered such that exogenous constructs are always
first, followed by constructs that only depend on exogenous constructs and/or
previously ordered constructs.}
\item{\verb{$measurement}}{A (J x K) matrix mimicking the measurement/composite
relationship between constructs and their related indicators. Rows are in the same
order as the matrix \verb{$structural} with row names equal to
the construct names. The order of the columns is such that \verb{$measurement}
forms a block diagonal matrix.}
\item{\verb{$error_cor}}{A (K x K) matrix mimicking the measurement error
correlation relationship. The row and column order is identical to
the column order of \verb{$measurement}.}
\item{\verb{$cor_specified}}{A matrix indicating the correlation relationships
between any variables of the model as specified by the user. Mainly for internal purposes.
Note that \verb{$cor_specified} may also contain inadmissible correlations
such as a correlation between measurement errors indicators and constructs.}
\item{\verb{$construct_type}}{A named vector containing the names of each construct
and their respective type ("Common factor" or "Composite").}
\item{\verb{$construct_order}}{A named vector containing the names of each construct
and their respective order ("First order" or "Second order").}
\item{\verb{$model_type}}{The type of model ("Linear" or "Nonlinear").}
\item{\verb{$instruments}}{Only if instruments are supplied: a list of structural
equations relating endogenous RHS variables to instruments.}
\item{\verb{$indicators}}{The names of the indicators
(i.e., observed variables and/or first-order constructs)}
\item{\verb{$cons_exo}}{The names of the exogenous constructs of the structural model
(i.e., variables that do not appear on the LHS of any structural equation)}
\item{\verb{$cons_endo}}{The names of the endogenous constructs of the structural model
(i.e., variables that appear on the LHS of at least one structural equation)}
\item{\verb{$vars_2nd}}{The names of the constructs modeled as second orders.}
\item{\verb{$vars_attached_to_2nd}}{The names of the constructs forming or building
a second order construct.}
\item{\verb{$vars_not_attached_to_2nd}}{The names of the constructs not forming or building
a second order construct.}
}
It is possible to supply an incomplete list to \code{\link[=parseModel]{parseModel()}}, resulting
in an incomplete \link{cSEMModel} list which can be passed
to all functions that require \code{.csem_model} as a mandatory argument. Currently,
only the structural and the measurement matrix are required.
However, specifying an incomplete \link{cSEMModel} list may lead to unexpected behavior
and errors. Use with care.
}
\description{
Turns a model written in \link[lavaan:model.syntax]{lavaan model syntax} into a
\link{cSEMModel} list.
}
\details{
Instruments must be supplied separately as a named list
of vectors of instruments.
The names of the list elements are the names of the dependent constructs of
the structural equation whose explanatory variables are endogenous.
The vectors contain the names of the instruments corresponding to each
equation. Note that exogenous variables of a given equation \strong{must} be
supplied as instruments for themselves.

By default \code{parseModel()} attempts to check if the model provided is correct
in a sense that all necessary components required to estimate the
model are specified (e.g., a construct of the structural model has at least
1 item). To prevent checking for errors use \code{.check_errors = FALSE}.
}
\examples{
# ===========================================================================
# Providing a model in lavaan syntax 
# ===========================================================================
model <- "
# Structural model
y1 ~ y2 + y3

# Measurement model
y1 =~ x1 + x2 + x3
y2 =~ x4 + x5
y3 =~ x6 + x7

# Error correlation
x1 ~~ x2
"

m <- parseModel(model)
m

# ===========================================================================
# Providing a complete model in cSEM format (class cSEMModel)
# ===========================================================================
# If the model is already a cSEMModel object, the model is returned as is:

identical(m, parseModel(m)) # TRUE

# ===========================================================================
# Providing a list 
# ===========================================================================
# It is possible to provide a list that contains at least the
# elements "structural" and "measurement". This is generally discouraged
# as this may cause unexpected errors.

m_incomplete <- m[c("structural", "measurement", "construct_type")]
parseModel(m_incomplete)

# Providing a list containing list names that are not part of a `cSEMModel`
# causes an error:

\dontrun{
m_incomplete[c("name_a", "name_b")] <- c("hello world", "hello universe")
parseModel(m_incomplete)
}

# Failing to provide "structural" or "measurement" also causes an error:

\dontrun{
m_incomplete <- m[c("structural", "construct_type")]
parseModel(m_incomplete)
}
}
