% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/families.R
\name{mixture}
\alias{mixture}
\title{Finite Mixture Families in \pkg{brms}}
\usage{
mixture(..., flist = NULL, nmix = 1, order = NULL)
}
\arguments{
\item{...}{One or more objects providing a description of the 
response distributions to be combined in the mixture model. 
These can be family functions, calls to family functions or 
character strings naming the families.
For details of supported families see 
\code{\link[brms:brmsfamily]{brmsfamily}}.}

\item{flist}{Optional list of objects, which are treated in the 
same way as objects passed via the \code{...} argument.}

\item{nmix}{Optional numeric vector specifying the number of times
each family is repeated. If specified, it must have the same length 
as the number of families passed via \code{...} or \code{flist}.}

\item{order}{Ordering constraint to identify mixture components.
If \code{'mu'} or \code{TRUE}, population-level intercepts
of the mean parameters are ordered. 
If \code{'none'} or \code{FALSE}, no ordering constraint is applied.
If \code{NULL} (the default), \code{order} is set to \code{'mu'}
if all families are the same and \code{'none'} otherwise.
Other ordering constraints may be implemented in the future.}
}
\value{
An object of class \code{mixfamily}.
}
\description{
Set up a finite mixture family for use in \pkg{brms}.
}
\details{
Most families supported by \pkg{brms} can be used to form 
mixtures. The response variable has to be valid for all components
of the mixture family. Currently, the number of mixture components 
has to be specified by the user. It is not yet possible to estimate 
the number of mixture components from the data.

For most mixture models, you may want to specify priors on the population-level
intercepts via \code{\link[brms:set_prior]{set_prior}} to improve convergence. 
In addition, it is sometimes necessary to set \code{inits = 0} in the call to 
\code{\link[brms:brms]{brm}} to allow chains to initialize properly.

For more details on the specification of mixture
models, see \code{\link[brms:brmsformula]{brmsformula}}.
}
\examples{
\dontrun{
## simulate some data
set.seed(1234)
dat <- data.frame(
  y = c(rnorm(200), rnorm(100, 6)), 
  x = rnorm(300),
  z = sample(0:1, 300, TRUE)
)

## fit a simple normal mixture model
mix <- mixture(gaussian, gaussian)
prior <- c(
  prior(normal(0, 7), Intercept, nlpar = mu1),
  prior(normal(5, 7), Intercept, nlpar = mu2)
)
fit1 <- brm(bf(y ~ x + z), dat, family = mix,
            prior = prior, chains = 2) 
summary(fit1)
pp_check(fit1)

## use different predictors for the components
fit2 <- brm(bf(y ~ 1, mu1 ~ x, mu2 ~ z), dat, family = mix,
            prior = prior, chains = 2) 
summary(fit2)

## fix the mixing proportions
fit3 <- brm(bf(y ~ x + z, theta1 = 1, theta2 = 2), 
            dat, family = mix, prior = prior, 
            inits = 0, chains = 2)
summary(fit3)
pp_check(fit3)    

## predict the mixing proportions
fit4 <- brm(bf(y ~ x + z, theta2 ~ x), 
            dat, family = mix, prior = prior, 
            inits = 0, chains = 2)
summary(fit4)
pp_check(fit4)           

## compare model fit
LOO(fit1, fit2, fit3, fit4)  
}

}
