\name{Imp.Estimates}
\alias{Imp.Estimates}
\alias{imp.moments}
\alias{imp.prob}
\alias{imp.quantile}
\alias{imp.reg}
\title{
Importance Sampling Estimates
}
\description{
Central moment, tail probability, and quantile estimates for a statistic
under importance resampling.
}
\usage{
imp.moments(boot.out=NULL, index=1, t=boot.out$t[, index], 
         w=NULL, def=TRUE, q=NULL)
imp.prob(boot.out=NULL, index=1, t0=boot.out$t0[index], 
         t=boot.out$t[, index], w=NULL, def=TRUE, q=NULL)
imp.quantile(boot.out=NULL, alpha=NULL, index=1, 
         t=boot.out$t[, index], w=NULL, def=TRUE, q=NULL)
}
\arguments{
\item{boot.out}{
A object of class \code{"boot"} generated by a call to \code{boot} or \code{tilt.boot}. 
Use of these functions makes sense only when the bootstrap resampling used
unequal weights for the observations.  If the importance weights \code{w} are not 
supplied then \code{boot.out} is a required argument. It is also required if
\code{t} is not supplied.
}
\item{alpha}{
The alpha levels for the required quantiles.  The default is to calculate
the 1\%, 2.5\%, 5\%, 10\%, 90\%, 95\%, 97.5\% and 99\% quantiles.
}
\item{index}{
The index of the variable of interest in the output of \code{boot.out$statistic}.
This is not used if the argument \code{t} is supplied.
}
\item{t0}{
The values at which tail probability estimates are required.  For each
value \code{t0[i]} the function will estimate the bootstrap cdf evaluated at
\code{t0[i]}.  If \code{imp.prob} is called without the argument \code{t0} then the bootstrap
cdf evaluated at the observed value of the statistic is found.
}
\item{t}{
The bootstrap replicates of a statistic.  By default these are taken from
the bootstrap output object \code{boot.out} but they can be supplied separately
if required (e.g. when the statistic of interest is a function of the
calculated values in \code{boot.out}).  Either \code{boot.out} or \code{t} must be supplied.
}
\item{w}{
The importance resampling weights for the bootstrap replicates.  If they are
not supplied then \code{boot.out} must be supplied, in
which case the importance weights are calculated by a call to \code{imp.weights}.
}
\item{def}{
A logical value indicating whether a defensive mixture is to be used for weight
calculation.  This is used only if \code{w} is missing and it is passed unchanged 
to \code{imp.weights} to calculate \code{w}.
}
\item{q}{
A vector of probabilities specifying the resampling distribution from which 
any estimates should be found.  In general this would correspond to the usual
bootstrap resampling distribution which gives equal weight to each of the 
original observations. The estimates depend on this distribution only through 
the importance weights \code{w} so this argument is ignored if \code{w} is supplied.  If
\code{w} is missing then \code{q} is passed as an argument to \code{imp.weights} and used to
find \code{w}.
}}
\value{
A list with the following components :



\item{alpha}{
The \code{alpha} levels used for the quantiles, if \code{imp.quantile} is used.
}
\item{t0}{
The values at which the tail probabilities are estimated, if \code{imp.prob}
is used.
}
\item{raw}{
The raw importance resampling estimates.  For \code{imp.moments} this has length 2,
the first component being the estimate of the mean and the second being the
variance estimate.  For \code{imp.prob}, \code{raw} is of the same length as \code{t0}, and
for \code{imp.quantile} it is of the same length as \code{alpha}.
}
\item{rat}{
The ratio importance resampling estimates.  In this method the weights \code{w} are
rescaled to have average value one before they are used.  The format of this
vector is the same as \code{raw}.
}
\item{reg}{
The regression importance resampling estimates.  In this method the weights
which are used are derived from a regression of \code{t*w} on \code{w}.  This choice of
weights can be shown to minimize the variance of the weights and also the
Euclidean distance of the weights from the uniform weights.  The format of this 
vector is the same as \code{raw}.
}}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Hesterberg, T. (1995) Weighted average importance sampling and defensive 
mixture distributions. \emph{Technometrics}, \bold{37}, 185--194.


Johns, M.V.  (1988) Importance sampling for bootstrap confidence intervals.
\emph{Journal of the American Statistical Association}, \bold{83}, 709--714.
}
\seealso{
\code{\link{boot}}, \code{\link{exp.tilt}}, \code{\link{imp.weights}}, \code{\link{smooth.f}}, \code{\link{tilt.boot}}
}
\examples{
# Example 9.8 of Davison and Hinkley (1997) requires tilting the 
# resampling distribution of the studentized statistic to be centred 
# at the observed value of the test statistic, 1.84.  In this example
# we show how certain estimates can be found using resamples taken from
# the tilted distribution.
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w, orig)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     as.vector(c(mns[2]-mns[1],s2hat,(mns[2]-mns[1]-orig)/sqrt(s2hat)))
}
grav.z0 <- grav.fun(grav1,rep(1,26),0)
grav.L <- empinf(data=grav1, statistic=grav.fun, stype="w", 
                 strata=grav1[,2], index=3, orig=grav.z0[1])
grav.tilt <- exp.tilt(grav.L,grav.z0[3],strata=grav1[,2])
grav.tilt.boot <- boot(grav1, grav.fun, R=199, stype="w", 
                       strata=grav1[,2], weights=grav.tilt$p,
                       orig=grav.z0[1])
# Since the weights are needed for all calculations, we shall calculate
# them once only.
grav.w <- imp.weights(grav.tilt.boot)
grav.mom <- imp.moments(grav.tilt.boot, w=grav.w, index=3)
grav.p <- imp.prob(grav.tilt.boot, w=grav.w, index=3, t0=grav.z0[3])
grav.q <- imp.quantile(grav.tilt.boot, w=grav.w, index=3, 
                       alpha=c(0.9,0.95,0.975,0.99))
}
\keyword{htest}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.
