% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Bcartime.R
\name{Bcartime}
\alias{Bcartime}
\title{Bayesian regression model fitting for areal and areal spatio-temporal data. 
Calculates parameter estimates, validation statistics, and 
estimated values of several Bayesian model choice criteria.}
\usage{
Bcartime(
  formula,
  data,
  family,
  link = NULL,
  trials = NULL,
  offsetcol = NULL,
  formula.omega = NULL,
  scol = NULL,
  tcol = NULL,
  package = "CARBayes",
  model = "glm",
  AR = 1,
  W = NULL,
  adj.graph = NULL,
  residtype = "response",
  interaction = TRUE,
  Z = NULL,
  W.binary = NULL,
  changepoint = NULL,
  knots = NULL,
  validrows = NULL,
  prior.mean.delta = NULL,
  prior.mean.beta = NULL,
  prior.var.beta = NULL,
  prior.mean.gamma = NULL,
  prior.var.gamma = NULL,
  prior.sigma2 = NULL,
  prior.tau2 = c(2, 1),
  prior.delta = NULL,
  prior.var.delta = NULL,
  prior.lambda = NULL,
  prior.nu2 = c(2, 1),
  epsilon = 0,
  G = NULL,
  ind.area = NULL,
  ind.re = NULL,
  trends = NULL,
  rho.T = NULL,
  rho.S = NULL,
  rho = NULL,
  rho.slo = NULL,
  rho.int = NULL,
  MALA = FALSE,
  N = 2000,
  burn.in = 1000,
  thin = 10,
  rseed = 44,
  Nchains = 4,
  verbose = TRUE,
  plotit = TRUE
)
}
\arguments{
\item{formula}{An object of class "formula" (or one that can be coerced to that class):
a symbolic description of the regression model to be fitted.}

\item{data}{The data frame for which the model formula is to be fitted.
The data frame should be in long format having one row for each location and  time
combination. The data frame must be ordered by time within each site, and should
optionally have a column, named s.index,  providing the site indices.
Thus the data,  with n sites and T times within each site, should be
organized in the order: (s1, t1), (s1, t2), ... (s1, T), ... (sn, t1), ... (sn, T). 
The data frame should also contain two columns giving the coordinates of the
locations for spatio temporal model fitting.}

\item{family}{One of either "gaussian", "binomial","poisson" or "zip", 
which respectively specify a Gaussian, binomial likelihood model with the 
logistic link function, a Poisson likelihood model with a log link function, 
or a zero-inflated Poisson model with a log link function.}

\item{link}{The link function to use for INLA based model fitting. This is 
ignored for the CARBayes and   CARBayesST models.}

\item{trials}{A vector the same length as the response containing the total number of trials 
for each area. Only used if family="binomial".}

\item{offsetcol}{Only used in INLA based modeling. The column name or number 
in the data frame that should be used as the offset.}

\item{formula.omega}{
A one-sided formula object with no response variable (left side of the "~") 
needed, specifying the covariates in the logistic regression model for 
modelling the probability of an observation being a structural zero. Each 
covariate (or an offset) needs to be a vector of length K*1. Only required for 
zero-inflated Poisson models.
}

\item{scol}{Either the name (character) or number of the column in the supplied data frame 
identifying the spatial units. The program will try to access data[, scol] 
to identify the spatial units. If this is omitted, no spatial modeling will be performed.}

\item{tcol}{Like the \code{scol} argument for the time identifier. 
 Either the name (character) or number of the column in the supplied data frame 
identifying the  time indices. The program will try to access data[, tcol] 
to identify the time points. If this is omitted, no temporal modeling will be performed.}

\item{package}{Which package is to be used in model fitting? Currently available 
packages are:
\itemize{  
\item{"inla"}{INLA model fitting for areal data.
 See \insertCite{gomezrubio;textual}{bmstdr}.}
\item{"CARBayes"}{All possible models in this package can be fitted. 
See \insertCite{LeeCARBayes2021;textual}{bmstdr}.}
\item{"CARBayesST"}{All possible models in this package can be fitted.
See \insertCite{CarBayesST;textual}{bmstdr}.}
Further details and more examples are provided in Chapters 10 and 11 of the book 
\insertCite{Sahubook;textual}{bmstdr}.}}

\item{model}{The specific spatio temporal model to be fitted. 
If the package is "INLA" then the model argument should be a vector with two elements 
giving the spatial model, e.g. "bym" as the first component and the temporal model which could be one of 
 "iid", "ar1" or "none" as the second component. 
 In case the second component is "none" then no temporal random effects 
 will be fitted. No temporal random effects will be fitted in case \code{model} is
 supplied as a  singleton.}

\item{AR}{
The order of the autoregressive time series process that must be either 1 or 2. 
}

\item{W}{A non-negative K by K neighborhood matrix (where K is the number of spatial units). 
Typically a binary specification is used, where the jkth element equals one if areas (j, k) 
are spatially close (e.g. share a common border) and is zero otherwise. 
The matrix can be non-binary, but each row must contain at least one non-zero entry.
This argument may not need to be specified if \code{adj.graph} is specified instead.}

\item{adj.graph}{Adjacency graph which may be specified instead of the adjacency matrix 
matrix. This argument is used if  \code{W} has not been supplied. The argument 
 \code{W} is used in case both \code{W} and \code{adj.graph} are supplied.}

\item{residtype}{Residual type, either "response" or "pearson",
in GLM fitting with the packages CARBayes and CARBayesST.
Default is "response" type observed minus fitted. The other option "pearson" is for 
Pearson residuals in GLM. For INLA based model fitting only the default response 
residuals are calculated.}

\item{interaction}{TRUE or FALSE indicating whether the spatio-temporal interaction 
random effects should be included. Defaults to TRUE unless family="gaussian" in which 
case interactions are not allowed.}

\item{Z}{
A list, where each element is a K by K matrix of non-negative dissimilarity metrics.
}

\item{W.binary}{
Logical, should the estimated neighbourhood matrix have only binary (0,1) values.
}

\item{changepoint}{
A scalar indicating the position of the change point should one of the change point 
trend functions be included in the trends vector, i.e. if "CP" or "CT" is specified.
}

\item{knots}{
A scalar indicating the number of knots to use should one of the monotonic cubic 
splines trend functions be included in the trends vector, i.e. if "MD" or "MI" is 
specified.
}

\item{validrows}{A vector providing the rows of the data frame which 
should be used for validation. 
The default NULL value instructs that validation will not be performed.}

\item{prior.mean.delta}{A vector of prior means for the regression parameters delta 
(Gaussian priors are assumed) for the zero probability logistic regression 
component of the model. Defaults to a vector of zeros.}

\item{prior.mean.beta}{
A vector of prior means for the regression parameters beta (Gaussian priors are 
assumed). Defaults to a vector of zeros.
}

\item{prior.var.beta}{
A vector of prior variances for the regression parameters beta (Gaussian priors 
are assumed). Defaults to a vector with values 100000.
}

\item{prior.mean.gamma}{
A vector of prior means for the temporal trend parameters (Gaussian priors are 
assumed). Defaults to a vector of zeros.
}

\item{prior.var.gamma}{
A vector of prior variances for the temporal trend parameters (Gaussian priors 
are assumed). Defaults to a vector with values 100000.
}

\item{prior.sigma2}{
The prior shape and scale in the form of c(shape, scale) for an Inverse-Gamma(shape, scale)
prior for sigma2. Defaults to c(1, 0.01).  
}

\item{prior.tau2}{
The prior shape and scale in the form of c(shape, scale) for an Inverse-Gamma(shape, scale) 
prior for tau2. Defaults to c(1, 0.01).  
}

\item{prior.delta}{
The prior maximum for the cluster smoothing parameter delta. Defaults to 10.  
}

\item{prior.var.delta}{A vector of prior variances for the regression parameters 
delta (Gaussian priors are assumed) for the zero probability logistic 
regression component of the model. Defaults to a vector with values 100000.}

\item{prior.lambda}{
A vector of prior samples sizes for the Dirichlet prior controlling the 
probabilities that each trend function is chosen. The vector should be the same 
length as the trends vector and defaults to a vector of ones.
}

\item{prior.nu2}{
The prior shape and scale in the form of c(shape, scale) for an Inverse-Gamma(shape, scale) 
prior for nu2. Defaults to c(1, 0.01) and only used if family="Gaussian".   
}

\item{epsilon}{Diagonal ridge parameter to add to the random effects prior precision matrix, 
only required when rho = 1, and the prior precision is improper. Defaults to 0. Only used for adaptive 
model fitting in CARBayesST.}

\item{G}{The maximum number of distinct intercept terms (groups) to allow in the localised model.}

\item{ind.area}{
        A vector of integers the same length as the number of data points (individuals) 
        giving which spatial unit (nunmbered from 1 to K to align with the rows of
        the W matrix) each individual belongs to.
        }

\item{ind.re}{
        A categorical variable where each level will be assigned a different independent
        random effect. If NULL these random effects are omitted.
        }

\item{trends}{
A vector containing the temporal trend functions to include in the model, which 
include: constant ("Constant""); linear decreasing ("LD"); linear increasing 
("LI"); Known change point, where the trend can increase towards the change point 
before subsequently decreasing ("CP"); or decrease towards the change point before 
subsequently increasing ("CT"); and monotonic cubic splines which are decreasing 
("MD") or increasing ("MI"). At least two trends have to be selected, with the 
constant trend always included. To avoid identifiability problems only one of "LI" 
or "MI" can be included at a given time (similarily for "LD" and "MD").
}

\item{rho.T}{The value in the interval [0, 1] that the temporal dependence parameter 
rho.T is fixed at if it should not be estimated. If this argument is NULL 
then rho.T is estimated in the model.}

\item{rho.S}{The value in the interval [0, 1] that the spatial dependence parameter rho.S is 
fixed at if it should not be estimated. If this argument is NULL then rho.S is 
estimated in the model.}

\item{rho}{The value in the interval [0, 1] that the spatial dependence parameter rho 
is fixed at if it should not be estimated. If this argument is NULL 
then rho is estimated in the model. Setting rho=1, reduces the random effects 
prior to the intrinsic CAR model but does require epsilon>0.}

\item{rho.slo}{The value in the interval [0, 1] that the spatial dependence parameter 
for the slope of the linear time trend, rho.slo, is fixed at if it should not be estimated. 
If this argument is NULL then rho.slo is estimated in the model.}

\item{rho.int}{The value in the interval [0, 1] that the spatial dependence parameter for 
the intercept of the linear time trend, rho.int, is fixed at if it should not 
be estimated. If this argument is NULL then rho.int is estimated in the model.}

\item{MALA}{Logical, should the function use Metropolis adjusted Langevin algorithm (MALA) 
updates (TRUE) or simple random walk (FALSE, default) updates for the 
regression parameters and random effects.}

\item{N}{MCMC sample size.}

\item{burn.in}{How many initial iterations to discard. 
Only relevant for MCMC based model fitting, i.e., when package is spBayes or Stan.}

\item{thin}{The level of thinning to apply to the MCMC samples to reduce 
their temporal autocorrelation. Defaults to 1 (no thinning).}

\item{rseed}{Random number seed that controls the starting point for the random number stream.
A set value is required to help reproduce the results.}

\item{Nchains}{
The number of parallel Markov chains to be used in the Metropolis coupled Markov 
chain Monte Carlo (MCMCMC) simulations. Defaults to 4.
}

\item{verbose}{Logical scalar value: whether to print various estimates and statistics.}

\item{plotit}{Logical scalar value: whether to plot the predictions against the observed values.}
}
\value{
A list containing:
\itemize{
   \item params -   A table of parameter estimates  
   \item fit -   The fitted model object.    
   \item fitteds -   A vector of fitted values.    
    \item mchoice -   Calculated model choice statistics if those have been 
    requested by the input argument \code{mchoice=T}. Not all model fits will contain 
    all the model choice statistics.  
   \item residuals -   A vector of residual values.   
    \item stats -   The four validation statistics: rmse, mae, crps and coverage. 
     This is present only if model validation has been performed.  
   \item yobs_preds -   A data frame containing the validation rows of the model 
   fitting data  frame. The last five columns of this data frame contains 
   the validation prediction summaries: mean, sd, median, and 95\% prediction interval. 
   This is present only if model validation has been performed.  
   \item valpreds -   A matrix containing the MCMC samples of the validation predictions. 
   The dimension of this matrix is the number of validations times the number of retained 
   MCMC samples. This is present only if model validation has been performed.   
   \item sn -   The number of areal units used in fitting.   
   \item tn -  The number of time points used in fitting.  
   \item formula - The input formula for the regression part of the model.   
    \item scale.transform -   It is there for compatibility with \code{Bsptime} output.    
   \item package -   The name of the package used for model fitting.   
   \item model -   The name of the fitted model.    
   \item call -   The command used to call the model fitting function.   
   \item computation.time -   Computation time required to run the model fitting.   
}
}
\description{
Bayesian regression model fitting for areal and areal spatio-temporal data. 
Calculates parameter estimates, validation statistics, and 
estimated values of several Bayesian model choice criteria.
}
\examples{


# Set the validation row numbers 
vs <- sample(nrow(engtotals), 5)
# Total number of iterations 
N <- 60
# Number of burn in iterations 
burn.in <- 10
# Number of thinning iterations
thin <- 1

# Set the model formula for binomial distribution based modeling 
f1 <- noofhighweeks ~ jsa + log10(houseprice) + log(popdensity) + sqrt(no2)
## Independent error logistic regression
M1 <- Bcartime(formula = f1, data = engtotals, family = "binomial",
    trials = engtotals$nweek, N = N, burn.in = burn.in, thin = thin,
    verbose = TRUE)
summary(M1)
# Leroux model
M1.leroux <- Bcartime(formula = f1, data = engtotals, scol = "spaceid",
    model = "leroux", W = Weng, family = "binomial", trials = engtotals$nweek,
    N = N, burn.in = burn.in, thin = thin)
summary(M1.leroux)
# BYM model
M1.bym <- Bcartime(formula = f1, data = engtotals, scol = "spaceid",
    model = "bym", W = Weng, family = "binomial", trials = engtotals$nweek,
    N = N, burn.in = burn.in, thin = thin, verbose = FALSE)
summary(M1.bym)

# Validation for the Leroux model
M1.leroux.v <- Bcartime(formula = f1, data = engtotals, scol = "spaceid",
    model = "leroux", W = Weng, family = "binomial", trials = engtotals$nweek,
    validrows = vs, N = N, burn.in = burn.in, thin = thin, verbose = FALSE)
summary(M1.leroux.v)


## Poisson Distribution based models ####################################
# Model formula
f2 <- covid ~ offset(logEdeaths) + jsa + log10(houseprice) + log(popdensity) +
    sqrt(no2)

# Independent error Poisson regression
M2 <- Bcartime(formula = f2, data = engtotals, family = "poisson", N = N,
    burn.in = burn.in, thin = thin, verbose = FALSE)
summary(M2)
## Poisson regression with Leroux Model
M2.leroux <- Bcartime(formula = f2, data = engtotals, scol = "spaceid",
    model = "leroux", family = "poisson", W = Weng, N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M2.leroux)
# Poisson regression with BYM Model
M2.bym <- Bcartime(formula = f2, data = engtotals, scol = "spaceid",
    model = "bym", family = "poisson", W = Weng, N = N, burn.in = burn.in,
    thin = thin)
summary(M2.bym)


## Gaussian distribution based models  ###############
f3 <- sqrt(no2) ~ jsa + log10(houseprice) + log(popdensity)

# Independent error model 
M3 <- Bcartime(formula = f3, data = engtotals, family = "gaussian", N = N,
    burn.in = burn.in, thin = thin, verbose = FALSE)
summary(M3)
# Leroux model 
M3.leroux <- Bcartime(formula = f3, data = engtotals, scol = "spaceid",
    model = "leroux", family = "gaussian", W = Weng, N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M3.leroux)

## Validation
M3.leroux.v <- Bcartime(formula = f3, data = engtotals, scol = "spaceid",
    model = "leroux", family = "gaussian", W = Weng, N = N, burn.in = burn.in,
    thin = thin, validrows = vs, verbose = FALSE)
summary(M3.leroux.v)

\donttest{

## Spatio-temporal modeling ##################################################
head(engdeaths)
dim(engdeaths)
colnames(engdeaths)
vs <- sample(nrow(engdeaths), 5)


## Binomial distribution
nweek <- rep(1, nrow(engdeaths))
f1 <- highdeathsmr ~ jsa + log10(houseprice) + log(popdensity)

M1st_linear <- Bcartime(formula = f1, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", trials = nweek, W = Weng, model = "linear",
    family = "binomial", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, verbose = TRUE)
summary(M1st_linear)
M1st_sepspat <- Bcartime(formula = f1, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", trials = nweek, W = Weng, model = "sepspatial",
    family = "binomial", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M1st_sepspat)
M1st_ar <- Bcartime(formula = f1, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", trials = nweek, W = Weng, model = "ar", AR = 1,
    family = "binomial", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M1st_ar)
# Model validation
M1st_ar.v <- Bcartime(formula = f1, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", trials = nweek, W = Weng, model = "ar", AR = 1,
    family = "binomial", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, validrows = vs, verbose = FALSE)
summary(M1st_ar.v)


## Spatio temporal Poisson models###################################
colnames(engdeaths)
f2 <- covid ~ offset(logEdeaths) + jsa + log10(houseprice) + log(popdensity) +
    n0

M2st_linear <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "linear", family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M2st_linear)
M2st_anova <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "anova", family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M2st_anova)
M2st_anova_nointer <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "anova", interaction = FALSE,
    family = "poisson", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M2st_anova_nointer)
M2st_sepspat <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "sepspatial", family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M2st_sepspat)
M2st_ar <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "ar", AR = 1, family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M2st_ar)
M2st_ar.v <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "ar", family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    validrows = vs, verbose = FALSE)
M2st_anova.v <- Bcartime(formula = f2, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "anova", family = "poisson",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    validrows = vs, verbose = FALSE)
summary(M2st_ar.v)
summary(M2st_anova.v)


## Spatio-temporal Normal models ###############################
colnames(engdeaths)
f3 <- sqrt(no2) ~ jsa + log10(houseprice) + log(popdensity)

M3st_linear <- Bcartime(formula = f3, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "linear", family = "gaussian",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M3st_linear)
M3st_anova <- Bcartime(formula = f3, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "anova", family = "gaussian",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M3st_anova)
M3st_anova_nointer <- Bcartime(formula = f3, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "anova", interaction = FALSE,
    family = "gaussian", package = "CARBayesST", N = N, burn.in = burn.in,
    thin = thin, verbose = FALSE)
summary(M3st_anova_nointer)
M3st_ar <- Bcartime(formula = f3, data = engdeaths, scol = "spaceid",
    tcol = "Weeknumber", W = Weng, model = "ar", AR = 2, family = "gaussian",
    package = "CARBayesST", N = N, burn.in = burn.in, thin = thin,
    verbose = FALSE)
summary(M3st_ar)

# Execute the following examples if INLA is available
if (require(INLA)) {
    N <- 55
    burn.in <- 5
    thin <- 1
    vs <- sample(nrow(engtotals), 5)
    nweek <- engtotals$nweek

    # Spatial Binomial GLM

    f1 <- noofhighweeks ~ jsa + log10(houseprice) + log(popdensity) +
        sqrt(no2)

    M1.inla.bym <- Bcartime(data = engtotals, formula = f1,
        W = Weng, scol = "spaceid", model = c("bym"), package = "inla",
        family = "binomial", trials = nweek, N = N, burn.in = burn.in,
        thin = thin)
    summary(M1.inla.bym)



    ## Spatial only Poisson

    f2inla <- covid ~ jsa + log10(houseprice) + log(popdensity) +
        sqrt(no2)

    M2.inla.bym <- Bcartime(data = engtotals, formula = f2inla,
        W = Weng, scol = "spaceid", offsetcol = "logEdeaths",
        model = c("bym"), package = "inla", link = "log",
        family = "poisson", N = N, burn.in = burn.in, thin = thin)
    summary(M2.inla.bym)



    ## Normal models

    f3 <- sqrt(no2) ~ jsa + log10(houseprice) + log(popdensity)

    ## Fit BYM with iid random effect
    M3.inla.bym <- Bcartime(data = engtotals, formula = f3,
        W = Weng, scol = "spaceid", model = c("bym"), package = "inla",
        family = "gaussian", N = N, burn.in = burn.in, thin = thin)
    summary(M3.inla.bym)

    # validation
    M3.inla.bym.v <- Bcartime(data = engtotals, formula = f3,
        W = Weng, scol = "spaceid", model = c("bym"), package = "inla",
        family = "gaussian", validrows = vs, N = N, burn.in = burn.in,
        thin = thin)

    summary(M3.inla.bym.v)


    ###### Spatio-temporal INLA models


    f1 <- highdeathsmr ~ jsa + log10(houseprice) + log(popdensity)
    nweek <- rep(1, nrow(engdeaths))


    ## INLA Binomial

    model <- c("bym", "ar1")
    M1st_inla.bym <- Bcartime(data = engdeaths, formula = f1,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        model = model, trials = nweek, family = "binomial",
        package = "inla", N = N, burn.in = burn.in, thin = thin)
    summary(M1st_inla.bym)

    M1st_inla_v <- Bcartime(data = engdeaths, formula = f1,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        offsetcol = "logEdeaths", model = model, trials = nweek,
        family = "binomial", package = "inla", validrow = vs,
        N = N, burn.in = burn.in, thin = thin)
    summary(M1st_inla_v)


    model <- c("bym", "none")
    M1st_inla.bym.none <- Bcartime(data = engdeaths, formula = f1,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        model = model, trials = nweek, family = "binomial",
        package = "inla", N = N, burn.in = burn.in, thin = thin)
    summary(M1st_inla.bym.none)


    model <- c("bym")
    M1st_inla.bym.none <- Bcartime(data = engdeaths, formula = f1,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        model = model, trials = nweek, family = "binomial",
        package = "inla", N = N, burn.in = burn.in, thin = thin)
    summary(M1st_inla.bym.none)


    ## Poisson models
    f2inla <- covid ~ jsa + log10(houseprice) + log(popdensity) +
        n0 + n1 + n2 + n3

    model <- c("bym", "ar1")
    M2stinla <- Bcartime(data = engdeaths, formula = f2inla,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        offsetcol = "logEdeaths", model = model, link = "log",
        family = "poisson", package = "inla", N = N, burn.in = burn.in,
        thin = thin)

    summary(M2stinla)

    M2stinla.v <- Bcartime(data = engdeaths, formula = f2inla,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        offsetcol = "logEdeaths", model = model, link = "log",
        family = "poisson", package = "inla", validrows = vs,
        N = N, burn.in = burn.in, thin = thin)

    summary(M2stinla.v)

    ## Normal models

    f3 <- sqrt(no2) ~ jsa + log10(houseprice) + log(popdensity)

    model <- c("bym", "iid")
    M3inla.bym.iid <- Bcartime(data = engdeaths, formula = f3,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        model = model, family = "gaussian", package = "inla",
        validrows = vs, N = N, burn.in = burn.in, thin = thin)
    summary(M3inla.bym.iid)

    model <- c("bym", "ar1")
    M3inla.bym.ar1 <- Bcartime(data = engdeaths, formula = f3,
        W = Weng, scol = "spaceid", tcol = "Weeknumber",
        model = model, family = "gaussian", package = "inla",
        validrows = vs, N = N, burn.in = burn.in, thin = thin)
    summary(M3inla.bym.ar1)

}
}

}
\references{
\insertAllCited{}
}
