% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design.r
\name{design}
\alias{design}
\title{General block and treatment designs.}
\usage{
design(
  treatments,
  blocks = NULL,
  treatments_model = NULL,
  weighting = 0.5,
  searches = NULL,
  seed = NULL,
  jumps = 1
)
}
\arguments{
\item{treatments}{a single treatment factor or a data frame containing any combination of 
treatment factors with any combination of treatment vectors.}

\item{blocks}{a single block factor or a data frame containing one or more qualitative level block
factors in the required order of fitting.}

\item{treatments_model}{a treatment model formula for the required treatment design.}

\item{weighting}{a weighting factor between 0 and 1 for weighting the 2-factor interaction effects of
factorial blocks.}

\item{searches}{the maximum number of local optima searched at each stage of an
optimization.}

\item{seed}{an integer initializing the random number generator.}

\item{jumps}{the number of pairwise random treatment swaps used to escape a local maxima.}
}
\value{
\item{Treatments}{The treatments included in the design and the replication of each individual 
treatment taken in de-randomized standard order.}
\item{Design}{The design layout showing the randomized allocation of treatments to blocks and plots.}
\item{Treatments_model}{The fitted treatment model, the number of model parameters (DF)
  and the D-efficiency of each sequentially fitted treatment model}
\item{Blocks_model}{The blocks sub-model design and 
the D- and A-efficiency factors of each successively fitted sub-blocks model.}
\item{seed}{Numerical seed for random number generator.}
\item{searches}{Maximum number of searches in each stratum.}
\item{jumps}{Number of random treatment swaps to escape a local maxima.}
}
\description{
Constructs D-optimal block and treatment designs for any feasible combination of block 
factors and any feasible linear treatment model.
}
\details{
\code{treatments} is a factor or a data frame containing one or more qualitative
or quantitative level treatment vectors defining a set of candidate treatments.
If the size of the candidate set is different from the size of the block design, 
or if the \code{treatments_model} is structured, the treatment design is optimized by
selection with replacement, otherwise the full fixed candidate set is used for the treatment design. 

\code{blocks} is a factor or a data frame containing one or more qualitative level block factors. The design is optimized
for each block factor added sequentially from left to right with the optimization constrained to be constant for all
previously added block factors. The length of the \code{blocks} object defines the total number of plots in the design. 
 
\code{treatments_model} is either a single formula or a compound formula split by the \code{|} operator.
The left hand side of each \code{|}, assuming all remaining \code{|} are replaced by 
\code{+}, is a partial model formula. Partial model formula define partial design matrices which are fitted
and optimized sequentially from left to right. Sequential model fitting provides improved flexibility
for fitting factors or variables of different status or importance (see examples below).

The treatment design criterion for each partial model is the generalized variance of the treatment design for that
model (D-optimality) and the design efficiency is the ratio of the generalized variance of the full candidate treatment model
for the required treatment model relative to the generalized variance of the optimized design. The efficiency is necessarily
less than or equal to 1 for purely factorial models but may exceed 1 for polynomial models. 

For crossed blocks, a weighting factor w is used to determine the relative importance
of the block main effects versus the block 2-factor interaction effects. If w = 0 the algorithm fits a simple additive
main effects model whereas if w = 1 the algorithm fits both main effects and 2-factor interaction effects.
For intermediate 0 < w < 1, the 2-factor interaction effects are downweighted relative to the main effects where
the smaller the value of w the greater the downweighting. The default weighting is 0.5 and 
provided that all block effects are estimable, this weighting gives a compromise design where all 
main effects and 2-factor interaction effects are included but where the main effects are given
greater realative importance than the 2-factor interaction effects.
See \code{vignette(package = "blocksdesign")} for more details. 

For more details see \code{vignette(package = "blocksdesign")}
}
\examples{

## For optimum results, the number of searches may need to be increased.

## 4 replicates of 12 treatments with 16 nested blocks of size 3
# rectangular lattice see Plan 10.10 Cochran and Cox 1957.
blocks = data.frame(Main = gl(4,12), Sub = gl(16,3))
design(treatments = factor(rep(1:12,4)),blocks)

 ## 6 replicates of 5 treatments with 2 super blocks of size 15 and 10 nested blocks of size 3
 blocks=data.frame(Main = gl(2,15,30), Sub = gl(10,3,30))
 design(treatments=gl(5,1,30),blocks)

## 4 x 12 design for 4 replicates of 12 treatments with 3 plots in each intersection block
## The optimal design is Trojan with known A-efficiency = 22/31 for the intersection blocks
blocks = data.frame(Rows = gl(4,12), Cols = gl(4,3,48))
design(treatments = factor(1:12),blocks)

## 4 x 12 design for 4 replicates of 12 treatments with 3 sub-column blocks nested 
## as above but showing 3 sub-columns nested within each main column
blocks = data.frame(Rows = gl(4,12), Cols = gl(4,3,48), subCols = gl(12,1,48))
\donttest{design(treatments = factor(1:12),blocks,searches=200)}

## 4 x 13 Row-and-column design for 4 replicates of 13 treatments 
## Youden design Plan 13.5 Cochran and Cox (1957).
blocks = data.frame(Rows = gl(4,13), Cols = gl(13,1,52))
\donttest{design(treatments = factor(1:13),blocks,searches = 700)}

## differential replication 
treatments=factor(c(rep(1:12,2),rep(13,12)))
blocks = data.frame(Main = gl(2,18),  Sub = gl(12,3,36))
design(treatments,blocks,searches = 5)

## 48 treatments in 2 replicate blocks with 2 nested rows in each replicate and 3 main columns
##  (Reps/Rows) x Cols
blocks = data.frame(Reps = gl(2,48), Rows = gl(4,24,96), Cols = gl(3,8,96))
design(treatments=factor(1:48),blocks,searches=5)

## 48 treatments in 2 replicate blocks with 2 main columns
## The default weighting gives non-estimable Reps:Cols effects due to inherent aliasing
## Increased weighting gives estimable Reps:Cols effects but non-orthogonal main effects
blocks = data.frame(Reps = gl(2,48), Cols = gl(2,24,96))
design(treatments=factor(1:48),blocks,searches=5)
design(treatments=factor(1:48),blocks,searches=5,weighting=.9)

## Factorial treatment designs defined by a single factorial treatment model

## Main effects of five 2-level factors in a half-fraction in 2/2/2 nested blocks design 
## (may require 100's of repeats to find a fully orthogonal solution - a VERY long wait!)
treatments = expand.grid(F1 = factor(1:2), F2 = factor(1:2),
 F3 = factor(1:2), F4 = factor(1:2), F5 = factor(1:2))
blocks = data.frame(b1 = gl(2,8),b2 = gl(4,4),b3 = gl(8,2))
model=" ~ F1 + F2 + F3 + F4 + F5"
\donttest{repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[3,3],1) ) ) break }
 print(z)}
 
# Second-order model for five qualitative 2-level factors in 4 randomized blocks
treatments = expand.grid(F1 = factor(1:2), F2 = factor(1:2), F3 = factor(1:2), 
F4 = factor(1:2), F5 = factor(1:2))
blocks = data.frame(blocks = gl(4,8))
model = " ~ (F1 + F2 + F3 + F4 + F5)^2"
design(treatments,blocks,treatments_model=model,searches = 10)

# Main effects of five 2-level factors in a half-fraction of 
# a 4 x 4 row-and column design.
treatments = expand.grid(F1 = factor(1:2), F2 = factor(1:2), F3 = factor(1:2), 
F4 = factor(1:2), F5 = factor(1:2))
blocks = data.frame(rows = gl(4,4), cols = gl(4,1,16))
model = "~ F1 + F2 + F3 + F4 + F5"
\donttest{repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[2,3],1) ) ) break }
 print(z)}

# Quadratic regression for three 3-level numeric factor assuming a 10/27 fraction
treatments = expand.grid(A = 1:3, B = 1:3, C = 1:3)
blocks=data.frame(main=gl(1,10))
model = " ~ ( A + B + C)^2 + I(A^2) + I(B^2) + I(C^2)"
design(treatments,blocks,treatments_model=model,searches=10) 

# Quadratic regression for three 3-level numeric factor crossed with a qualitative 2-level factor
treatments = expand.grid(F = factor(1:2), A = 1:3, B = 1:3, C = 1:3)
blocks=data.frame(main=gl(1,18))
model = " ~ F + A + B + C + F:A + F:B + F:C + A:B + A:C + B:C + I(A^2) + I(B^2) + I(C^2)"
design(treatments,blocks,treatments_model=model,searches=5) 

# 1st-order model for 1/3rd fraction of four qualitative 3-level factors in 3  blocks
treatments = expand.grid(F1 = factor(1:3), F2 = factor(1:3), F3 = factor(1:3), 
F4 = factor(1:3))
blocks = data.frame(main = gl(3,9))
model = " ~ F1 + F2 + F3 + F4"
\donttest{ design(treatments,blocks,treatments_model=model,searches=25)}

# 2nd-order model for a 1/3rd fraction of five qualitative 3-level factors in 3 blocks
# (may require many repeats to find a fully orthogonal solution)
treatments = expand.grid(F1 = factor(1:3), F2 = factor(1:3), F3 = factor(1:3), 
F4 = factor(1:3), F5 = factor(1:3))
blocks=data.frame(main=gl(3,27))
model = " ~ (F1 + F2 + F3 + F4 + F5)^2"
\donttest{ repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[1,3],1) ) ) break}
 print(z) }

# 2nd-order model for two qualitative and two quantitative level factors in 2 blocks of size 18
treatments = expand.grid(F1 = factor(1:2), F2 = factor(1:3), V1 = 1:3, V2 = 1:4)
blocks = data.frame(main = gl(2,18))
model = " ~ (F1 + F2 + V1 + V2)^2 +  I(V1^2) +  I(V2^2)"
\donttest{ design(treatments,blocks,treatments_model=model,searches=5)}
 
# Plackett and Burman design for eleven 2-level factors in 12 runs 
GF = expand.grid(F1 = factor(1:2,labels=c("a","b")), F2 = factor(1:2,labels=c("a","b")), 
                 F3 = factor(1:2,labels=c("a","b")), F4 = factor(1:2,labels=c("a","b")),
                 F5 = factor(1:2,labels=c("a","b")), F6 = factor(1:2,labels=c("a","b")),
                 F7 = factor(1:2,labels=c("a","b")), F8 = factor(1:2,labels=c("a","b")), 
                 F9 = factor(1:2,labels=c("a","b")), F10= factor(1:2,labels=c("a","b")), 
                 F11= factor(1:2,labels=c("a","b")) )
blocks=data.frame(main=gl(1,12))
model = "~ F1 + F2 + F3 + F4 + F5 + F6 + F7 + F8 + F9 + F10 + F11"
\donttest{design(GF,blocks,treatments_model=model,searches=25)}

## Factorial treatment designs defined by sequentially fitted factorial treatment models

## 2 varieties x 3 levels of N x 3 levels of K assuming 1st-order interactions and 12 plots
## the single stage model gives an unequal 7 + 5 split for the two varieties
## whereas the two stage model forces an equal 6 + 6 split
## NB the two stage model is slightly less efficient than the single stage model 
treatments = expand.grid(Variety = factor(rep(1:2)), N = 1:3, K = 1:3)
blocks=data.frame(main=gl(1,12))
treatments_model = " ~  (Variety + N + K)^2  + I(N^2) + I(K^2)"
\donttest{design(treatments,blocks,treatments_model=treatments_model,searches=10)} 
treatments_model = " ~ Variety | (Variety + N + K)^2 + I(N^2) + I(K^2)"
\donttest{design(treatments,blocks,treatments_model=treatments_model,searches=10)}

## A 6 x 6 row-and-column design with linear row by linear column interaction.
## Crossed blocks with interactions fitted in the treatments model and additive 
## treatments fitted inthe blocks model as a dual design
##  see vignette(package = "blocksdesign") for further discussion
## may require many separate attempts to get the best overall design efficiency 
LS_grid   = expand.grid(rows=factor(1:6), cols=factor(1:6))
blocks = data.frame(varieties=factor(rep(1:6,6)))
lin_rows = as.numeric(levels(LS_grid$rows))[LS_grid$rows]
lin_cols = as.numeric(levels(LS_grid$cols))[LS_grid$cols]
latin_sq = "~  rows | cols + lin_rows:lin_cols "
\donttest{design(LS_grid,blocks,latin_sq,searches=2000)} 

}
\references{
Cochran W. G. & Cox G. M. (1957) Experimental Designs 2nd Edition John Wiley & Sons.
}
