#' Helpers for supplying SMTP credentials
#'
#' These helper functions, the credential helpers, are used to supply SMTP
#' configuration and authorization information for the [smtp_send()] function.
#' The [creds_file()], [creds_anonymous()], [creds_key()], and [creds()]
#' functions are to be used expressly with the `credentials` argument of
#' [smtp_send()].
#'
#' The [creds()] credential helper allows for manual specification of SMTP
#' configuration and authentication.
#'
#' The [creds_anonymous()] credential helper is similar to [creds()] but
#' provides convenient defaults for authenticating anonymously with an SMTP
#' server.
#'
#' The [creds_key()] credential helper gets credentials stored in the
#' system-wide key-value store. We can set that key and the credentials data
#' using the [create_smtp_creds_key()] function.
#'
#' The [creds_file()] credential helper is used to obtain credentials from a
#' file stored on disk. We can create that file using the
#' [create_smtp_creds_file()] function.
#'
#' @param user The username for the email account. Typically, this is the email
#'   address associated with the account.
#' @param provider An optional email provider shortname for autocompleting STMP
#'   configuration details (the `host`, `port`, `use_ssl` options). Options
#'   currently include `gmail`, `outlook`, and `office365`. If nothing is
#'   provided then values for `host`, `port`, and `use_ssl` are expected.
#' @param host,port,use_ssl Configuration info for the SMTP server. The `host`
#'   and `port` parameters are the address and port for the SMTP server;
#'   `use_ssl` is an option as to whether to use SSL: supply a `TRUE` or `FALSE`
#'   value.
#' @param id When using the [creds_key()] credential helper, the ID value of the
#'   key (in the system key-value store) needs to be given here. This was
#'   explicitly provided when using the [create_smtp_creds_key()] function (with
#'   its own `id` argument). To get an information table with all available
#'   \pkg{blastula} keys in the key-value store, we can use the
#'   [view_credential_keys()] function.
#' @param file When using the [creds_file()] credential helper, we need to
#'   specify the location of the credential file, and, this is where that is
#'   done. The credential file was ideally generated by the
#'   [create_smtp_creds_file()] function.
#' @name credential_helpers
#' @return A credentials list object.
NULL

#' @rdname credential_helpers
#' @export
creds <- function(user = NULL,
                  provider = NULL,
                  host = NULL,
                  port = NULL,
                  use_ssl = TRUE) {

  # Create a credentials list from the function inputs
  creds_list <-
    create_credentials_list(
      provider = provider,
      user = user,
      host = host,
      port = port,
      use_ssl = use_ssl
    )

  class(creds_list) <- c("creds", "blastula_creds")
  creds_list
}

#' @rdname credential_helpers
#' @export
creds_anonymous <- function(provider = NULL,
                            host = NULL,
                            port = NULL,
                            use_ssl = TRUE) {

  creds_list <-
    creds_internal(
      user = NULL,
      password = NULL,
      provider = provider,
      host = host,
      port = port,
      use_ssl = use_ssl
    )

  class(creds_list) <- c("creds_anonymous", "blastula_creds")
  creds_list
}

#' @rdname credential_helpers
#' @export
creds_key <- function(id) {

  creds_list <- get_smtp_keyring_creds(id = id)

  class(creds_list) <- c("creds_key", "blastula_creds")
  creds_list
}

#' @rdname credential_helpers
#' @export
creds_file <- function(file) {

  creds_list <- get_smtp_file_creds(file_name = file)

  class(creds_list) <- c("creds_file", "blastula_creds")
  creds_list
}


#' @noRd
#' @export
format.blastula_creds <- function(x, ...) {

  if (!is.null(x$password)) {
    x$password <- "****"
  }

  utils::capture.output(utils::str(x)) %>%
    .[-1 * c(1, length(.))] %>%
    paste0(collapse = "\n")
}

#' @noRd
#' @export
print.blastula_creds <- function(x, ...) {

  cat(format(x, ...), "\n")
}
