\name{blackbox}
\alias{blackbox}
\alias{blackbox-package}
\docType{package}
\title{Black box optimization and response surface exploration}
\description{
blackbox allows prediction and optimization of a response function from simulated response values.  It also includes procedures designed mainly or only to be called, in a completely automated way without any input by users, by other R packages such as the Infusion package (under development) or by R code automatically generated by the Migraine software (see Details). For prediction, blackbox interfaces a C++ library for \dQuote{ordinary Kriging} (which is jargon for: prediction in a linear mixed model with a constant term as fixed effect). It uses generalized cross validation (GCV) by default to estimate smoothing parameters.
}
\details{
Beyond the usage illustrated below, this package is used in particular for smoothing the output of the \code{Migraine} software for likelihood analysis of population genetic data
(\url{https://kimura.univ-montp2.fr/~rousset/Migraine.htm}). In the latter application  the response function is a simulated log-likelihood surface and the procedures generate plots of the (profile) log-likelihood, compute (profile) likelihood ratio confidence intervals, and design new parameter points where the likelihood should be simulated. This package provides documentation for all user-level functions in the R script written by Migraine. Control from Migraine uses many variables stored globally in the list of options accessible through \code{blackbox.options().}

The C++ DLL was originally a c++ reimplementation of some of the internal functions of the fields package, circa 2005-2006. To estimate smoothing parameters, it requires pairs of responses values for some values of the predictor variables, but will not allow more than pairs.
}
\author{\enc{François}{Francois} Rousset, with contributions by \enc{Raphaël}{Raphael} Leblois.}
\references{
Chambers J.M. (2008) Software for data analysis: Programming with R. Springer-Verlag.

Fields Development Team (2006). fields: Tools for Spatial Data. National Center for Atmospheric Research, Boulder, CO. \url{https://www.image.ucar.edu/GSP/Software/Fields/}.
}
\examples{
fr <- function(v) { ## Rosenbrock Banana function with noise
  10 * (v["y"] - v["x"]^2)^2 + (1 - v["x"])^2 + rnorm(1,sd=0.1)
}
set.seed(123)

# Initial parameter values, including duplicates. See ?init_grid.
parsp <- init_grid(lower=c(x=0,y=0),upper=c(x=2,y=2))

# add function values
simuls <- cbind(parsp,bb=apply(parsp,1,"fr"))

## The following shows the backbone of the 'bboptim' code:

sorted_etc <- prepareData(data=simuls)
#   Then smoothing using GCV (beware of implicit 'decreasing=FALSE' argument)
gcvres <- calcGCV(sorted_etc)

## The results can be used as input to functions from other packages,
##  e.g. corrHLfit from spaMM:
\dontrun{
require(spaMM)
corrHLfit(bb ~ 1+Matern(1|x+y),data=sorted_etc,
          ranFix=list(rho=1/gcvres$CovFnParam[c("x", "y")],
          #         note '1/...'
                      nu=gcvres$CovFnParam[["smoothness"]],
                      phi=gcvres$pureRMSE^2,
          # note distinct meaning of lambda notation in spaMM and blackbox
                      lambda=with(gcvres,(pureRMSE^2)/lambdaEst)))

## GCV is distinct from an REML fit:
corrHLfit(bb ~ 1+Matern(1|x+y),data=sorted_etc,
          init.corrHLfit=list(rho=c(1,1)))
}

}
