\name{predRes}
\alias{predRes}
\alias{plot.predRes}
\title{Evaluation of the prediction accuracy of a prediction model}
\description{
  This function computes several criteria to assess the prediction
  accuracy of a prediction model.
}
\usage{
predRes(res, method, traindata, newdata, is.2cv, fold.2cv = 5, horizon,
  trace = TRUE, ncores = 1)

\method{plot}{predRes}(x, method, crit = c("C", "PE", "dC"),
  xlim, ylim, xlab, ylab, col,...)
}
\arguments{
  \item{res}{an object of class '\code{resBMsel}' generated by \code{\link{BMsel}}.}
  \item{method}{methods for which prediction criteria are computed. If missing, all methods contained in \code{res} are computed.}
  \item{traindata}{input \code{\link{data.frame}} used to compute the
\code{res} object. This object is mandatory.}
  \item{newdata}{input \code{\link{data.frame}} not used to compute the
\code{res} object. This object is not mandatory (see Details section).}
  \item{is.2cv}{logical parameter indicating if a double
cross-validation process (2CV) should be performed to mimick an external
validation set.}
  \item{fold.2cv}{number of folds for the double cross-validation. Considering a large value for \code{fold.2cv} should provide extremely large computation time. \code{fold.2cv} must not be considered when \code{is.2cv = FALSE}.}
  \item{horizon}{time horizon to compute the prediction criteria.}
  \item{trace}{logical parameter indicating if messages should be printed.}
  \item{ncores}{number of CPUs used (for the double cross-validation).}
  \item{x}{an object of class '\code{predRes}' generated from \code{\link{predRes}}.}
  \item{crit}{parameter indicating the criterion for which the results will be printed (\code{C}: concordance via Uno's C-statistic, \code{PE}: prediction error via integrated Brier score and \code{dC}: delta Uno's C-statistic (for the interaction setting only)).}
  \item{xlim, ylim, xlab, ylab, col}{usual parameters for \link{plot}.}
  \item{...}{other paramaters for \link{plot}.}
}
\details{
To evaluate the accuracy of the selected models, three predictive accuracy measures are implemented: \cr
- the integrated Brier score (\code{PE}) to measure the overall prediction error of the prediction model. The time-dependent Brier score is a quadratic score based on the predicted time-dependent survival probability.\cr
- the Uno's C-statistic (\code{C}) to evaluate the discrimination of the prediction model. It's one of the least biased concordance statistic estimator in the presence of censoring (Uno et al., 2011).\cr
- the absolute difference of the treatment-specific Uno's C-statistics (\code{dC}) to evaluate the interaction strength of the prediction model (Ternes et al., 2016).\cr
For simulated datasets, the predictive accuracy metrics are also computed for the "oracle model" that is the unpenalized Cox proportional hazards model fitted to the active biomarkers only.
}
\value{
  A \code{list} of the same length of the \code{horizon} considered. Each element of the list contains between 1 and 3 sublists depending on the chosen validation (i.e. training set [always computed], internal validation through double cross-validation (2CV) [if \code{is.2cv} = \code{TRUE}] and/or external validation [if \code{newdata} is provided]). Each sublist is a \code{matrix} containing the predictive accuracy metrics of the implemented methods.
}
\author{Nils Ternes, Federico Rotolo, and Stefan Michiels\cr
Maintainer: Nils Ternes \email{nils.ternes@yahoo.com}
}
\references{
  Ternes N, Rotolo F and Michiels S.
    Empirical extensions of the lasso penalty to reduce
    the false discovery rate in high-dimensional Cox regression models.
    \emph{Statistics in Medicine} 2016;35(15):2561-2573.
    doi:\href{http://onlinelibrary.wiley.com/doi/10.1002/sim.6927/abstract}{10.1002/sim.6927}\cr
  Ternes N, Rotolo F, Heinze G and Michiels S.
    Identification of biomarker-by-treatment interactions in randomized
    clinical trials with survival outcomes and high-dimensional spaces.
    \emph{Biometrical journal}. In press.
    doi:\href{http://onlinelibrary.wiley.com/doi/10.1002/bimj.201500234/abstract}{10.1002/bimj.201500234}\cr
  Uno H, Cai T, Pencina MJ, DAgostino RB and Wei LJ.
    On the C-statistics for evaluating overall adequacy 
    of risk prediction procedures with censored survival data.
    \emph{Statistics in Medicine} 2011;30:1105-1117.
    doi:\href{http://onlinelibrary.wiley.com/doi/10.1002/sim.4154/abstract}{10.1002/sim.4154}
}

\examples{
########################################
# Simulated data set
########################################

## Low calculation time
  set.seed(654321)
  sdata <- simdata(
    n = 500, p = 20, q.main = 3, q.inter = 0,
    prob.tt = 0.5, alpha.tt = 0,
    beta.main = -0.8,
    b.corr = 0.6, b.corr.by = 4,
    m0 = 5, wei.shape = 1, recr = 4, fu = 2,
    timefactor = 1)
  
  newdata <- simdataV(
    traindata = sdata,
    Nvalid = 500
  )
   
  resBM <- BMsel(
    data = sdata, 
    method = c("lasso", "lasso-pcvl"), 
    inter = FALSE, 
    folds = 5)
  
  predAcc <- predRes(
    res = resBM,
    traindata = sdata,
    newdata = newdata,
    horizon = 1:5)
    
  plot(predAcc, crit = "C")

\dontrun{
## Moderate calculation time
  set.seed(123456)
  sdata <- simdata(
    n = 500, p = 100, q.main = 5, q.inter = 5,
    prob.tt = 0.5, alpha.tt = -0.5,
    beta.main = c(-0.5, -0.2), beta.inter = c(-0.7, -0.4),
    b.corr = 0.6, b.corr.by = 10,
    m0 = 5, wei.shape = 1, recr = 4, fu = 2,
    timefactor = 1,
    active.inter = c("bm003", "bm021", "bm044", "bm049", "bm097"))

  resBM <- BMsel(
    data = sdata, 
    method = c("lasso", "lasso-pcvl"), 
    inter = TRUE, 
    folds = 5)
  
  predAcc <- predRes(
    res = resBM,
    traindata = sdata, 
    is.2cv = TRUE, 
    horizon = 1:5, 
    ncores = 5)
  plot(predAcc, crit = "dC")
}

########################################
# Breast cancer data set
########################################

\dontrun{
  data(Breast)
  dim(Breast)
  
  set.seed(123456)
  resBM <-  BMsel(
    data = Breast,
    x = 4:ncol(Breast),
    y = 2:1,
    tt = 3,
    inter = FALSE,
    std.x = TRUE,
    folds = 5,
    method = c("lasso", "lasso-pcvl"))

  summary(resBM)

  predAcc <- predRes(
    res = resBM,
    traindata = Breast,
    horizon = 1:4,
    trace = TRUE)
  plot(predAcc, crit = "C")
}

########################################
########################################
}

\keyword{prediction}
