% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_dynamic_growth.R
\name{predict_dynamic_growth}
\alias{predict_dynamic_growth}
\title{Growth under dynamic conditions}
\usage{
predict_dynamic_growth(
  times,
  env_conditions,
  primary_pars,
  secondary_models,
  ...,
  check = TRUE,
  formula = . ~ time
)
}
\arguments{
\item{times}{Numeric vector of storage times to make the predictions}

\item{env_conditions}{Tibble describing the variation of the environmental
conditions during storage. It must have with the elapsed time (named \code{time} 
by default; can be changed with the "formula" argument), 
and as many additional columns as environmental factors.}

\item{primary_pars}{A named list defining the parameters of the primary model
and the initial values of the model variables. That is, with names \code{mu_opt},
\code{Nmax}, \code{N0}, \code{Q0}.}

\item{secondary_models}{A nested list describing the secondary models.}

\item{...}{Additional arguments for \code{\link{ode}}.}

\item{check}{Whether to check the validity of the models. \code{TRUE} by default.}

\item{formula}{An object of class "formula" describing the x variable.
\code{. ~ time} as a default.}
}
\value{
An instance of \code{\link{DynamicGrowth}}.
}
\description{
Predicts population growth under dynamic conditions based on the
Baranyi model (Baranyi and Roberts, 1994) and secondary models
based on the gamma concept (Zwietering et al. 1992).
}
\details{
Model predictions are done by linear interpolation of the environmental
conditions defined in \code{env_conditions}.
}
\examples{

## Definition of the environmental conditions

library(tibble)

my_conditions <- tibble(time = c(0, 5, 40),
    temperature = c(20, 30, 35),
    pH = c(7, 6.5, 5)
    )

## Definition of the model parameters

my_primary <- list(mu_opt = 2,
    Nmax = 1e8,N0 = 1e0,
    Q0 = 1e-3)

sec_temperature <- list(model = "Zwietering",
    xmin = 25, xopt = 35, n = 1)

sec_pH = list(model = "CPM",
    xmin = 5.5, xopt = 6.5,
    xmax = 7.5, n = 2)

my_secondary <- list(
    temperature = sec_temperature,
    pH = sec_pH
    )

my_times <- seq(0, 50, length = 1000)

## Do the simulation

dynamic_prediction <- predict_dynamic_growth(my_times,
    my_conditions, my_primary,
    my_secondary)

## Plot the results

plot(dynamic_prediction)

## We can plot some environmental factor with add_factor

plot(dynamic_prediction, add_factor = "temperature", ylims= c(0, 8),
    label_y1 = "Microbial count (log CFU/ml)",
    label_y2 = "Storage temperature (C)")

}
