% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_vp.R
\name{calculate_vp}
\alias{calculate_vp}
\title{Calculate a vertical profile (\code{vp}) from a polar volume (\code{pvol})}
\usage{
calculate_vp(
  file,
  vpfile = "",
  pvolfile_out = "",
  autoconf = FALSE,
  verbose = FALSE,
  mount = dirname(file[1]),
  sd_vvp_threshold,
  rcs = 11,
  dual_pol = FALSE,
  rho_hv = 0.95,
  elev_min = 0,
  elev_max = 90,
  azim_min = 0,
  azim_max = 360,
  range_min = 5000,
  range_max = 35000,
  n_layer = 20L,
  h_layer = 200,
  dealias = TRUE,
  nyquist_min = if (dealias) 5 else 25,
  dbz_quantity = "DBZH",
  mistnet = FALSE,
  local_install,
  pvolfile
)
}
\arguments{
\item{file}{string or a vector of strings with radar file(s) for a radar polar volume.
Provide either a single file containing a polar volume, or multiple files with single scans/sweeps.
Data format should be either
\href{https://github.com/adokter/vol2bird/blob/master/doc/OPERA2014_O4_ODIM_H5-v2.2.pdf}{ODIM}
format, which is the implementation of the OPERA data information model in
\href{https://support.hdfgroup.org/HDF5/}{HDF5} format, or a format
supported by the
\href{http://trmm-fc.gsfc.nasa.gov/trmm_gv/software/rsl/}{RSL library}, or Vaisala IRIS (IRIS RAW) format.}

\item{vpfile}{character. Filename for the vertical profile to be
generated in ODIM HDF5 format (optional).}

\item{pvolfile_out}{character. Filename for the polar volume to be
generated in ODIM HDF5 format (optional, e.g. for converting RSL formats
to ODIM).}

\item{autoconf}{logical. When TRUE, default optimal configuration settings
are selected automatically, and other user settings are ignored.}

\item{verbose}{logical. When TRUE, pipe Docker stdout to R console. On
Windows always TRUE.}

\item{mount}{character. String with the mount point (a directory path) for
the Docker container.}

\item{sd_vvp_threshold}{numeric. Lower threshold in radial velocity standard
deviation (profile quantity \code{sd_vvp}) in m/s. Biological signals with
\code{sd_vvp} < \code{sd_vvp_threshold} are set to zero. Defaults to 2 m/s
for C-band radars and 1 m/s for S-band radars if not specified.}

\item{rcs}{numeric. Radar cross section per bird in cm^2.}

\item{dual_pol}{logical. When \code{TRUE} use dual-pol mode, in which
meteorological echoes are filtered using the correlation coefficient
\code{rho_hv}. When \code{FALSE} use single polarization mode based only
on reflectivity and radial velocity quantities.}

\item{rho_hv}{numeric. Lower threshold in correlation coefficient used to
filter meteorological scattering.}

\item{elev_min}{numeric. Minimum scan elevation in degrees.}

\item{elev_max}{numeric. Maximum scan elevation in degrees.}

\item{azim_min}{numeric. Minimum azimuth in degrees clockwise from north.}

\item{azim_max}{numeric. Maximum azimuth in degrees clockwise from north.}

\item{range_min}{numeric. Minimum range in m.}

\item{range_max}{numeric. Maximum range in m.}

\item{n_layer}{numeric. Number of altitude layers in the profile.}

\item{h_layer}{numeric. Width of altitude layers in meter.}

\item{dealias}{logical. Whether to dealias radial velocities; this should
typically be done when the scans in the polar volume have low Nyquist
velocities (below 25 m/s).}

\item{nyquist_min}{numeric. Minimum Nyquist velocity of scans in m/s for
scans to be included in the analysis.}

\item{dbz_quantity}{character. One of the available reflectivity factor
quantities in the ODIM radar data format, e.g. DBZH, DBZV, TH, TV.}

\item{mistnet}{logical. Whether to use MistNet segmentation model.}

\item{local_install}{(optional) String with path to local vol2bird installation, see details.}

\item{pvolfile}{deprecated argument renamed to \code{file}.}
}
\value{
A vertical profile object of class \link[=summary.vp]{vp}. When
defined, output files \code{vpfile} and \code{pvolfile_out} are saved to disk.
}
\description{
Calculates a vertical profile of biological scatterers (vp) from a polar volume (pvol)
using the algorithm \href{https://github.com/adokter/vol2bird/}{vol2bird} (Dokter et al. 2011).
}
\details{
Requires a running \href{https://www.docker.com/}{Docker} daemon
(unless a local installation of vol2bird is specified with \code{local_install}).

Common arguments set by users are \code{file}, \code{vpfile},
\code{autoconf} and \code{mount}.

Turn on \code{autoconf} to automatically select the optimal parameters for a
given radar file. The default for C-band data is to apply rain-filtering in
single polarization mode, as well as dual polarization mode when available.

The default for S-band data is to apply precipitation filtering in
dual-polarization mode.

Arguments that sometimes require non-default values are: \code{rcs},
\code{sd_vvp_threshold}, \code{range_max}, \code{dual_pol}, \code{dealias}.

Other arguments are typically left at their defaults.

\code{azim_min} and \code{azim_max} only affects reflectivity-derived
estimates in the profile (DBZH,eta,dens), not radial-velocity derived
estimates (u, v, w, ff, dd, sd_vvp), which are estimated on all azimuths at
all times. \code{azim_min}, \code{azim_max} may be set to exclude an angular
sector with high ground clutter.

\code{range_max} may be extended up to 40,000 m for volumes with low
elevations only, in order to extend coverage to higher altitudes.

For altitude layers with a VVP-retrieved radial velocity standard deviation
value below the threshold \code{sd_vvp_threshold}, the bird density \code{dens} is set
to zero (see vertical profile \link[=summary.vp]{vp} class). This threshold
might be dependent on radar processing settings. Results from validation
campaigns so far indicate that 2 m/s is the best choice for this parameter
for most C-band weather radars, which is used as the C-band default. For S-band,
the default threshold is 1 m/s.
The algorithm has been tested and developed for altitude layers with
\code{h_layer} = 200 m. Smaller widths are not recommended as they may cause
instabilities of the volume velocity profiling (VVP) and dealiasing routines,
and effectively lead to pseudo-replicated altitude data, since altitudinal
patterns smaller than the beam width cannot be resolved.

The default radar cross section (11 cm^2) corresponds to the average value
found by Dokter et al. in a calibration campaign of a full migration autumn
season in western Europe at C-band. It's value may depend on radar
wavelength. \code{rcs} will scale approximately \eqn{M^{2/3}} with \code{M}
the bird's mass.

Using default values of \code{range_min} and \code{range_max} is
recommended. Ranges closer than 5 km tend to be contaminated by ground
clutter, while range gates beyond 35 km become too wide to resolve the
default altitude layer width of 200 meter (see \link{beam_width}).

For dealiasing, the torus mapping method by Haase et al. is used.

At S-band (radar wavelength ~ 10 cm), currently only \code{dual_pol=TRUE}
mode is recommended.

On repeated calls of \code{calculate_vp}, the Docker container mount can be
recycled from one call to the next if subsequent calls share the same
\code{mount} argument. Re-mounting a Docker container takes time, therefore
it is advised to choose a mountpoint that is a parent directory of all
volume files to be processed, such that \code{calculate_vp} calls are as fast
as possible.

If you have installed the vol2bird algorithm locally (not possible on Windows)
you can call vol2bird through this local installation (bypassing the Docker container),
which will be faster. Simply point \code{local_install} to the path
of your local vol2bird executable. Your local vol2bird executable will be called
through a bash login shell. LD_LIBRARY_PATH (Linux) or DYLD_LIBRARY_PATH (Mac) should be
correctly specified in your .bashrc or .bash_profile file
and contain all the required shared libraries by vol2bird. See vol2bird installation
pages on Github for details.
}
\examples{
# locate example polar volume file:
pvolfile <- system.file("extdata", "volume.h5", package = "bioRad")

# copy to a home directory with read/write permissions:
file.copy(pvolfile, "~/volume.h5")

# calculate the profile:
\dontrun{
profile <- calculate_vp("~/volume.h5")
# print some summary info:
profile
# convert profile to a data.frame:
as.data.frame(profile)
}

# clean up:
file.remove("~/volume.h5")
}
\references{
Dokter et al. (2011) is the main reference for the profiling algorithm
(vol2bird) underlying this function. When using the \code{mistnet} option,
please also cite Lin et al. 2019. When de-aliasing data, please also cite Haase et al. 2004.

\itemize{
  \item Adriaan M. Dokter, Felix Liechti,
  Herbert Stark, Laurent Delobbe, Pierre Tabary, Iwan Holleman, 2011.
  Bird migration flight altitudes studied by a network of
  operational weather radars,
  Journal of the Royal Society Interface 8 (54), pp. 30--43.
  \url{https://doi.org/10.1098/rsif.2010.0116}
  \item Haase, G. and Landelius, T., 2004. Dealiasing of Doppler radar
  velocities using a torus mapping. Journal of Atmospheric and Oceanic
  Technology, 21(10), pp.1566--1573.
  \url{https://doi.org/10.1175/1520-0426(2004)021<1566:DODRVU>2.0.CO;2}
  \item Tsung-Yu Lin, Kevin Winner, Garrett Bernstein, Abhay Mittal, Adriaan M. Dokter
  Kyle G. Horton, Cecilia Nilsson, Benjamin M. Van Doren, Andrew Farnsworth
  Frank A. La Sorte, Subhransu Maji, Daniel Sheldon, 2019.
  MistNet: Measuring historical bird migration in the US
  using archived weather radar data and convolutional neural networks
  Methods in Ecology and Evolution 10 (11), pp. 1908--22.
  \url{https://doi.org/10.1111/2041-210X.13280}
}
}
