% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classification.R
\name{HB.ROC}
\alias{HB.ROC}
\title{ROC curves for the Hanson and Brennan approach.}
\usage{
HB.ROC(
  x = NULL,
  reliability,
  testlength,
  truecut,
  true.model = "4P",
  failsafe = TRUE,
  l = 0,
  u = 1,
  AUC = FALSE,
  maxJ = FALSE,
  maxAcc = FALSE,
  locate = NULL,
  raw.out = FALSE,
  grainsize = testlength
)
}
\arguments{
\item{x}{A vector of observed results (sum scores) or a list of parameter values (see documentation for the \code{HB.beta.tp.fit() function}.}

\item{reliability}{The reliability coefficient of the test.}

\item{testlength}{The total number of test items (or maximum possible score). Must be an integer.}

\item{truecut}{The point along the x-scale that marks true category membership.}

\item{true.model}{The probability distribution to be fitted to the moments of the true-score distribution. Options are \code{"4P"} (default) and \code{"2P"}, referring to four- and two-parameter Beta distributions. The \code{"4P"} method produces a four-parameter Beta distribution with the same first four moments (mean, variance, skewness, and kurtosis) as the estimated true-score distribution, while the \code{"2P"} method produces a two-parameter Beta distribution with the first two moments (mean and variance) as the estimated true-score distribution.}

\item{failsafe}{If true-model == "4P": Whether to engage a fail-safe reverting to a two-parameter true-score distribution solution should the four-parameter fitting procedure produce impermissible results. Default is TRUE (engage fail-safe in the event of impermissible estimates).}

\item{l}{If \code{true.model == "2P"} or \code{failsafe == TRUE}: The lower-bound location parameter of the two-parameter true-score distribution solution.}

\item{u}{If \code{true.model == "2P"} or \code{failsafe == TRUE}: The upper-bound location parameter of the two-parameter true-score distribution solution.}

\item{AUC}{Logical. Calculate and include the area under the curve? Default is \code{FALSE}.}

\item{maxJ}{Logical. Mark the point along the curve where Youden's J statistic is maximized? Default is \code{FALSE}.}

\item{maxAcc}{Logical. Mark the point along the curve where the Accuracy statistic is maximized? Default is \code{FALSE}.}

\item{locate}{Ask the function to locate the cut-point at which sensitivity or NPV is greater than or equal to some value, or specificity or PPV is lesser than or equal to some value. Take as input a character-vector of length 2, with the first argument being which index is to be found (e.g., "sensitivity"), and the second argument the value to locate (e.g., "0.75"). For example: c("sensitivity", "0.75").}

\item{raw.out}{Give raw coordinates as output rather than plot? Default is \code{FALSE}}

\item{grainsize}{Specify the number of cutoff-points for which the ROC curve is to be calculated. The greater this number the greater the accuracy. Default is set to the stated test length (N).}
}
\value{
A plot tracing the ROC curve for the test, or matrix of coordinates if raw.out is \code{TRUE}.
}
\description{
Generate a ROC curve plotting the false-positive rate against the true-positive rate at different cut-off values across the observed-score scale.
}
\note{
This implementation of the Hanson-Brennan approach is much slower than the implementation of the Livingston and Lewis approach, as there is no native implementation of Lord's two-term approximation to the Compound-Binomial distribution in R. This implementation uses a "brute-force" method of computing the cumulative probabilities from the compound-Binomial distribution, which will by necessity be more resource intensive.
}
\examples{
# Generate some fictional data. Say, 1000 individuals take a test with a
# maximum score of 50.
# Generate some fictional data. Say, 1000 individuals take a 20-item test.
set.seed(1234)
p.success <- rBeta.4P(1000, 0.15, 0.85, 6, 4)
 for (i in 1:20) {
   if (i == 1) {
     rawdata <- matrix(nrow = 1000, ncol = 20)
     }
   rawdata[, i] <- rbinom(1000, 1, p.success)
 }

# Suppose the cutoff value for attaining a pass is 10 items correct, and
# that the reliability of this test was estimated using the Cronbach's Alpha
# estimator. To draw the ROC-graph and locate the points at which Youden's J
# and Accuracy are maximized:
HB.ROC(rowSums(rawdata), cba(rawdata), 20, 10, maxAcc = TRUE, maxJ = TRUE)

# For further examples regarding how to use the locate argument to locate
# points at which various criteria are satisfied, see documentation for the
# LL.ROC() function.
}
