% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geofacet-plot.R
\name{geofacet_plot}
\alias{geofacet_plot}
\title{Generate a geofacet plot of population trajectories by province/state}
\usage{
geofacet_plot(
  indices_list = NULL,
  select = FALSE,
  stratify_by = NULL,
  ci_width = 0.95,
  multiple = FALSE,
  trends = NULL,
  slope = FALSE,
  add_observed_means = FALSE,
  species = "",
  col_viridis = FALSE
)
}
\arguments{
\item{indices_list}{Dataframe of strata or state/province indices produced by
\code{generate_strata_indices} or \code{generate_regional_indices}}

\item{select}{logical flag to indicate if the strata_level data need to be selected out of an indices_list object that includes stratum, national, or other region-types. Default is FALSE}

\item{stratify_by}{How were the data stratified?}

\item{ci_width}{quantile to define the width of the plotted credible interval. Defaults to 0.95, lower = 0.025 and upper = 0.975}

\item{multiple}{Logical, if TRUE, multiple strata-level trajectories are plotted within each prov/state facet}

\item{trends}{Optional dataframe of matching strata or state/province trends produced by
\code{generate_strata_trends} or \code{generate_regional_trends}. If included trajectories are coloured based on the same
colour scale used in \code{generate_map}}

\item{slope}{Optional Logical, if dataframe of trends is included, colours in the plot are based on slope trends, Default = FALSE}

\item{add_observed_means}{Should the facet plots include points indicating the observed mean counts. Defaults to FALSE.  Note: scale of observed means and annual indices may not match due to imbalanced sampling among strata}

\item{species}{Species name to be added onto the plot}

\item{col_viridis}{Logical flag to use "viridis" colour-blind friendly palette. Default is FALSE}
}
\value{
ggplot object
}
\description{
\code{geofacet_plot} allows you to generate a faceted plot of population trajectories
  for each strata by province/state. Given a model stratified by "state", "bbs_cws", or "bbs_usgs"
  and indices generated by \code{generate_strata_indices} or \code{generate_regional_indices}, this
  function will generate a faceted plot showing the population trajectories. All geofacet plots have
  one facet per state/province, so if strata-level indices from the "bbs_cws" or "bbs_usgs" are given,
  the function plots multiple trajectories (one for each of the relevant strata) within each facet.
}
\examples{

# Toy example with Pacific Wren sample data
# First, stratify the sample data

strat_data <- stratify(by = "bbs_cws", sample_data = TRUE)

# Prepare the stratified data for use in a JAGS model.
jags_data <- prepare_jags_data(strat_data = strat_data,
                               species_to_run = "Pacific Wren",
                               model = "firstdiff",
                               min_year = 2009,
                               max_year = 2018)

# Now run a JAGS model.
jags_mod <- run_model(jags_data = jags_data,
                      n_adapt = 0,
                      n_burnin = 0,
                      n_iter = 10,
                      n_thin = 1)

# Generate the  stratum indices
indices <- generate_indices(jags_mod = jags_mod,
                            jags_data = jags_data,
                            regions = c("stratum"))

# Now make the geofacet plot.
gp <- geofacet_plot(indices_list = indices,
                    stratify_by = "bbs_cws",
                    species = "Pacific Wren",
                    multiple = TRUE)

# There is an unfortunate conflict between geofacet function in the geofacet package
# and the S3 +.gg method in other ggplot-extension-packages like ggmcmc
# The geofacet_plot function may fail with the following error message:
#  Error: Don't know how to add e2 to a plot
# If this happens, you can fix the problem by following these steps
#   1 - save your model output
#   2 - restart your R-session
#   3 - reload the bbsBayes package (do not re-load the other conflicting package, e.g., ggmcmc)

}
