% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/describe_posterior.R
\name{describe_posterior}
\alias{describe_posterior}
\alias{describe_posterior.numeric}
\alias{describe_posterior.data.frame}
\alias{describe_posterior.stanreg}
\alias{describe_posterior.brmsfit}
\title{Describe Posterior Distributions}
\usage{
describe_posterior(posterior, ...)

\method{describe_posterior}{numeric}(
  posterior,
  centrality = "median",
  dispersion = FALSE,
  ci = 0.95,
  ci_method = "eti",
  test = c("p_direction", "rope"),
  rope_range = "default",
  rope_ci = 0.95,
  keep_iterations = FALSE,
  bf_prior = NULL,
  BF = 1,
  verbose = TRUE,
  ...
)

\method{describe_posterior}{data.frame}(
  posterior,
  centrality = "median",
  dispersion = FALSE,
  ci = 0.95,
  ci_method = "eti",
  test = c("p_direction", "rope"),
  rope_range = "default",
  rope_ci = 0.95,
  keep_iterations = FALSE,
  bf_prior = NULL,
  BF = 1,
  rvar_col = NULL,
  verbose = TRUE,
  ...
)

\method{describe_posterior}{stanreg}(
  posterior,
  centrality = "median",
  dispersion = FALSE,
  ci = 0.95,
  ci_method = "eti",
  test = c("p_direction", "rope"),
  rope_range = "default",
  rope_ci = 0.95,
  keep_iterations = FALSE,
  bf_prior = NULL,
  diagnostic = c("ESS", "Rhat"),
  priors = FALSE,
  effects = c("fixed", "random", "all"),
  component = c("location", "all", "conditional", "smooth_terms", "sigma",
    "distributional", "auxiliary"),
  parameters = NULL,
  BF = 1,
  verbose = TRUE,
  ...
)

\method{describe_posterior}{brmsfit}(
  posterior,
  centrality = "median",
  dispersion = FALSE,
  ci = 0.95,
  ci_method = "eti",
  test = c("p_direction", "rope"),
  rope_range = "default",
  rope_ci = 0.95,
  keep_iterations = FALSE,
  bf_prior = NULL,
  diagnostic = c("ESS", "Rhat"),
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all", "location",
    "distributional", "auxiliary"),
  parameters = NULL,
  BF = 1,
  priors = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{posterior}{A vector, data frame or model of posterior draws.
\strong{bayestestR} supports a wide range of models (see \code{methods("describe_posterior")})
and not all of those are documented in the 'Usage' section, because methods
for other classes mostly resemble the arguments of the \code{.numeric} method.}

\item{...}{Additional arguments to be passed to or from methods.}

\item{centrality}{The point-estimates (centrality indices) to compute. Character
(vector) or list with one or more of these options: \code{"median"}, \code{"mean"}, \code{"MAP"}
(see \code{\link[=map_estimate]{map_estimate()}}), \code{"trimmed"} (which is just \code{mean(x, trim = threshold)}),
\code{"mode"} or \code{"all"}.}

\item{dispersion}{Logical, if \code{TRUE}, computes indices of dispersion related
to the estimate(s) (\code{SD} and \code{MAD} for \code{mean} and \code{median}, respectively).
Dispersion is not available for \code{"MAP"} or \code{"mode"} centrality indices.}

\item{ci}{Value or vector of probability of the CI (between 0 and 1)
to be estimated. Default to \code{0.95} (\verb{95\%}).}

\item{ci_method}{The type of index used for Credible Interval. Can be \code{"ETI"}
(default, see \code{\link[=eti]{eti()}}), \code{"HDI"} (see \code{\link[=hdi]{hdi()}}), \code{"BCI"} (see \code{\link[=bci]{bci()}}),
\code{"SPI"} (see \code{\link[=spi]{spi()}}), or \code{"SI"} (see \code{\link[=si]{si()}}).}

\item{test}{The indices of effect existence to compute. Character (vector) or
list with one or more of these options: \code{"p_direction"} (or \code{"pd"}),
\code{"rope"}, \code{"p_map"}, \code{"equivalence_test"} (or \code{"equitest"}),
\code{"bayesfactor"} (or \code{"bf"}) or \code{"all"} to compute all tests. For each
"test", the corresponding \pkg{bayestestR} function is called (e.g.
\code{\link[=rope]{rope()}} or \code{\link[=p_direction]{p_direction()}}) and its results included in the summary
output.}

\item{rope_range}{ROPE's lower and higher bounds. Should be a vector of two
values (e.g., \code{c(-0.1, 0.1)}), \code{"default"} or a list of numeric vectors of
the same length as numbers of parameters. If \code{"default"}, the bounds are
set to \code{x +- 0.1*SD(response)}.}

\item{rope_ci}{The Credible Interval (CI) probability, corresponding to the
proportion of HDI, to use for the percentage in ROPE.}

\item{keep_iterations}{If \code{TRUE}, will keep all iterations (draws) of
bootstrapped or Bayesian models. They will be added as additional columns
named \verb{iter_1, iter_2, ...}. You can reshape them to a long format by
running \code{\link[=reshape_iterations]{reshape_iterations()}}.}

\item{bf_prior}{Distribution representing a prior for the computation of
Bayes factors / SI. Used if the input is a posterior, otherwise (in the
case of models) ignored.}

\item{BF}{The amount of support required to be included in the support interval.}

\item{verbose}{Toggle off warnings.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{diagnostic}{Diagnostic metrics to compute.  Character (vector) or list
with one or more of these options: \code{"ESS"}, \code{"Rhat"}, \code{"MCSE"} or \code{"all"}.}

\item{priors}{Add the prior used for each parameter.}

\item{effects}{Should results for fixed effects, random effects or both be
returned? Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the
conditional model or the zero-inflated part of the model be returned? May
be abbreviated. Only applies to \pkg{brms}-models.}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\description{
Compute indices relevant to describe and characterize the posterior distributions.
}
\details{
One or more components of point estimates (like posterior mean or median),
intervals and tests can be omitted from the summary output by setting the
related argument to \code{NULL}. For example, \code{test = NULL} and \code{centrality = NULL} would only return the HDI (or CI).
}
\examples{
library(bayestestR)

if (require("logspline")) {
  x <- rnorm(1000)
  describe_posterior(x, verbose = FALSE)
  describe_posterior(x,
    centrality = "all",
    dispersion = TRUE,
    test = "all",
    verbose = FALSE
  )
  describe_posterior(x, ci = c(0.80, 0.90), verbose = FALSE)

  df <- data.frame(replicate(4, rnorm(100)))
  describe_posterior(df, verbose = FALSE)
  describe_posterior(
    df,
    centrality = "all",
    dispersion = TRUE,
    test = "all",
    verbose = FALSE
  )
  describe_posterior(df, ci = c(0.80, 0.90), verbose = FALSE)

  df <- data.frame(replicate(4, rnorm(20)))
  head(reshape_iterations(
    describe_posterior(df, keep_iterations = TRUE, verbose = FALSE)
  ))
}
\donttest{
# rstanarm models
# -----------------------------------------------
if (require("rstanarm") && require("emmeans")) {
  model <- suppressWarnings(
    stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
  )
  describe_posterior(model)
  describe_posterior(model, centrality = "all", dispersion = TRUE, test = "all")
  describe_posterior(model, ci = c(0.80, 0.90))
  describe_posterior(model, rope_range = list(c(-10, 5), c(-0.2, 0.2), "default"))

  # emmeans estimates
  # -----------------------------------------------
  describe_posterior(emtrends(model, ~1, "wt"))
}

# BayesFactor objects
# -----------------------------------------------
if (require("BayesFactor")) {
  bf <- ttestBF(x = rnorm(100, 1, 1))
  describe_posterior(bf)
  describe_posterior(bf, centrality = "all", dispersion = TRUE, test = "all")
  describe_posterior(bf, ci = c(0.80, 0.90))
}
}
}
\references{
\itemize{
\item Makowski, D., Ben-Shachar, M. S., Chen, S. H. A., and Lüdecke, D. (2019).
\emph{Indices of Effect Existence and Significance in the Bayesian Framework}.
Frontiers in Psychology 2019;10:2767. \doi{10.3389/fpsyg.2019.02767}
\item \href{https://easystats.github.io/bayestestR/articles/region_of_practical_equivalence.html}{Region of Practical Equivalence (ROPE)}
\item \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{Bayes factors}
}
}
