\name{rmnpGibbs}
\alias{rmnpGibbs}
\concept{bayes}
\concept{multinomial probit}
\concept{MCMC}
\concept{Gibbs Sampling}

\title{ Gibbs Sampler for Multinomial Probit }
\description{
  \code{rmnpGibbs} implements the McCulloch/Rossi Gibbs Sampler for the multinomial probit model.
}

\usage{
rmnpGibbs(Data, Prior, Mcmc)
}

\arguments{
  \item{Data}{ list(p, y, X)}
  \item{Prior}{ list(betabar,A,nu,V) (optional)}
  \item{Mcmc}{ list(beta0,sigma0,R,keep) (R required) }
}

\details{
  model:  \cr
    \eqn{w_i = X_i\beta + e}.    \eqn{e} \eqn{\sim}{~} \eqn{N(0,Sigma)}.     note: \eqn{w_i, e} are (p-1) x 1.\cr
    \eqn{y_i = j},  if \eqn{w_{ij} > max(0,w_{i,-j})}  j=1,\ldots,p-1.  \eqn{w_{i,-j}} means elements of \eqn{w_i}
     other than the jth. \cr
    \eqn{y_i = p},  if all \eqn{w_i < 0}.\cr
  
  priors:\cr
    \eqn{beta} \eqn{\sim}{~} \eqn{N(betabar,A^{-1})} \cr
    \eqn{Sigma} \eqn{\sim}{~} IW(nu,V)\cr

  to make up X matrix use \code{\link{createX}} with \code{DIFF=TRUE}.

  List arguments contain  
  \itemize{
    \item{\code{p}}{number of choices or possible multinomial outcomes}
    \item{\code{y}}{n x 1 vector of multinomial outcomes}
    \item{\code{X}}{n*(p-1) x k Design Matrix}
    \item{\code{betabar}}{k x 1 prior mean (def: 0)}
    \item{\code{A}}{k x k prior precision matrix (def: .01I)} 
    \item{\code{nu}}{ d.f. parm for IWishart prior (def: (p-1) + 3)}
    \item{\code{V}}{ pds location parm for IWishart prior (def: nu*I)}
    \item{\code{beta0}}{ initial value for beta}
    \item{\code{sigma0}}{ initial value for sigma }
    \item{\code{R}}{ number of MCMC draws }
    \item{\code{keep}}{ thinning parameter - keep every keepth draw (def: 1)}
  }
}

\value{
  a list containing: 
  \item{betadraw }{R/keep x k array of betadraws}
  \item{sigmadraw}{R/keep x (p-1)*(p-1) array of sigma draws -- each row is in vector form}
}
\note{
  beta is not identified.  beta/sqrt(\eqn{sigma_{11}}) and Sigma/\eqn{sigma_{11}} are.  See Allenby et al or
  example below for details.
}

\references{ For further discussion, see \emph{Bayesian Statistics and Marketing}
  by Allenby, McCulloch, and Rossi, Chapter 4. \cr
  \url{http://gsbwww.uchicago.edu/fac/peter.rossi/research/bsm.html}
}

\author{ Peter Rossi, Graduate School of Business, University of Chicago,
  \email{Peter.Rossi@ChicagoGsb.edu}.
}

\seealso{ \code{\link{rmvpGibbs}} }
\examples{
##

set.seed(66)
p=3
n=500
beta=c(-1,1,1,2)
Sigma=matrix(c(1,.5,.5,1),ncol=2)
k=length(beta)
x1=runif(n*(p-1),min=-1,max=1); x2=runif(n*(p-1),min=-1,max=1)
I2=diag(rep(1,p-1)); xadd=rbind(I2)
for(i in 2:n) { xadd=rbind(xadd,I2)}
X=cbind(xadd,x1,x2)
simout=simmnp(X,p,500,beta,Sigma)

R=2000
Data=list(p=p,y=simout$y,X=simout$X)
Mcmc=list(R=R,keep=1)

out=rmnpGibbs(Mcmc=Mcmc,Data=Data)

cat(" Betadraws ",fill=TRUE)
mat=apply(out$betadraw/sqrt(out$sigmadraw[,1]),2,quantile,probs=c(.01,.05,.5,.95,.99))
mat=rbind(beta,mat); rownames(mat)[1]="beta"; print(mat)
cat(" Sigmadraws ",fill=TRUE)
mat=apply(out$sigmadraw/out$sigmadraw[,1],2,quantile,probs=c(.01,.05,.5,.95,.99))
mat=rbind(as.vector(Sigma),mat); rownames(mat)[1]="sigma"; print(mat)

}
\keyword{ models }
