% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MASI.AMMI.R
\name{MASI.AMMI}
\alias{MASI.AMMI}
\title{Modified AMMI Stability Index}
\usage{
MASI.AMMI(model, n, alpha = 0.05, ssi.method = c("farshadfar", "rao"),
  a = 1)
}
\arguments{
\item{model}{The AMMI model (An object of class \code{AMMI} generated by
\code{\link[agricolae]{AMMI}}).}

\item{n}{The number of principal components to be considered for computation.
The default value is the number of significant IPCs.}

\item{alpha}{Type I error probability (Significance level) to be considered
to identify the number of significant IPCs.}

\item{ssi.method}{The method for the computation of simultaneous selection
index. Either \code{"farshadfar"} or \code{"rao"} (See
\code{\link[ammistability]{SSI}}).}

\item{a}{The ratio of the weights given to the stability components for
computation of SSI when \code{method = "rao"} (See
\code{\link[ammistability]{SSI}}).}
}
\value{
A data frame with the following columns:  \item{MASI}{The MASI
  values.} \item{SSI}{The computed values of simultaneous selection index for
  yield and stability.} \item{rMASI}{The ranks of MASI values.} \item{rY}{The
  ranks of the mean yield of genotypes.} \item{means}{The mean yield of the
  genotypes.}

  The names of the genotypes are indicated as the row names of the data
  frame.
}
\description{
\code{MASI.AMMI} computes the Modified AMMI Stability Index (MASI) from a
modified formula of AMMI Stability Index (ASI) (Jambhulkar et al., 2014;
Jambhulkar et al., 2015; Jambhulkar et al., 2017).  Unlike ASI, MASI
calculates stability value considering all significant interaction principal
components (IPCs) in the AMMI model. Using MASI, the Simultaneous Selection
Index for Yield and Stability (SSI) is also calculated according to the
argument \code{ssi.method}.
}
\details{
The Modified AMMI Stability Index
(\ifelse{html}{\out{<i>MASI</i>}}{\eqn{MASI}}) is computed as follows:

\ifelse{html}{\out{<p style="text-align: center;"><i>MASI = <big>&radic;[
&sum;</big><sup>N';</sup><sub style="line-height: 1.8; margin-left:
-4ex;">n=1</sub> PC<sup>2</sup><sub style="line-height: 1.8; margin-left:
-1ex;">n</sub> &times; &theta;<sup>2</sup><sub style="line-height: 1.8;
margin-left: -1ex;">n</sub><big>]</big></i></p>}}{\deqn{MASI = \sqrt{
\sum_{n=1}^{N'} PC_{n}^{2} \times \theta_{n}^{2}}}}

Where, \ifelse{html}{\out{<i>PC<sub>n</sub></i>}}{\eqn{PC_{n}}} are the
scores of \ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th IPC; and
\ifelse{html}{\out{<i>&theta;<sub>n</sub></i>}}{\eqn{\theta_{n}}} is the
percentage sum of squares explained by the
\ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th principal component interaction
effect.
}
\examples{
library(agricolae)
data(plrv)

# AMMI model
model <- with(plrv, AMMI(Locality, Genotype, Rep, Yield, console = FALSE))

# ANOVA
model$ANOVA

# IPC F test
model$analysis

# Mean yield and IPC scores
model$biplot

# G*E matrix (deviations from mean)
array(model$genXenv, dim(model$genXenv), dimnames(model$genXenv))

# With default n (N') and default ssi.method (farshadfar)
MASI.AMMI(model)

# With n = 4 and default ssi.method (farshadfar)
MASI.AMMI(model, n = 4)

# With default n (N') and ssi.method = "rao"
MASI.AMMI(model, ssi.method = "rao")

# Changing the ratio of weights for Rao's SSI
MASI.AMMI(model, ssi.method = "rao", a = 0.43)

# ASI.AMMI same as MASI.AMMI with n = 2

a <- ASI.AMMI(model)
b <- MASI.AMMI(model, n = 2)

identical(a$ASI, b$MASI)

}
\references{
\insertRef{jambhulkar_ammi_2014}{ammistability}

\insertRef{jambhulkar_genotype_2015}{ammistability}

\insertRef{jambhulkar_stability_2017}{ammistability}
}
\seealso{
\code{\link[agricolae]{AMMI}},
  \code{\link[ammistability]{ASI.AMMI}}, \code{\link[ammistability]{SSI}}
}
