% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Diversity.R
\name{alphaDiversity}
\alias{alphaDiversity}
\title{Calculate clonal alpha diversity}
\usage{
alphaDiversity(data, min_q = 0, max_q = 4, step_q = 0.1, ci = 0.95, ...)
}
\arguments{
\item{data}{data.frame with Change-O style columns containing clonal assignments or
a \link{AbundanceCurve} generate by \link{estimateAbundance} object 
containing a previously calculated bootstrap distributions of clonal abundance.}

\item{min_q}{minimum value of \eqn{q}.}

\item{max_q}{maximum value of \eqn{q}.}

\item{step_q}{value by which to increment \eqn{q}.}

\item{ci}{confidence interval to calculate; the value must be between 0 and 1.}

\item{...}{additional arguments to pass to \link{estimateAbundance}. Additional arguments
are ignored if a \link{AbundanceCurve} is provided as input.}
}
\value{
A \link{DiversityCurve} object summarizing the diversity scores.
}
\description{
\code{alphaDiversity} takes in a data.frame or \link{AbundanceCurve} and computes
diversity scores (\eqn{D}) over an interval of diversity orders (\eqn{q}).
}
\details{
Clonal diversity is calculated using the generalized diversity index (Hill numbers) 
proposed by Hill (Hill, 1973). See \link{calcDiversity} for further details.

Diversity is calculated on the estimated complete clonal abundance distribution.
This distribution is inferred by using the Chao1 estimator to estimate the number
of seen clones, and applying the relative abundance correction and unseen clone
frequency described in Chao et al, 2015.

To generate a smooth curve, \eqn{D} is calculated for each value of \eqn{q} from
\code{min_q} to \code{max_q} incremented by \code{step_q}.  When \code{uniform=TRUE}
variability in total sequence counts across unique values in the \code{group} column 
is corrected by repeated resampling from the estimated complete clonal distribution to a 
common number of sequences.

The diversity index (\eqn{D}) for each group is the mean value of over all resampling 
realizations. Confidence intervals are derived using the standard deviation of the 
resampling realizations, as described in Chao et al, 2015.

Of note, the complete clonal abundance distribution is inferred by using the Chao1 
estimator to estimate the number of seen clones, and then applying the relative abundance 
correction and unseen clone frequencies described in Chao et al, 2015.
}
\examples{
# Group by sample identifier in two steps
abund <- estimateAbundance(ExampleDb, group="sample_id", nboot=100)
div <- alphaDiversity(abund, step_q=1, max_q=10)
plotDiversityCurve(div, legend_title="Sample")
                   
# Grouping by isotype rather than sample identifier in one step
div <- alphaDiversity(ExampleDb, group="c_call", min_n=40, step_q=1, max_q=10, 
                      nboot=100)
plotDiversityCurve(div, legend_title="Isotype")

}
\references{
\enumerate{
  \item  Hill M. Diversity and evenness: a unifying notation and its consequences. 
           Ecology. 1973 54(2):427-32.
  \item  Chao A. Nonparametric Estimation of the Number of Classes in a Population. 
           Scand J Stat. 1984 11, 265270.
  \item  Chao A, et al. Rarefaction and extrapolation with Hill numbers: 
           A framework for sampling and estimation in species diversity studies. 
           Ecol Monogr. 2014 84:45-67.
  \item  Chao A, et al. Unveiling the species-rank abundance distribution by 
           generalizing the Good-Turing sample coverage theory. 
           Ecology. 2015 96, 11891201.
}
}
\seealso{
See \link{calcDiversity} for the basic calculation and 
          \link{DiversityCurve} for the return object. 
          See \link{plotDiversityCurve} for plotting the return object.
}
