% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_advs_params.R
\name{derive_param_waisthgt}
\alias{derive_param_waisthgt}
\title{Adds a Parameter for Waist to Height Ratio}
\usage{
derive_param_waisthgt(
  dataset,
  by_vars,
  wstcir_code = "WSTCIR",
  height_code = "HEIGHT",
  set_values_to = exprs(PARAMCD = "WAISTHGT"),
  filter = NULL,
  constant_by_vars = NULL,
  get_unit_expr
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} argument are expected to be in the dataset.
\code{PARAMCD}, and \code{AVAL} are expected as well.

The variable specified by \code{by_vars} and \code{PARAMCD} must be a unique key of the input dataset
after restricting it by the filter condition (\code{filter} argument) and to the parameters
specified by \code{wstcir_code} and \code{height_code}.}

\item{by_vars}{Grouping variables

For each group defined by \code{by_vars} an observation is added to the output
dataset. Only variables specified in \code{by_vars} will be populated
in the newly created records.

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{wstcir_code}{Waist Circumference parameter code

The observations where \code{PARAMCD} equals the specified value are considered
as the Waist Circumference.

\emph{Permitted Values:} character value}

\item{height_code}{Height parameter code

The observations where \code{PARAMCD} equals the specified value are considered as the Height.

\emph{Permitted Values:} character value}

\item{set_values_to}{Variables to be set

The specified variables are set to the specified values for the new
observations. For example \code{exprs(PARAMCD = "RATIO")} defines the parameter code
for the new parameter.

\emph{Permitted Values:} List of variable-value pairs}

\item{filter}{Filter condition

The specified condition is applied to the input dataset before deriving the
new parameter, i.e., only observations fulfilling the condition are taken
into account.

\emph{Permitted Values:} a condition}

\item{constant_by_vars}{By variables for when Height is constant

When Height is constant, the Height parameters (measured only once) are merged
to the other parameters using the specified variables.

If Height is constant (e.g. only measured once at screening or baseline) then use
\code{constant_by_vars} to select the subject-level variable to merge on (e.g. \code{USUBJID}).
This will produce Waist to Height Ratio at all visits where Waist Circumference is measured.
Otherwise it will only be calculated at visits with both Height and Waist Circumference
collected.

\emph{Permitted Values}: list of variables created by \code{exprs()}, e.g. \code{exprs(USUBJID, VISIT)}}

\item{get_unit_expr}{An expression providing the unit of the parameter

The result is used to check the units of the input parameters. If the units are not consistent
within each parameter, an error will be thrown.

Additionally, if the input parameters are measured in different units but are mutually
convertible (e.g., centimeters for one parameter and inches for another), an automatic
conversion will be performed in order to uniform the values before calculating the ratio.

\strong{Note:} Conversion factors come from unit definitions as per CDISC standards.
\if{html}{\out{<br/>}}\emph{m} is defined as 100 cm
\if{html}{\out{<br/>}}\emph{mm} is defined as 0.1 cm
\if{html}{\out{<br/>}}\emph{in} is defined as 2.54 cm
\if{html}{\out{<br/>}}\emph{ft} is defined as 30.48 cm

\emph{Permitted Values:} A variable of the input dataset or a function call}
}
\value{
The input dataset with the new parameter added. Note, a variable will only
be populated in the new parameter rows if it is specified in \code{by_vars}.
}
\description{
Adds a parameter for Waist to Height Ratio using Waist Circumference and Height
for each by group (e.g., subject and visit) where the source parameters are available.

\strong{Note:} This is a wrapper function for the more generic \code{\link[admiral:derive_param_computed]{admiral::derive_param_computed()}}.
}
\details{
The analysis value of the new parameter is derived as
\deqn{WAISTHGT = \frac{WSTCIR}{HEIGHT}}{WAISTHGT = WSTCIR / HEIGHT}
}
\examples{
library(tibble)
library(rlang)

# Example 1: Derive Waist to Height Ratio where Height is measured only once

advs <- tribble(
  ~USUBJID, ~PARAMCD, ~PARAM, ~AVAL, ~AVALU, ~VISIT,
  "01-101-1001", "HEIGHT", "Height (cm)", 147, "cm", "SCREENING",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 110, "cm", "SCREENING",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 108, "cm", "WEEK 2",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 107, "cm", "WEEK 3",
  "01-101-1002", "HEIGHT", "Height (cm)", 163, "cm", "SCREENING",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 120, "cm", "SCREENING",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 118, "cm", "WEEK 2",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 117, "cm", "WEEK 3",
)

derive_param_waisthgt(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  wstcir_code = "WSTCIR",
  height_code = "HEIGHT",
  set_values_to = exprs(
    PARAMCD = "WAISTHGT",
    PARAM = "Waist to Height Ratio"
  ),
  constant_by_vars = exprs(USUBJID),
  get_unit_expr = admiral::extract_unit(PARAM)
)

# Example 2: Same as above but only adding Waist to Height Ratio
# at certain visits

derive_param_waisthgt(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  wstcir_code = "WSTCIR",
  height_code = "HEIGHT",
  set_values_to = exprs(
    PARAMCD = "WAISTHGT",
    PARAM = "Waist to Height Ratio"
  ),
  constant_by_vars = exprs(USUBJID),
  get_unit_expr = admiral::extract_unit(PARAM),
  filter = VISIT \%in\% c("SCREENING", "WEEK 3")
)

# Example 3: Pediatric study where Height and Waist Circumference
# are measured multiple times

advs <- tribble(
  ~USUBJID, ~PARAMCD, ~PARAM, ~AVAL, ~AVALU, ~VISIT,
  "01-101-1001", "HEIGHT", "Height (cm)", 147, "cm", "SCREENING",
  "01-101-1001", "HEIGHT", "Height (cm)", 148, "cm", "WEEK 2",
  "01-101-1001", "HEIGHT", "Height (cm)", 149, "cm", "WEEK 3",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 100, "cm", "SCREENING",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 99, "cm", "WEEK 2",
  "01-101-1001", "WSTCIR", "Waist Circumference (cm)", 98, "cm", "WEEK 3",
  "01-101-1002", "HEIGHT", "Height (cm)", 163, "cm", "SCREENING",
  "01-101-1002", "HEIGHT", "Height (cm)", 164, "cm", "WEEK 2",
  "01-101-1002", "HEIGHT", "Height (cm)", 165, "cm", "WEEK 3",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 110, "cm", "SCREENING",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 109, "cm", "WEEK 2",
  "01-101-1002", "WSTCIR", "Waist Circumference (cm)", 108, "cm", "WEEK 3"
)

derive_param_waisthgt(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  wstcir_code = "WSTCIR",
  height_code = "HEIGHT",
  set_values_to = exprs(
    PARAMCD = "WAISTHGT",
    PARAM = "Waist to Height Ratio"
  ),
  get_unit_expr = admiral::extract_unit(PARAM)
)

# Example 4: Automatic conversion is performed when deriving the ratio
# if parameters are provided in different units (e.g. centimeters and inches)

advs <- tribble(
  ~USUBJID, ~PARAMCD, ~PARAM, ~AVAL, ~AVALU, ~VISIT,
  "01-101-1001", "HEIGHT", "Height (cm)", 147, "cm", "SCREENING",
  "01-101-1001", "WSTCIR", "Waist Circumference (in)", 39.37, "in", "SCREENING",
  "01-101-1001", "WSTCIR", "Waist Circumference (in)", 38.98, "in", "WEEK 2",
  "01-101-1001", "WSTCIR", "Waist Circumference (in)", 38.58, "in", "WEEK 3",
  "01-101-1002", "HEIGHT", "Height (cm)", 163, "cm", "SCREENING",
  "01-101-1002", "WSTCIR", "Waist Circumference (in)", 43.31, "in", "SCREENING",
  "01-101-1002", "WSTCIR", "Waist Circumference (in)", 42.91, "in", "WEEK 2",
  "01-101-1002", "WSTCIR", "Waist Circumference (in)", 42.52, "in", "WEEK 3"
)

derive_param_waisthgt(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  wstcir_code = "WSTCIR",
  height_code = "HEIGHT",
  set_values_to = exprs(
    PARAMCD = "WAISTHGT",
    PARAM = "Waist to Height Ratio"
  ),
  constant_by_vars = exprs(USUBJID),
  get_unit_expr = admiral::extract_unit(PARAM)
)
}
\seealso{
\code{\link[admiral:derive_param_computed]{admiral::derive_param_computed()}}

ADVS Functions for adding Parameters: 
\code{\link{derive_param_waisthip}()}
}
\concept{der_prm_advs}
\keyword{der_prm_advs}
