#' Country Code Lookup
#'
#' @description
#' These pre-defined country codes are sourced from
#' [ISO 3166 Standards](https://www.iso.org/iso-3166-country-codes.html).
#' See also [Wikipedia](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-3).
#'
#' @details
#'
#' `country_code` is the 3-letter ISO 3166-1 county code commonly found in the
#' ADSL `COUNTRY` variable.
#' `country_name` is the country long name corresponding to to the 3-letter code.
#' `country_number` is the numeric code corresponding to an alphabetic sorting of
#' the 3-letter codes.
#'
#' To see the entire table in the console, run `print(country_code_lookup)`.
#'
#' @seealso [dose_freq_lookup]
#'
#' @export
#'
#' @keywords metadata
#'
#' @family metadata
#'
#' @examples
#' library(tibble)
#' library(dplyr, warn.conflicts = FALSE)
#' library(lubridate)
#'
#' # Create reference dataset for periods
#' adsl <- tribble(
#'   ~USUBJID, ~SEX, ~COUNTRY,
#'   "ST01-01", "F", "AUT",
#'   "ST01-02", "M", "MWI",
#'   "ST01-03", "F", "GBR",
#'   "ST01-04", "M", "CHE",
#'   "ST01-05", "M", "NOR",
#'   "ST01-06", "F", "JPN",
#'   "ST01-07", "F", "USA"
#' )
#'
#' covar <- adsl %>%
#'   derive_vars_merged(
#'     dataset_add = country_code_lookup,
#'     new_vars = exprs(COUNTRYN = country_number, COUNTRYL = country_name),
#'     by_vars = exprs(COUNTRY = country_code)
#'   )
#' covar
#'
#' @rdname country_code_lookup
country_code_lookup <- tribble(
  ~country_code, ~country_name,
  "ABW", "Aruba",
  "AFG", "Afghanistan",
  "AGO", "Angola",
  "AIA", "Anguilla",
  "ALA", "Aland Islands",
  "ALB", "Albania",
  "AND", "Andorra",
  "ARE", "United Arab Emirates",
  "ARG", "Argentina",
  "ARM", "Armenia",
  "ASM", "American Samoa",
  "ATA", "Antarctica",
  "ATF", "French Southern Territories",
  "ATG", "Antigua and Barbuda",
  "AUS", "Australia",
  "AUT", "Austria",
  "AZE", "Azerbaijan",
  "BDI", "Burundi",
  "BEL", "Belgium",
  "BEN", "Benin",
  "BES", "Bonaire, Sint Eustatius and Saba",
  "BFA", "Burkina Faso",
  "BGD", "Bangladesh",
  "BGR", "Bulgaria",
  "BHR", "Bahrain",
  "BHS", "Bahamas",
  "BIH", "Bosnia and Herzegovina",
  "BLM", "Saint Barthelemy",
  "BLR", "Belarus",
  "BLZ", "Belize",
  "BMU", "Bermuda",
  "BOL", "Bolivia, Plurinational State of",
  "BRA", "Brazil",
  "BRB", "Barbados",
  "BRN", "Brunei Darussalam",
  "BTN", "Bhutan",
  "BVT", "Bouvet Island",
  "BWA", "Botswana",
  "CAF", "Central African Republic",
  "CAN", "Canada",
  "CCK", "Cocos (Keeling) Islands",
  "CHE", "Switzerland",
  "CHL", "Chile",
  "CHN", "China",
  "CIV", "Cote d'Ivoire",
  "CMR", "Cameroon",
  "COD", "Congo, Democratic Republic of the",
  "COG", "Congo",
  "COK", "Cook Islands",
  "COL", "Colombia",
  "COM", "Comoros",
  "CPV", "Cabo Verde",
  "CRI", "Costa Rica",
  "CUB", "Cuba",
  "CUW", "Curacao",
  "CXR", "Christmas Island",
  "CYM", "Cayman Islands",
  "CYP", "Cyprus",
  "CZE", "Czechia",
  "DEU", "Germany",
  "DJI", "Djibouti",
  "DMA", "Dominica",
  "DNK", "Denmark",
  "DOM", "Dominican Republic",
  "DZA", "Algeria",
  "ECU", "Ecuador",
  "EGY", "Egypt",
  "ERI", "Eritrea",
  "ESH", "Western Sahara",
  "ESP", "Spain",
  "EST", "Estonia",
  "ETH", "Ethiopia",
  "FIN", "Finland",
  "FJI", "Fiji",
  "FLK", "Falkland Islands (Malvinas)",
  "FRA", "France",
  "FRO", "Faroe Islands",
  "FSM", "Micronesia, Federated States of",
  "GAB", "Gabon",
  "GBR", "United Kingdom of Great Britain and Northern Ireland",
  "GEO", "Georgia",
  "GGY", "Guernsey",
  "GHA", "Ghana",
  "GIB", "Gibraltar",
  "GIN", "Guinea",
  "GLP", "Guadeloupe",
  "GMB", "Gambia",
  "GNB", "Guinea-Bissau",
  "GNQ", "Equatorial Guinea",
  "GRC", "Greece",
  "GRD", "Grenada",
  "GRL", "Greenland",
  "GTM", "Guatemala",
  "GUF", "French Guiana",
  "GUM", "Guam",
  "GUY", "Guyana",
  "HKG", "Hong Kong",
  "HMD", "Heard Island and McDonald Islands",
  "HND", "Honduras",
  "HRV", "Croatia",
  "HTI", "Haiti",
  "HUN", "Hungary",
  "IDN", "Indonesia",
  "IMN", "Isle of Man",
  "IND", "India",
  "IOT", "British Indian Ocean Territory",
  "IRL", "Ireland",
  "IRN", "Iran, Islamic Republic of",
  "IRQ", "Iraq",
  "ISL", "Iceland",
  "ISR", "Israel",
  "ITA", "Italy",
  "JAM", "Jamaica",
  "JEY", "Jersey",
  "JOR", "Jordan",
  "JPN", "Japan",
  "KAZ", "Kazakhstan",
  "KEN", "Kenya",
  "KGZ", "Kyrgyzstan",
  "KHM", "Cambodia",
  "KIR", "Kiribati",
  "KNA", "Saint Kitts and Nevis",
  "KOR", "Korea, Republic of",
  "KWT", "Kuwait",
  "LAO", "Lao People's Democratic Republic",
  "LBN", "Lebanon",
  "LBR", "Liberia",
  "LBY", "Libya",
  "LCA", "Saint Lucia",
  "LIE", "Liechtenstein",
  "LKA", "Sri Lanka",
  "LSO", "Lesotho",
  "LTU", "Lithuania",
  "LUX", "Luxembourg",
  "LVA", "Latvia",
  "MAC", "Macao",
  "MAF", "Saint Martin (French part)",
  "MAR", "Morocco",
  "MCO", "Monaco",
  "MDA", "Moldova, Republic of",
  "MDG", "Madagascar",
  "MDV", "Maldives",
  "MEX", "Mexico",
  "MHL", "Marshall Islands",
  "MKD", "North Macedonia",
  "MLI", "Mali",
  "MLT", "Malta",
  "MMR", "Myanmar",
  "MNE", "Montenegro",
  "MNG", "Mongolia",
  "MNP", "Northern Mariana Islands",
  "MOZ", "Mozambique",
  "MRT", "Mauritania",
  "MSR", "Montserrat",
  "MTQ", "Martinique",
  "MUS", "Mauritius",
  "MWI", "Malawi",
  "MYS", "Malaysia",
  "MYT", "Mayotte",
  "NAM", "Namibia",
  "NCL", "New Caledonia",
  "NER", "Niger",
  "NFK", "Norfolk Island",
  "NGA", "Nigeria",
  "NIC", "Nicaragua",
  "NIU", "Niue",
  "NLD", "Netherlands, Kingdom of the",
  "NOR", "Norway",
  "NPL", "Nepal",
  "NRU", "Nauru",
  "NZL", "New Zealand",
  "OMN", "Oman",
  "PAK", "Pakistan",
  "PAN", "Panama",
  "PCN", "Pitcairn",
  "PER", "Peru",
  "PHL", "Philippines",
  "PLW", "Palau",
  "PNG", "Papua New Guinea",
  "POL", "Poland",
  "PRI", "Puerto Rico",
  "PRK", "Korea, Democratic People's Republic of",
  "PRT", "Portugal",
  "PRY", "Paraguay",
  "PSE", "Palestine, State of",
  "PYF", "French Polynesia",
  "QAT", "Qatar",
  "REU", "Reunion",
  "ROU", "Romania",
  "RUS", "Russian Federation",
  "RWA", "Rwanda",
  "SAU", "Saudi Arabia",
  "SDN", "Sudan",
  "SEN", "Senegal",
  "SGP", "Singapore",
  "SGS", "South Georgia and the South Sandwich Islands",
  "SHN", "Saint Helena, Ascension and Tristan da Cunha",
  "SJM", "Svalbard and Jan Mayen",
  "SLB", "Solomon Islands",
  "SLE", "Sierra Leone",
  "SLV", "El Salvador",
  "SMR", "San Marino",
  "SOM", "Somalia",
  "SPM", "Saint Pierre and Miquelon",
  "SRB", "Serbia",
  "SSD", "South Sudan",
  "STP", "Sao Tome and Principe",
  "SUR", "Suriname",
  "SVK", "Slovakia",
  "SVN", "Slovenia",
  "SWE", "Sweden",
  "SWZ", "Eswatini",
  "SXM", "Sint Maarten (Dutch part)",
  "SYC", "Seychelles",
  "SYR", "Syrian Arab Republic",
  "TCA", "Turks and Caicos Islands",
  "TCD", "Chad",
  "TGO", "Togo",
  "THA", "Thailand",
  "TJK", "Tajikistan",
  "TKL", "Tokelau",
  "TKM", "Turkmenistan",
  "TLS", "Timor-Leste",
  "TON", "Tonga",
  "TTO", "Trinidad and Tobago",
  "TUN", "Tunisia",
  "TUR", "Turkey",
  "TUV", "Tuvalu",
  "TWN", "Taiwan",
  "TZA", "Tanzania, United Republic of",
  "UGA", "Uganda",
  "UKR", "Ukraine",
  "UMI", "United States Minor Outlying Islands",
  "URY", "Uruguay",
  "USA", "United States of America",
  "UZB", "Uzbekistan",
  "VAT", "Holy See",
  "VCT", "Saint Vincent and the Grenadines",
  "VEN", "Venezuela, Bolivarian Republic of",
  "VGB", "Virgin Islands (British)",
  "VIR", "Virgin Islands (U.S.)",
  "VNM", "Viet Nam",
  "VUT", "Vanuatu",
  "WLF", "Wallis and Futuna",
  "WSM", "Samoa",
  "YEM", "Yemen",
  "ZAF", "South Africa",
  "ZMB", "Zambia",
  "ZWE", "Zimbabwe",
) %>%
  arrange(country_code)

# Convert ISO 3166 alpha 3 country codes to numbers 1-249
country_code_lookup$country_number <- as.numeric(seq_len(nrow(country_code_lookup)))
