#' @title ExportVirusGt: (Experimental) Export Graphical Tables
#'
#' @description ExportVirusGt allows the user to export graphical tables in different formats.
#'
#'
#'
#'
#' @param gtable A graphical table object.
#' @param filename Name of the output file (default: table.docx"). Make sure to provide an extension
#' compatible with the output types: .html, .tex, .ltx, .rtf, .docx. If a custom save function is provided,
#' the file extension is ignored.
#' @param export_gt_obj (optional): If TRUE, exports the input data frame in .rds format with the same name as specified in filename (default: FALSE).
#' @param path Path of the directory to save plot to: path and filename are combined to create the
#' fully qualified file name (default: current corking directory).
#' @param create.dir Whether to create new directories if a non-existing directory is specified in
#' the filename or path (TRUE) or return an error (FALSE, default). If FALSE and run in an interactive session,
#' a prompt will appear asking to create a new directory when necessary.
#' @param ... Pass any other options to the corresponding internal saving function.
#'
#' @details
#'
#' Export graphical tables (gt) generated by functions within the Virusparies package.
#'
#' This feature is in an experimental phase and may not currently function as expected.
#'
#' The exportVirusGt function utilizes the gt package for table manipulation and formatting.
#'
#' For HTML output file names with .html or .htm extensions, an HTML document is generated using the gt package.
#' Pass TRUE or FALSE to inline_css to include or exclude CSS styles inline (default is FALSE).
#' Additional options can be passed through .... For RTF output file names with .rtf extension,
#' an RTF file is created. Use the page_numbering option to control page numbering (default is none).
#'
#' For image files, use .png for PNG and .pdf for PDF. The gt package relies on Google Chrome installation
#' for PNG and PDF images. Pass options to webshot2::webshot() through .... Useful PNG options include zoom
#' (default: 2) and expand (default: 5).
#'
#' For LaTeX output file names with .tex, .ltx, or .rnw extensions, and .rtf for RTF, the corresponding
#' documents are generated. No additional options available.
#'
#' For .docx output, requires rmarkdown package.
#'
#' When `create.dir` is set to TRUE, it generates a directory at the specified 'path' argument if the path doesn't already exist.
#'
#' The optional `export_gt_obj` argument enables the user to export the data frame as a .rds file alongside the graphical table.
#'
#' @return A message indicating that export was successful.
#'
#'
#' @author Sergej Ruff
#' @examples
#' path <- system.file("extdata", "virushunter.tsv", package = "Virusparies")
#' vh_file <- ImportVirusTable(path)
#'
#'
#' # using first 10 rows of SRA_run,num_hits,bestquery,ViralRefSeq_E and Identity col.
#' vh_file_part <- vh_file[c(1:10),c(1,7,9,10,11)]
#'
#' table <- VhgTabularRasa(vh_file_part,title = "first 10 rows of vh_file",subtitle =
#' "example for any table",names_ = c("Runs","Number of Contigs","Best Query Result",
#' "Reference E-Value","Reference Identity"))
#'
#'
#' \donttest{
#'
#' ExportVirusGt(gtable=table,filename="vh_parttable.docx",path=tempdir())
#'
#' }
#'
#' @seealso
#' VirusHunterGatherer is available here: \url{https://github.com/lauberlab/VirusHunterGatherer}.
#'
#'
#'
#' @import dplyr
#' @importFrom gt gtsave
#' @importFrom tools file_ext
#' @importFrom chromote find_chrome
#' @importFrom utils write.csv
#' @export
ExportVirusGt <- function(gtable,
                          filename="table.docx",
                          export_gt_obj = FALSE,
                          path = NULL,
                          create.dir = FALSE,
                          ...
                          ){



  fileext <- file_ext(filename)

  if ((fileext %in% c("png", "pdf")) && is.null(find_chrome())) {
    stop("Export of tables in PNG and PDF format requires Chrome installation.")
  }


  if (create.dir && !is.null(path)) {
    if (!file.exists(path)) {
      dir.create(path, recursive = TRUE)
      message("Directory created: ", path)
    }
  }

  if(export_gt_obj){

    # if(is.null(path)){
    #   path <- getwd()
    # }

    filename_without_extension <- tools::file_path_sans_ext(filename)
    new_filename <- paste0(filename_without_extension, ".rds")
    saveRDS(gtable, file = paste0(path,"/",new_filename))
  }



  # Attempt to save the table
  tryCatch({
    gtable %>% gtsave(filename = filename, path = path, ...)
    message("Table export completed successfully.")
  }, error = function(e) {
    warning("An error occurred during table export: ", e$message)
  })
}
