\name{MixedTypeEngine-class}
\alias{MixedTypeEngine-class}
\alias{MixedTypeEngine}
\alias{rand,MixedTypeEngine-method}
\alias{summary,MixedTypeEngine-method}
\docType{class}
\title{The "MixedTypeEngine" Class}
\description{
A MixedTypeEngine combines a ClinicalEngine (which defines the combinatorics
of hits and block hyperparameters that determine cluster identities and behavior),
a stored \code{\link{ClinicalNoiseModel}}, and cutpoints for generating mixed type
data generated by \code{\link{makeDataTypes}} into an object that can be used to
re-generate downstream datasets with shared parameters.
}
\usage{
MixedTypeEngine(ce, noise, cutpoints)
\S4method{rand}{MixedTypeEngine}(object, n, keepall = FALSE, \dots)
\S4method{summary}{MixedTypeEngine}(object, \dots)
}
\arguments{
  \item{ce}{Object of class \code{ClinicalEngine} (or a list; see Details).}
  \item{noise}{Object of class \code{NoiseModel}, preferably constructed
    using the function \code{ClinicalNoiseModel}. Alternatively, a list;
    see Details.}
  \item{cutpoints}{a list with the properties of the \code{cutpoints}
    element produced by the function \code{makeDataTypes}. Alternatively,
    a list; see Details. }
  \item{object}{object of class \code{MixedTypeEngine}}
  \item{n}{a non-negative integer}
  \item{keepall}{a logical value}
  \item{\dots}{additional arguments for generic functions.}
}
\section{Objects from the Class}{
  Objects can be created by a direct call to
  \link[methods]{new}, though using the constructor
  \code{MixedTypeEngine} is preferred.
}
\section{Methods}{
  \describe{
    \item{rand(object, n, keepall, \dots)}{Generates \eqn{nrow(Engine)*n} matrix
      representing clinical features of \code{n} patients following the
      underlying distribution, noise, and data discretization pattern
      captured in the object of \code{MixedTypeEngine}. If \code{keepall
      == TRUE}, it reurns a list containing a data frame named
    \code{clinical} and three data matrices called \code{raw},
    \code{noisy}, and \code{binned}. If \code{keepall == FALSE}, then
    noly the \code{clinical} and \code{binned} components are returned.}
  \describe{
    \item{summary(object, \dots)}{Prints a summary of the object.}
  }
}}
\details{
The MixedTypeEngine is a device for a parameter set used to generate 
a simulated set of clinical data which can be used to store these parameters
and to generate related datasets downstream. Building a MixedTypeEngine
requires many parameters. You can supply these parameters in mutliple
steps:
\enumerate{
  \item Construct a \code{ClinicalEngine}.
  \item Contruct a \code{ClinicalNoiseModel}.
  \item Use \code{rand}rand to generate a "raw" data set from the
    \code{ClinicalEngine}.
  \item Use \code{blur} to add noise to the raw data.
  \item Feed the noisy data into \code{makeDataTypes} to generate a
    mixed-type dataset, with cut points.
  \item Pass the \code{ClinicalEngine}, \code{ClinicalNoiseModel}, and
    cutpoints into the \code{MixedTypeEngine} constructor.
  }
The alternative method is to pass the parameters for Steps 1, 2, and 5
directly into the \code{MixedTypeEngine} directly, as lists, and it will
carry out steps 3-5 automatically. Note, however, that instead of
passing a \code{dataset} to be used by the \code{makeDataTypes} function,
you instead set the value of \code{N} to the desired number of patients
used during construction. Also, if you use the explicit steps, you
can save the intermediate data sets that are generated. If you simply
pass all of the parameters to the constructor, those intermediate data
sets are discarded, and you must generate a new data set using
\code{rand}.
}
\author{
  Kevin R. Coombes \email{krc@silicovore.com},
  Caitlin E. Coombes \email{caitlin.coombes@osumc.edu}
}
\seealso{
  \code{\linkS4class{Engine}}
  \code{\linkS4class{CancerModel}}
  \code{\linkS4class{CancerEngine}}
  \code{\link{ClinicalNoiseModel}}
  \code{\link{makeDataTypes}}
}
\examples{
## Generate a Clinical Engine of continuous data 
## with clusters generated from variation on the base CancerEngine
ce <- ClinicalEngine(20, 4, TRUE)
summary(ce)

## Generate an initial data set
set.seed(194718)
dset <- rand(ce, 300)
class(dset)
names(dset)
summary(dset$clinical)
dim(dset$data)

## Add noise before binning mixed type data
cnm <- ClinicalNoiseModel(nrow(ce@localenv$eng)) # default
noisy <- blur(cnm, dset$data)

## Set the data mixture
dt <- makeDataTypes(dset$data, 1/3, 1/3, 1/3, 0.3)
## Store the cutpoints
cp <- dt$cutpoints

## Use the pieces from above to create an MTE.
mte <- MixedTypeEngine(ce,
           noise = cnm,
           cutpoints = dt$cutpoints)

## Use the MTE rand method to generate
## multiple data sets with the same parameters
R <- rand(mte, 20)
summary(R)

S <- rand(mte, 20)
summary(S)
}
\keyword{classes}
\keyword{datagen}
