% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameters.R
\name{parameters}
\alias{parameters}
\alias{parameters.PWMs}
\alias{parameters.TLMoments}
\title{Converting TL-moments to distribution parameters}
\usage{
parameters(x, distr, ...)

\method{parameters}{PWMs}(x, distr, ...)

\method{parameters}{TLMoments}(x, distr, ...)
}
\arguments{
\item{x}{object returned by TLMoments (or PWMs, in which case TLMoments with no trimming is used).}

\item{distr}{character object defining the distribution. Supported types are
"gev", "gum", "gpd", and "ln3".}

\item{...}{additional arguments.}
}
\value{
numeric vector, matrix, list, or data.frame of parameter estimates with
class \code{parameters}. The object contains the following attributes: \itemize{
 \item \code{distribution}: a character indicating the used distribution
 \item \code{source}: a list with background information (used function, data, n, formula, trimmings; mainly for internal purposes)
}
The attributes are hidden in the print-function for a clearer presentation.
}
\description{
Converts TL-moments (or PWMs) to distribution parameters. By now, conversions for gev, gumbel, gpd, and ln3
are available. Important trimming options are calculated through known formulas (see references for
some of them), other options are calculated through a numerical optimization.
}
\section{Methods (by class)}{
\itemize{
\item \code{PWMs}: parameters for PWMs-object

\item \code{TLMoments}: parameters for TLMoments-object
}}

\examples{
xmat <- matrix(rgev(100, shape = .2), nc = 4)
xvec <- xmat[, 3]
xlist <- lapply(1L:ncol(xmat), function(i) xmat[, i])
xdat <- data.frame(
 station = rep(letters[1:2], each = 50),
 season = rep(c("S", "W"), 50),
 hq = as.vector(xmat)
)

# TLMoments-objects or PWMs-objects can be used. However, in case of PWMs
# simply the TLMoments(., leftrim = 0, rightrim = 0)-variant is used.

parameters(PWMs(xvec), "gev")
tlm <- TLMoments(xvec, leftrim = 0, rightrim = 0)
parameters(tlm, "gev")

tlm <- TLMoments(xmat, leftrim = 1, rightrim = 1)
parameters(tlm, "gum")

tlm <- TLMoments(xlist)
parameters(tlm, "gpd")

tlm <- TLMoments(xdat, hq ~ station, leftrim = 0, rightrim = 2)
parameters(tlm, "gev")

tlm <- TLMoments(xdat, hq ~ station + season, leftrim = 0, rightrim = 2)
parameters(tlm, "gev")


# If no explicit formula is implemented, it is tried to calculate
# parameters numerically. The attribute source$param.computation.method
# indicates if this is the case.

tlm <- TLMoments(rgum(200, loc = 5, scale = 2), leftrim = 1, rightrim = 4)
parameters(tlm, "gum")

tlm <- TLMoments(rgev(200, loc = 10, scale = 5, shape = .4), leftrim = 2, rightrim = 2)
parameters(tlm, "gev")

tlm <- TLMoments(rln3(200, loc = 3, scale = 1.5, shape = 2), leftrim = 0, rightrim = 1)
parameters(tlm, "ln3")

# Numerical calculation is A LOT slower:
\dontrun{
system.time(replicate(500,
  parameters(TLMoments(rgum(100, loc = 5, scale = 2), 1, 1), "gum")
))[3]
system.time(replicate(500,
  parameters(TLMoments(rgum(100, loc = 5, scale = 2), 1, 2), "gum")
))[3]
}

# Using magrittr
library(magrittr)

TLMoments(rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 0) \%>\%
 parameters("gpd")

TLMoments(rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 0) \%>\%
 parameters("gpd", u = 10)

TLMoments(rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 1) \%>\%
 parameters("gpd")

TLMoments(rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 2) \%>\%
 parameters("gpd")

}
\references{
Elamir, E. A. H. (2010). Optimal choices for trimming in trimmed L-moment method. Applied Mathematical Sciences, 4(58), 2881-2890.

Fischer, S., Fried, R., & Schumann, A. (2015). Examination for robustness of parametric estimators for flood statistics in the context of extraordinary extreme events. Hydrology and Earth System Sciences Discussions, 12, 8553-8576.

Hosking, J. R. (1990). L-moments: analysis and estimation of distributions using linear combinations of order statistics. Journal of the Royal Statistical Society. Series B (Methodological), 105-124.

Hosking, J. R. M. (2007). Some theory and practical uses of trimmed L-moments. Journal of Statistical Planning and Inference, 137(9), 3024-3039.
}
\seealso{
\code{\link{PWMs}}, \code{\link{TLMoments}}, \code{\link{quantiles}}, \code{\link{summary.parameters}}, \code{\link{as.parameters}}. Built-in distributions: \code{\link{pgev}}, \code{\link{pgum}}, \code{\link{pgpd}}, \code{\link{pln3}}.
}
