#' A function to calculate the interval estimate of the surrogate R-squared measure
#'
#' @description This function generates the interval measure of surrogate R-squared by bootstrap.
#' @param surr_rsq A object of class \code{"surr_rsq"} that is generated by the function \code{"surr_rsq"}.
#' It contains the following components: \code{surr_rsq}, \code{reduced_model}, \code{full_model}, and \code{data}.
#' @param alpha The significance level alpha. The confidence level is 1-alpha.
#' @param B The number of bootstrap replications.
#' @param ... Additional optional arguments.
#'
#' @return An list that contains the CI_lower, CI_upper.
#'
#' @importFrom progress progress_bar
#' @importFrom stats update lm nobs quantile
#' @importFrom scales percent
#'
#' @examples
#' data("RedWine")
#'
#' full_formula <- as.formula(quality ~ fixed.acidity + volatile.acidity + citric.acid
#' + residual.sugar + chlorides + free.sulfur.dioxide +
#' total.sulfur.dioxide + density + pH + sulphates + alcohol)
#'
#' fullmodel <- polr(formula = full_formula,data=RedWine, method  = "probit")
#'
#' select_model <- update(fullmodel, formula. = ". ~ . - fixed.acidity -
#' citric.acid - residual.sugar - density")
#'
#' surr_rsq_select <- surr_rsq(select_model, fullmodel, data = RedWine, avg.num = 30)
#'
#' # surr_rsq_ci(surr_rsq_select, alpha = 0.05, B = 1000) # Not run, it takes time.
#'
#' @export
#'
surr_rsq_ci <-
  function(surr_rsq,
           alpha = 0.05,
           B     = 1000,
           ...){
    # Save B+1 surrogate rsq, the first one is calculated from full data.
    B <- B + 1

    # Add progress bar --------------------------------------------------------
    pb <- progress_bar$new(
      format = "Replication = :letter [:bar] :percent :elapsed | eta: :eta",
      total = B,    # 300
      width = 80)
    progress_repNo <- c(1:B)  # token reported in progress bar


    # Estract components from surr_rsq object
    res_s <- surr_rsq[[1]]
    reduced_model <- surr_rsq[[2]]
    full_model <- surr_rsq[[3]]
    data <- surr_rsq[[4]]

    n <- nrow(data)
    # resultTable <- array(NA, dim = c(dim(data),1,B))
    # resultTable[,,1,1] <- res_s
    resultTable <- rep(NA, B)
    resultTable[1] <- res_s[[1]]

    for (j in 2:B) {
      BS_data <- data[sample(1:n, n, replace = T), ]
      try(
        resultTable[j] <- surr_rsq(reduced_model, full_model,data)[[1]]
      )
      while( is.na(resultTable[j])) {
        BS_data <- data[sample(1:n, n, replace = T), ]
        try(
          resultTable[j] <- surr_rsq(reduced_model, full_model, data)[[1]]
        )
      }

      # ProgressBar
      pb $tick(tokens = list(letter = progress_repNo[j]))
    }

    CI_lower <- quantile(x = resultTable[-1], probs = c(alpha/2))
    CI_lower <- round(CI_lower, 3)

    CI_upper <- quantile(x = resultTable[-1], probs = c(1 - alpha/2))
    CI_upper <- round(CI_upper, 3)

    return_list <- data.frame(Lower = c(percent(alpha/2, 0.01), CI_lower),
                              Upper = c(percent(1 - alpha/2, 0.01), CI_upper),
                              row.names = c("Percentile", "Confidence Interval"))


    return(return_list)
  }
