#' Plot directional spillover index
#'
#' @param data An object class \code{directional.spillover} resulting from applying \code{dynamic_spillover}
#' @param direction A character string indicating the direction of the spillover, see Diebold and Yilmaz (2012) 
#'
#' @return a ggplot2 object consisting of directional plots
#' @export
#' @import ggplot2 
#' @importFrom tidyr pivot_longer
#' @importFrom tidyr tibble
#' @importFrom dplyr select
#' @importFrom dplyr mutate
#' @importFrom dplyr "%>%"
#'
#' @references
#' Diebold, F. X. & Yilmaz, K.(2012). \emph{Better to Give than to Receive: Predictive Directional Measurement of Volatility Spillovers}. International Journal of Forecasting.
#' 
#' @examples
#'\donttest{ 
#' library(Spillover)
#' require(tidyr)
#' require(dplyr)
#' require(ggplot2)
#' 
#' data(dy2012)
#' dy_results <- dynamic_spillover(data=dy2012, width=200, remove.own = FALSE) 
#' 
#' pp_from <- plotdy(dy_results, direction = "from")
#'
#'pp_from +
#'  labs(title="Any title") +
#'  facet_wrap(~variables, ncol = 2)
#'
#'pp_to <- plotdy(dy_results, direction = "to")
#'
#'pp_net <- plotdy(dy_results, direction = "net")
#'
#'pp_netpairwise <- plotdy(dy_results, direction = "net_pairwise")
#'
#'pp_from_to_pairwise <- plotdy(dy_results, direction = "from_to_pairwise")
#'dy_results$from_to_pairwise %>% head
#'
#'pp_to +
#'  scale_x_date(breaks=date_breaks("6 months"),
#'               labels=date_format("%b %y"))+
#'  theme(axis.text.x = element_text(angle=45, hjust = 1))
#'
#' 
#' }
#' 
plotdy <- function(data, direction = c("from", "to", "net", "net_pairwise", "from_to_pairwise")){
  
  
  if (!inherits(data, "directional.spillover")) {
    stop("\nPlease provide an object of class 'directional.spillover', generated by 'dynamic.spillover()'.\n")
  }
  
  Date_1 <- value <- variables <- NULL
  
   
  plot_out <- switch(match.arg(direction), 
                     from = {
                       data[["from"]] %>% 
                         dplyr::mutate(Date_1=as.Date(as.character(data[["from"]][,1]))) %>% 
                         tibble %>% 
                         dplyr::select(-1) %>% 
                         pivot_longer(cols=-Date_1, names_to = "variables") %>% 
                         ggplot(aes(x=Date_1, y=value))+
                         geom_area() + 
                         facet_wrap(~variables, scales = "free_y")+
                         labs(title="Directional: From",
                              x="", y="")
                     },
                     
                     to = {
                       data[["to"]] %>% 
                         dplyr::mutate(Date_1=as.Date(as.character(data[["to"]][,1]))) %>% 
                         tibble %>% 
                         dplyr::select(-1) %>% 
                         pivot_longer(cols=-Date_1, names_to = "variables") %>% 
                         ggplot(aes(x=Date_1, y=value))+
                         geom_area() + 
                         facet_wrap(~variables, scales = "free_y")+
                         labs(title="Directional: To",
                              x="", y="")
                     },
                     
                     net = {
                       data[["net"]] %>% 
                         dplyr::mutate(Date_1=as.Date(as.character(data[["net"]][,1]))) %>% 
                         tibble %>% 
                         dplyr::select(-1) %>% 
                         pivot_longer(cols=-Date_1, names_to = "variables") %>% 
                         ggplot(aes(x=Date_1, y=value))+
                         geom_area() + 
                         facet_wrap(~variables, scales = "free_y")+
                         labs(title="Directional: Net",
                              x="", y="")
                     },
                     
                     net_pairwise = {
                       data[["net_pairwise"]] %>% 
                         dplyr::mutate(Date_1=as.Date(as.character(data[["net_pairwise"]][,1]))) %>% 
                         tibble %>% 
                         dplyr::select(-1) %>% 
                         pivot_longer(cols=-Date_1, names_to = "variables") %>% 
                         ggplot(aes(x=Date_1, y=value))+
                         geom_area() + 
                         facet_wrap(~variables, scales = "free_y")+
                         labs(title="Directional: Net pairwise",
                              x="", y="")
                     },
                     
                     from_to_pairwise = {
                       data[["from_to_pairwise"]] %>% 
                         dplyr::mutate(Date_1=as.Date(as.character(data[["from_to_pairwise"]][,1]))) %>% 
                         tibble %>% 
                         dplyr::select(-1) %>% 
                         dplyr::mutate(direction = as.factor(variables)) %>% 
                         ggplot(aes(x=Date_1, y=value))+
                         geom_area() + 
                         facet_wrap(~variables, scales = "free_y")+
                         labs(title="Directional: From/To pairwise",
                              x="", y="")
                     }
                     
  ) # switch ends
  
  
  
  print(plot_out)
  
} # plot.directional.spillover ends
