% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparse.tscgm.R
\name{sparse.tscgm}
\alias{sparse.tscgm}
\title{Sparse time series chain graphical models}
\usage{
sparse.tscgm(
  data = data,
  lam1 = NULL,
  lam2 = NULL,
  nlambda = NULL,
  model = c("ar1", "ar2"),
  penalty = c("scad", "lasso"),
  optimality = c("NULL", "bic", "bic_ext", "bic_mod", "aic", "gic"),
  control = list()
)
}
\arguments{
\item{data}{Longitudinal data format (matrix or data.frame).}

\item{lam1}{Numeric. Scalar or vector of tuning parameter values for the precision matrix penalty. 
If \code{NULL}, the program generates a sequence based on \code{nlambda}.}

\item{lam2}{Numeric. Scalar or vector of tuning parameter values for the autoregression matrix penalty. 
If \code{NULL}, a sequence is generated based on \code{nlambda}.}

\item{nlambda}{Integer. Number of tuning parameter values to generate if \code{lam1} or \code{lam2} are \code{NULL}. Default is 10.}

\item{model}{Character. Order of the vector autoregressive model. Choices: \code{"ar1"} or \code{"ar2"}.}

\item{penalty}{Character. Type of penalty to use. Choices: \code{"scad"} (default) or \code{"lasso"}.}

\item{optimality}{Character. Information criterion for model selection. Choices:
\code{"NULL"} (no selection), \code{"bic"}, \code{"bic_ext"}, \code{"bic_mod"}, \code{"aic"}, \code{"gic"}.}

\item{control}{List. Control parameters for the algorithm:
\describe{
  \item{maxit.out}{Maximum outer iterations (default 5).}
  \item{maxit.in}{Maximum inner iterations (default 50).}
  \item{tol.out}{Convergence tolerance (default 1e-4).}
  \item{silent}{TRUE/FALSE, whether to print progress messages (default TRUE).}
}}
}
\value{
A list containing:
  \describe{
    \item{theta}{Estimated precision matrix. Nonzero entries represent undirected edges.}
    \item{gamma}{Estimated autoregressive coefficient matrix. Nonzero entries represent directed edges.}
    \item{lam1.opt}{Optimal tuning parameter for the precision matrix.}
    \item{lam2.opt}{Optimal tuning parameter for the autoregression matrix.}
    \item{min.ic}{Minimum value of the selected information criterion.}
    \item{tun.ic}{Matrix of tuning parameters and corresponding information criterion values.}
    \item{lam1.seq}{Sequence of precision matrix tuning parameters.}
    \item{lam2.seq}{Sequence of autoregression matrix tuning parameters.}
    \item{s.theta}{Sparsity level of the precision matrix.}
    \item{s.gamma}{Sparsity level of the autoregression matrix.}
  }
}
\description{
Computes sparse vector autoregressive coefficient matrices of order 1 and 2 
and precision matrix estimates for time series chain graphical models using 
SCAD or LASSO penalties. In time series chain graphs, \code{directed} edges 
are identified by nonzero entries of the autoregressive coefficients matrix 
and \code{undirected} edges are identified by nonzero entries of the precision matrix.
}
\details{
For description of the objective functions and computational details, see Abegaz and Wit (2013).
}
\examples{
seed <- 321
datas <- sim.data(model="ar1", time=10, n.obs=10, n.var=5, seed=seed, prob0=0.35, network="random")
data.fit <- datas$data1
prec_true <- datas$theta
autoR_true <- datas$gamma

res.tscgm <- sparse.tscgm(data=data.fit, lam1=NULL, lam2=NULL, nlambda=NULL, 
  model="ar1", penalty="scad", optimality="bic_mod", 
  control=list(maxit.out=10, maxit.in=100))

# Estimated sparse precision and autoregression matrices
prec <- res.tscgm$theta
autoR <- res.tscgm$gamma

# Optimal tuning parameter values
lambda1.opt <- res.tscgm$lam1.opt
lambda2.opt <- res.tscgm$lam2.opt

# Sparsity levels
sparsity_theta <- res.tscgm$s.theta
sparsity_gamma <- res.tscgm$s.gamma

# Graphical visualization
oldpar <- par(mfrow=c(2,2))
plot.tscgm(datas, mat="precision", main="True precision matrix")
plot.tscgm(res.tscgm, mat="precision", main="Estimated precision matrix")
plot.tscgm(datas, mat="autoregression", main="True autoregression coef. matrix")
plot.tscgm(res.tscgm, mat="autoregression", main="Estimated autoregression coef. matrix")
par(oldpar)
}
\references{
Fentaw Abegaz and Ernst Wit (2013). Sparse time series chain graphical models 
for reconstructing genetic networks. \emph{Biostatistics}, 14(3), 586–599.

Rothman, Levina, and Zhu (2010). Sparse multivariate regression with 
covariance estimation. \emph{Journal of Computational and Graphical Statistics}, 19, 947–962.
}
