% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/distanceFromEachPoint.R
\name{pointDistance2}
\alias{pointDistance2}
\alias{pointDistance3}
\alias{distanceFromEachPoint}
\alias{.pointDistance}
\alias{pointDistance}
\title{Point distance with C++}
\usage{
pointDistance2(to, from)

pointDistance3(fromX, toX, fromY, toY, maxDistance)

distanceFromEachPoint(from, to = NULL, landscape, angles = NA_real_,
  maxDistance = NA_real_, cumulativeFn = NULL, distFn = function(dist)
  1/(1 + dist), cl, ...)

.pointDistance(from, to, angles = NA, maxDistance = NA_real_,
  otherFromCols = FALSE)
}
\arguments{
\item{to}{Numeric matrix with 2  or 3 columns (or optionally more, all of which
will be returned),
x and y, representing x and y coordinates of "to" cells, and
optional "id" which will be matched with "id" from \code{from}. Default is all cells.}

\item{from}{Numeric matrix with 2 or 3 or more columns. They must include x and y,
representing x and y coordinates of "from" cell. If there is a column
named "id", it will be "id" from \code{to}, i.e,. specific pair distances.
All other columns will be included in the return value of the function.}

\item{fromX}{Numeric vector of x coordinates for 'from' points}

\item{toX}{Numeric vector of x coordinates for 'to' points}

\item{fromY}{Numeric vector of y coordinates for 'from' points}

\item{toY}{Numeric vector of y coordinates for 'to' points}

\item{maxDistance}{Numeric scalar. The maximum distance cutoff for returned distances.}

\item{landscape}{RasterLayer. optional. This is only used if \code{to} is NULL, in which case
all cells are considered \code{to}.}

\item{angles}{Logical. If \code{TRUE}, then the function will return angles in radians,
as well as distances.}

\item{cumulativeFn}{A function that can be used to incrementally accumulate
values in each \code{to} location, as the function iterates
through each \code{from}. See Details.}

\item{distFn}{A function. This can be a function of \code{landscape},
\code{fromCells}, \code{toCells}, and \code{dist}.
If \code{cumulativeFn} is supplied, this will be used to convert
the distances to some other set of units that will be accumulated
by the \code{cumulativeFn}. See Details and examples.}

\item{cl}{A cluster object. Optional. This would generally be created using
parallel::makeCluster or equivalent. This is an alternative way, instead
of \code{beginCluster()}, to use parallelism for this function, allowing for
more control over cluster use.}

\item{...}{Any additional objects needed for \code{distFn}.}

\item{maxDistance}{Numeric in units of number of cells. The algorithm will build
the whole surface (from \code{from} to \code{to}), but will
remove all distances that are above this distance.
Using this will keep memory use down.}
}
\value{
\code{pointDistance2}: a matrix with all the \code{to} columns plus one extra
\code{dists} column.
\code{pointDistance3}: a matrix with \code{x} and \code{y} columns from
\code{to} plus one extra \code{dists} column.

A matrix with 3 columns, x0, y0 and dists.

A sorted matrix on \code{id} with same number of rows as \code{to},
        but with one extra column, \code{"dists"}, indicating the distance
        between \code{from} and \code{to}.
}
\description{
\code{pointDistance2} performs Pythagorean Theorem and \code{cbind}s all
columns from \code{to} to new dists column.
It is only defined for one point (\code{from}) to many (\code{to}) points.
\code{pointDistance3} performs Pythagorean Theorem and is to be used internally
within \code{distanceFromEachPoint} as an alternative to \code{.pointDistance},
where it does many points (\code{from}) to many (\code{to}) points, one
\code{from} point at a time. The results are then \code{rbind}ed internally.
It does not \code{cbind} extra columns from \code{to}.

This is a modification of \code{\link[raster]{distanceFromPoints}} for the case of many points.
This version can often be faster for a single point because it does not return a RasterLayer.
This is different than \code{\link[raster]{distanceFromPoints}} because it does not take the
minimum distance from the set of points to all cells.
Rather this returns the every pair-wise point distance.
As a result, this can be used for doing inverse distance weightings, seed rain,
cumulative effects of distance-based processes etc.
If memory limitation is an issue, maxDistance will keep memory use down,
but with the consequences that there will be a maximum distance returned.
This function has the potential to use a lot of memory if there are a lot of
\code{from} and \code{to} points.
}
\details{
A slightly faster way to calculate distances.

This function is cluster aware. If there is a cluster running, it will use it.
To start a cluster use \code{\link[raster]{beginCluster}}, with \code{N} being
the number of cores to use. See examples in \code{SpaDES.core::experiment}.



If the user requires an id (indicating the \code{from} cell for each \code{to} cell)
to be returned with the fuction, the user must add an identifier to the
\code{from} matrix, such as \code{"id"}.
Otherwise, the function will only return the coordinates and distances.

\code{distanceFromEachPoint} calls \code{.pointDistance}, which is not intended to be called
directly by the user.

This function has the potential to return a very large object, as it is doing pairwise
distances (and optionally directions) between from and to. If there are memory
limitations because there are many
\code{from} and many \code{to} points, then \code{cumulativeFn} and \code{distFn} can be used.
These two functions together will be used iteratively through the \code{from} points. The
\code{distFn} should be a transformation of distances to be used by the
\code{cumulativeFn} function. For example, if \code{distFn} is \code{1 / (1+x)},
the default, and \code{cumulativeFn} is \code{`+`}, then it will do a sum of
inverse distance weights.
See examples.
}
\examples{
library(raster)
library(quickPlot)

n <- 2
distRas <- raster(extent(0, 40, 0, 40), res = 1)
coords <- cbind(x = round(runif(n, xmin(distRas), xmax(distRas))) + 0.5,
                y = round(runif(n, xmin(distRas), xmax(distRas))) + 0.5)

# inverse distance weights
dists1 <- distanceFromEachPoint(coords, landscape = distRas)
indices <- cellFromXY(distRas, dists1[, c("x", "y")])
invDist <- tapply(dists1[, "dists"], indices, function(x) sum(1 / (1 + x))) # idw function
distRas[] <- as.vector(invDist)
if (interactive()) {
  clearPlot()
  Plot(distRas)
}

# With iterative summing via cumulativeFn to keep memory use low, with same result
dists1 <- distanceFromEachPoint(coords[, c("x", "y"), drop = FALSE],
                                landscape = distRas, cumulativeFn = `+`)
idwRaster <- raster(distRas)
idwRaster[] <- dists1[, "val"]
if (interactive()) Plot(idwRaster)

all(idwRaster[] == distRas[]) # TRUE

# A more complex example of cumulative inverse distance sums, weighted by the value
#  of the origin cell
ras <- raster(extent(0, 34, 0, 34), res = 1, val = 0)
rp <- randomPolygons(ras, numTypes = 10) ^ 2
n <- 15
cells <- sample(ncell(ras), n)
coords <- xyFromCell(ras, cells)
distFn <- function(landscape, fromCell, dist) landscape[fromCell] / (1 + dist)

#beginCluster(3) # can do parallel
dists1 <- distanceFromEachPoint(coords[, c("x", "y"), drop = FALSE],
                                landscape = rp, distFn = distFn, cumulativeFn = `+`)
#endCluster() # if beginCluster was run

idwRaster <- raster(ras)
idwRaster[] <- dists1[, "val"]
if (interactive()) {
  clearPlot()
  Plot(rp, idwRaster)
  sp1 <- SpatialPoints(coords)
  Plot(sp1, addTo = "rp")
  Plot(sp1, addTo = "idwRaster")
}
}
\seealso{
\code{\link{rings}}, \code{\link{cir}}, \code{\link[raster]{distanceFromPoints}},
which can all be made to do the same thing, under specific combinations of arguments.
But each has different primary use cases. Each is also faster under different conditions.
For instance, if \code{maxDistance} is relatively small compared to the number of cells
in the \code{landscape}, then \code{\link{cir}} will likely be faster. If a minimum
distance from all cells in the \code{landscape} to any cell in \code{from}, then
\code{distanceFromPoints} will be fastest. This function scales best when there are
many \code{to} points or all cells are used \code{to = NULL} (which is default).
}
\keyword{internal}
