###############################################################################################
#' Run SpaCCI Analysis
#'
#' This function runs the SpaCCI analysis to infer cell-cell interactions based on ligand-receptor pairs at global, regional, or local spatial scales. It integrates gene expression data, cell type proportions, and spatial coordinates with a user-specified ligand-receptor database.
#'
#' @param gene_spot_expression_dataframe A data frame of gene expression values, where row names are genes and column names are spot IDs.
#' @param spot_cell_proportion_dataframe A data frame of cell type proportions, where row names are spot IDs and column names are cell types.
#' @param spatial_coordinates_dataframe A data frame containing the spatial coordinates of the spots. The columns should include \code{"Spot_ID"}, \code{"imagerow"}, and \code{"imagecol"}. And the row names must be the names of \code{"Spot_ID"}.
#' @param LR_database_list A list containing ligand-receptor pairs and additional information, generated by functions using \code{`LR_database()`}.
#' @param specific_LR_pair Required if \code{analysis_scale} is \code{"local"}. A vector of ligand-receptor pair names for localized analysis. The names should match those in row names in \code{`LR_database_list$possible_LR_pairs_info`}.
#' @param analysis_scale A string specifying the scale of analysis: \code{"global"}, \code{"regional"}, or \code{"local"}.
#' @param region_spot_IDs Required if \code{analysis_scale} is "regional". A vector of spot IDs defining the region for regional analysis.
#' @param local_scale_proportion Optional. A numeric value ranging from 0 to 1, (0,1] specifying the proportion of spots to use for localized analysis. Default is \code{1}, meaning using \code{100}\% proportion of spots. One could modified if want to reducing computing time.
#' @param neighborhood_radius Optional. A numeric value specifying the radius of the neighborhood for localized analysis. Default is \code{2.5}, according to the 10X Visium ST data accounting for 200-250 \eqn{\mu}m interacting distance.
#'
#' @return A list containing:
#' \describe{
#'   \item{If \code{analysis_scale} is \code{"local"}:}{
#'     A list containing:
#'     \describe{
#'       \item{\code{dataframelist}}{A list of data frames, each representing the inferred interactions for a specific center spot. Each data frame includes information on ligand and receptor cell types, P-values, and adjusted P-values.}
#'       \item{\code{RegionIDs_matrix}}{A list of matrices, each containing the IDs of the spots within the specified radius of each center spot.}
#'     }
#'   }
#'   \item{If \code{analysis_scale} is \code{"regional"} or \code{"global"}:}{
#'     A list containing:
#'     \describe{
#'       \item{\code{pvalue_df}}{A data frame of inferred interactions within the specified region or globally, including information on ligand and receptor cell types, P-values, and adjusted P-values.}
#'     }
#'   }
#' }
#'

#'
#' @details The function supports three scales of analysis:
#' \describe{
#'   \item{\code{global}}{Analyzes interactions across the entire dataset.}
#'   \item{\code{regional}}{Analyzes interactions within a specified region of spots. Requires \code{region_spot_IDs}.}
#'   \item{\code{local}}{Analyzes localized hotspot of interactions for specific ligand-receptor pairs on the entire slides. Requires \code{specific_LR_pair}.}
#' }
#'
#' @examples
#' library(SpaCCI)
#' library(nnls)
#' #Load the example data
#' data(test_data)
#' gene_spot_df <- test_data$gene_spot_df
#' cell_prop_df <- test_data$cell_prop_df
#' spatial_coords_df <- test_data$spatial_coords_df
#'
#' result <- LR_database(species = "Human",
#'                       database_name = "CellChat",
#'                       gene_spot_expression_dataframe = gene_spot_df)
#'# global
#' result_global <- run_SpaCCI(gene_spot_expression_dataframe = gene_spot_df,
#'                             spot_cell_proportion_dataframe = cell_prop_df,
#'                             spatial_coordinates_dataframe = spatial_coords_df,
#'                             LR_database_list = result,
#'                             analysis_scale = "global")
#'
#' # local
#'result_local <- run_SpaCCI(gene_spot_expression_dataframe = gene_spot_df,
#'                           spot_cell_proportion_dataframe = cell_prop_df,
#'                           spatial_coordinates_dataframe = spatial_coords_df,
#'                           LR_database_list = result,
#'                           specific_LR_pair = "EDN2_EDNRA",
#'                           analysis_scale = "local",
#'                           local_scale_proportion = 0.1,
#'                           neighborhood_radius = 2.5)
#'
#' @export
#'

run_SpaCCI <- function(gene_spot_expression_dataframe,
                       spot_cell_proportion_dataframe,
                       spatial_coordinates_dataframe,
                       LR_database_list,
                       specific_LR_pair = NULL,
                       analysis_scale ,
                       region_spot_IDs = NULL,
                       local_scale_proportion = 1,
                       neighborhood_radius = 2.5) {

  # Validate analysis scale input
  if (!analysis_scale %in% c("global", "regional", "local")) {
    stop("Invalid analysis scale. Please specify 'global', 'regional', or 'local'.")
  }

  if (analysis_scale == "global") {
    result <- SpaCCI_global(gene_spot_df = gene_spot_expression_dataframe,
                            spot_cell_prop_df = spot_cell_proportion_dataframe,
                            matching_L_R_pairs = LR_database_list$possible_LR_pairs,
                            matching_L_R_pairs_info = LR_database_list$possible_LR_pairs_info)

  } else if (analysis_scale == "regional") {
    if (is.null(region_spot_IDs)) {
      stop("Please input a vector of the selected region spot IDs for regional analysis.")
    } else {
      result <- SpaCCI_region(gene_spot_df = gene_spot_expression_dataframe,
                              spot_cell_prop_df = spot_cell_proportion_dataframe,
                              region_spot_IDs = region_spot_IDs,
                              matching_L_R_pairs = LR_database_list$possible_LR_pairs,
                              matching_L_R_pairs_info = LR_database_list$possible_LR_pairs_info)
    }

  } else if (analysis_scale == "local") {
    if (is.null(specific_LR_pair)) {
      stop("For localized detection, please specify the Ligand-Receptor pair names as a vector.\nUsers can check the Ligand-Receptor pair names by looking into the rownames of LR_database_list$possible_LR_pairs_info.")
    } else {
      message(paste0("Now analyzing localized detection using ", local_scale_proportion * 100, "% of spots in the whole slide, with a radius of ", neighborhood_radius, "."))

      specific_LR <- LR_database_list$possible_LR_pairs[which(LR_database_list$possible_LR_pairs_info$interaction_name %in% specific_LR_pair),]
      specific_LR_info <- LR_database_list$possible_LR_pairs_info[which(LR_database_list$possible_LR_pairs_info$interaction_name %in% specific_LR_pair),]

      result <- SpaCCI_local(gene_spot_df = gene_spot_expression_dataframe,
                             spot_cell_prop_df = spot_cell_proportion_dataframe,
                             spatial_coord = spatial_coordinates_dataframe,
                             prop = local_scale_proportion,
                             radius = neighborhood_radius,
                             matching_L_R_pairs = specific_LR,
                             matching_L_R_pairs_info = specific_LR_info)
    }
  } else {
    stop("Invalid input provided.")
  }

  return(result)
}





