/*
 *  SimInf, a framework for stochastic disease spread simulations
 *  Copyright (C) 2015 - 2017  Stefan Engblom
 *  Copyright (C) 2015 - 2017  Stefan Widgren
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "SimInf.h"
#include "SimInf_forward_euler_linear_decay.h"
#include "SimInf_local_spread.h"

/* Offset in integer compartment state vector */
enum {S_1, I_1, S_2, I_2, S_3, I_3};

/* Offset in real-valued continuous state vector */
enum {PHI};

/* Offsets in node local data (ldata) to parameters in the model */
enum {END_T1, END_T2, END_T3, END_T4, NEIGHBOR};

/* Offsets in global data (gdata) to parameters in the model */
enum {UPSILON_1, UPSILON_2, UPSILON_3, GAMMA_1, GAMMA_2, GAMMA_3,
      ALPHA, BETA_T1, BETA_T2, BETA_T3, BETA_T4, COUPLING};

/**
 * In age category 1; susceptible to infected: S -> I
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity.
 */
double SISe3_sp_S_1_to_I_1(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[UPSILON_1] * v[PHI] * u[S_1];
}

/**
 * In age category 2; susceptible to infected: S -> I
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity.
 */
double SISe3_sp_S_2_to_I_2(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[UPSILON_2] * v[PHI] * u[S_2];
}

/**
 *  In age category 3; susceptible to infected: S -> I
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity.
 */
double SISe3_sp_S_3_to_I_3(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[UPSILON_3] * v[PHI] * u[S_3];
}

/**
 *  In age category 1; infected to susceptible: I -> S
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity.
 */
double SISe3_sp_I_1_to_S_1(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[GAMMA_1] * u[I_1];
}

/**
 * In age category 2; infected to susceptible: I -> S
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity.
 */
double SISe3_sp_I_2_to_S_2(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[GAMMA_2] * u[I_2];
}

/**
 * In age category 3; infected to susceptible: I -> S
 *
 * @param u The compartment state vector in node.
 * @param v The continuous state vector in node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param t Current time.
 * @return propensity
 */
double SISe3_sp_I_3_to_S_3(
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    double t)
{
    return gdata[GAMMA_3] * u[I_3];
}

/**
 * Update environmental infectious pressure phi
 *
 * Decay environmental infectious pressure phi, add contribution from
 * infected individuals and proximity coupling.
 * @param v_new The continuous state vector in the node after the post
 * time step
 * @param u The compartment state vector in the node.
 * @param v The current continuous state vector in the node.
 * @param ldata The local data vector for the node.
 * @param gdata The global data vector.
 * @param node The node.
 * @param t The current time.
 * @param rng The random number generator.
 * @return error code (<0), or 1 if node needs to update the
 * transition rates, or 0 when it doesn't need to update the
 * transition rates.
 */
int SISe3_sp_post_time_step(
    double *v_new,
    const int *u,
    const double *v,
    const double *ldata,
    const double *gdata,
    int node,
    double t)
{
    const int day = (int)t % 365;
    const double I_i = u[I_1] + u[I_2] + u[I_3];
    const double N_i = u[S_1] + u[S_2] + u[S_3] + I_i;
    const double phi = v[PHI];
    const int Nc = 6;

    /* Deterimine the pointer to the continuous state vector in the
     * first node. Use this to find phi at neighbours to the current
     * node. */
    const double *phi_0 = &v[-node];

    /* Deterimine the pointer to the compartment state vector in the
     * first node. Use this to find the number of individuals at
     * neighbours to the current node. */
    const int *u_0 = &u[-Nc*node];

    /* Time dependent beta in each of the four intervals of the
     * year. Forward Euler step. */
    v_new[PHI] = SimInf_forward_euler_linear_decay(
        phi, day,
        ldata[END_T1], ldata[END_T2], ldata[END_T3], ldata[END_T4],
        gdata[BETA_T1], gdata[BETA_T2], gdata[BETA_T3], gdata[BETA_T4]);

    /* Local spread among proximal nodes. */
    if (N_i > 0.0) {
        v_new[PHI] += gdata[ALPHA] * I_i / N_i +
            SimInf_local_spread(&ldata[NEIGHBOR], phi_0, u_0,
                                N_i, phi, Nc, gdata[COUPLING]);
    }

    if (!isfinite(v_new[PHI]))
        return SIMINF_ERR_V_IS_NOT_FINITE;
    if (v_new[PHI] < 0.0)
        return SIMINF_ERR_V_IS_NEGATIVE;
    return phi != v_new[PHI]; /* 1 if needs update */
}

/**
 * Run simulation with the SISe3_sp model
 *
 * @param model The SISe3_sp model.
 * @param threads Number of threads.
 * @param solver The numerical solver.
 * @return The simulated trajectory.
 */
SEXP SISe3_sp_run(SEXP model, SEXP threads, SEXP solver)
{
    TRFun tr_fun[] = {&SISe3_sp_S_1_to_I_1, &SISe3_sp_I_1_to_S_1,
                      &SISe3_sp_S_2_to_I_2, &SISe3_sp_I_2_to_S_2,
                      &SISe3_sp_S_3_to_I_3, &SISe3_sp_I_3_to_S_3};

    return SimInf_run(model, threads, solver, tr_fun, &SISe3_sp_post_time_step);
}
