## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2018  Stefan Engblom
## Copyright (C) 2015 - 2018  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

## C code: heading
C_heading <- function() {
    c(sprintf("/* Generated by SimInf (v%s) %s */",
            packageVersion("SimInf"),
            format(Sys.time(), "%Y-%m-%d %H:%M")),
      "")
}

## C code: include directives
C_include <- function() {
    c("#include <R_ext/Rdynload.h>",
      "#include \"SimInf.h\"",
      "")
}

## C code: transition rate functions.
C_trFun <- function(transitions) {
    lines <- character(0)
    for (i in seq_len(length(transitions))) {
        lines <- c(lines,
                   sprintf("double trFun%i(", i),
                   "    const int *u,",
                   "    const double *v,",
                   "    const double *ldata,",
                   "    const double *gdata,",
                   "    double t)",
                   "{",
                   sprintf("    return %s;", transitions[[i]]$propensity),
                   "}",
                   "")
    }
    lines
}

## C code: post time step function.
C_ptsFun <- function() {
    c("int ptsFun(",
      "    double *v_new,",
      "    const int *u,",
      "    const double *v,",
      "    const double *ldata,",
      "    const double *gdata,",
      "    int node,",
      "    double t)",
      "{",
      "    return 0;",
      "}",
      "")
}

## C code: SimInf model run function.
C_run <- function(transitions) {
    c("SEXP SimInf_model_run(SEXP model, SEXP threads, SEXP solver)",
      "{",
      sprintf("    TRFun tr_fun[] = {%s};",
              paste0("&trFun", seq_len(length(transitions)), collapse = ", ")),
      "    DL_FUNC SimInf_run = R_GetCCallable(\"SimInf\", \"SimInf_run\");",
      "    return SimInf_run(model, threads, solver, tr_fun, &ptsFun);",
      "}",
      "")
}

## C code: Generate C code for mparse.
C_code_mparse <- function(transitions, rates, compartments) {
    c(C_heading(),
      C_include(),
      C_trFun(transitions),
      C_ptsFun(),
      C_run(transitions))
}

## Split the propensity in order to separate preprocessor and
## punctuator tokens from identifiers, for example:
##
## > tokens(" bR * R ")
## [1] "bR" "*"  "R"
tokens <- function(propensity) {
    ## List of valid preprocessor operator or punctuator tokens.
    operators <- c("...", "<<=", ">>=", "!=", "%=", "##", "&&", "&=", "*=",
                   "++", "+=", "--", "-=", "->", "/=", "<<", "<=", "==",
                   ">=", ">>", "^=", "|=", "||", "!", "~", "%", "&", "(",
                   ")", "*", "+", ",", "-", ".", "/", ":", ";", "<", "=",
                   ">", "?", "[", "]", "^", "{", "|", "}", "#")

    ## Create a matrix (1 x 2) of the propensity, where the first
    ## column is the token and the second column indicates if the
    ## token is one of the operators (indicated with 'op').
    propensity <- cbind(token = propensity, type = "")

    ## Iterate over each operator and try to split each row in the
    ## propensity in smaller pieces.
    for (op in operators) {
        propensity <- lapply(seq_len(nrow(propensity)), function(i) {
            x <- propensity[i, , drop = FALSE]

            ## Is it a non-operator token that we could split?
            if (nchar(x[1, 2]) == 0) {
                m <- gregexpr(op, x[1, 1], fixed = TRUE)[[1]]
                if (m[1] != -1) {
                    ## The operator exists in the token. Split the
                    ## token in smaller pieces. The cut-points are
                    ## deterimined by the position and length of op
                    ## e.g. "A op B" -> "A", "op", "B".
                    x <- as.character(x[1, 1])
                    j <- 1
                    xx <- NULL
                    for (i in seq_len(length(m))) {
                        if (m[i] > j)
                            xx <- c(xx, substr(x, j, m[i] - 1))
                        j <- m[i] + attr(m, "match.length")[i]
                        xx <- c(xx, substr(x, m[i], j - 1))
                    }

                    ## Make sure last sub-string is copied.
                    if (j <= nchar(x))
                        xx <- c(xx, substr(x, j, nchar(x)))

                    ## Remove leading and trailing whitespace and drop
                    ## empty strings
                    xx <- gsub("(^\\s+)|(\\s+$)", "", xx)
                    xx <- xx[nchar(xx) > 0]

                    ## Create a 2-column matrix from all sub-strings
                    x <- cbind(token = xx, type = ifelse(xx == op, "op", ""))
                }
            }

            x
        })

        propensity <- do.call("rbind", propensity)
    }

    propensity[, 1]
}

## Rewrite propensity
##
## Rewrite the propensity by replacing all compartments by
## \code{u[compartments[j]]} where \code{j} is the numbering in
## compartments. On return, 'depends' contains all compartments upon
## which the propensity depends.
rewriteprop <- function(propensity, compartments, gdata) {
    orig_prop <- propensity
    propensity <- tokens(propensity)
    depends <- integer(length(compartments))

    ## Find compartments in propensity
    i <- match(propensity, compartments)
    propensity <- ifelse(is.na(i), propensity, sprintf("u[%i]", i-1L))
    i <- i[!is.na(i)]
    if (length(i))
        depends[i] <- 1

    ## Find gdata in propensity
    i <- match(propensity, gdata)
    propensity <- ifelse(is.na(i), propensity, sprintf("gdata[%i]", i-1L))

    list(orig_prop  = orig_prop,
         propensity = paste0(propensity, collapse = ""),
         depends    = depends)
}

## Generate labels from the parsed transitions
as_labels <- function(transitions) {
    sapply(transitions, function(x) {
        if (length(x$from)) {
            from <- paste0(x$from, collapse = " + ")
        } else {
            from <- "@"
        }

        if (length(x$dest)) {
            dest <- paste0(x$dest, collapse = " + ")
        } else {
            dest <- "@"
        }

        paste(from, "->", dest)
    })
}

##' Model parser to define new models to run in \code{SimInf}
##'
##' Describe your model in a logical way in R. \code{mparse} creates a
##' \code{\linkS4class{SimInf_model}} object with your model
##' definition that is ready to \code{\link{run}}.
##' @param transitions character vector containing transitions on the
##'     form \code{"X -> ... -> Y"}. The left (right) side is the
##'     initial (final) state and the propensity is written in between
##'     the \code{->}-signs. The special symbol \code{@} is reserved
##'     for the empty set. For example, \code{transitions =
##'     c("S -> k1*S*I -> I", "I -> k2*I -> R")} expresses a SIR
##'     model.
##' @param compartments contains the names of the involved
##'     compartments, for example, \code{compartments = c("S", "I",
##'     "R")}.
##' @param gdata named numeric vector with rate-constants for the
##'     model.
##' @param u0 A \code{data.frame} (or an object that can be coerced to
##'     a \code{data.frame} with \code{as.data.frame}) with the
##'     initial state in each node.
##' @template tspan-param
##' @param events A \code{data.frame} with the scheduled
##'     events. Default is \code{NULL} i.e. no scheduled events in the
##'     model.
##' @param E matrix to handle scheduled events, see
##'     \code{\linkS4class{SimInf_events}}. Default is \code{NULL}
##'     i.e. no scheduled events in the model.
##' @param N matrix to handle scheduled events, see
##'     \code{\linkS4class{SimInf_events}}. Default is \code{NULL}
##'     i.e. no scheduled events in the model.
##' @return a \code{\linkS4class{SimInf_model}} object
##' @export
##' @importFrom methods as
##' @importFrom methods new
##' @importFrom utils packageVersion
##' @template mparse-example
mparse <- function(transitions = NULL, compartments = NULL, gdata = NULL,
                   u0 = NULL, tspan = NULL, events = NULL, E = NULL, N = NULL)
{
    if (is.null(transitions))
        stop("'transitions' must be specified.")
    if(!is.character(transitions))
        stop("'transitions' must be specified in a character vector.")

    if (is.null(compartments))
        stop("'compartments' must be specified.")
    if(!is.character(compartments))
        stop("'compartments' must be specified in a character vector.")

    if (is.null(gdata))
        stop("'gdata' must be specified.")
    if (!is.numeric(gdata))
        stop("'gdata' must be a named numeric vector.")

    if (!all(identical(length(compartments), length(unique(compartments))),
             identical(length(names(gdata)), length(unique(names(gdata))))))
        stop("'compartments' and 'gdata' must consist of unique names.")

    reserved = c("v_new", "u", "v", "ldata", "gdata", "node", "t", "rng")
    if (length(intersect(compartments, reserved)))
        stop(paste("Invalid compartment names:",
                   paste0(intersect(compartments, reserved), collapse = ", ")))
    if (length(intersect(names(gdata), reserved)))
        stop(paste("Invalid gdata names:",
                   paste0(intersect(names(gdata), reserved), collapse = ", ")))

    transitions <- lapply(strsplit(transitions, "->"), function(x) {
        if (!identical(length(x), 3L))
            stop(paste0("Invalid transition: '", paste0(x, collapse = "->"), "'"))

        ## Remove spaces and the empty set
        from <- gsub(" ", "", gsub("@", "", x[1]))
        propensity <- gsub(" ", "", x[2])
        dest <- gsub(" ", "", gsub("@", "", x[3]))

        ## Split from and dest into 'compartment1 + compartment2 + ..'
        from <- unlist(strsplit(from, "+", fixed = TRUE))
        dest <- unlist(strsplit(dest, "+", fixed = TRUE))

        ## Assign each compartment into its number according to the
        ## ordering in compartments
        ifrom <- match(from, compartments)
        if (any(is.na(ifrom)))
            stop(sprintf("Unknown compartment: '%s'.", from[is.na(ifrom)]))
        idest <- match(dest, compartments)
        if (any(is.na(idest)))
            stop(sprintf("Unknown compartment: '%s'.", dest[is.na(idest)]))

        ## The corresponding column in the state change vector S is
        ## now known.
        S <- integer(length(compartments))
        S[ifrom] <- -1
        S[idest] <- 1

        propensity <- rewriteprop(propensity, compartments, names(gdata))

        list(from       = from,
             dest       = dest,
             orig_prop  = propensity$orig_prop,
             propensity = propensity$propensity,
             depends    = propensity$depends,
             S          = S)
    })

    S <- as(do.call("cbind", lapply(transitions, function(x) x$S)), "dgCMatrix")
    depends <- do.call("rbind", lapply(transitions, function(x) x$depends))
    G <- as(((depends %*% abs(S)) > 0) * 1, "dgCMatrix")

    colnames(G) <- as.character(seq_len(dim(G)[2]))
    rownames(G) <- as_labels(transitions)
    colnames(S) <- as.character(seq_len(dim(S)[2]))
    rownames(S) <- compartments

    ## Check u0
    if (!is.data.frame(u0))
        u0 <- as.data.frame(u0)
    if (!all(compartments %in% names(u0)))
        stop("Missing columns in u0")
    u0 <- u0[, compartments, drop = FALSE]

    SimInf_model(G      = G,
                 S      = S,
                 E      = E,
                 N      = N,
                 tspan  = tspan,
                 events = events,
                 ldata  = NULL,
                 gdata  = gdata,
                 u0     = u0,
                 v0     = NULL,
                 C_code = C_code_mparse(transitions, gdata, compartments))
}

##' Extract the C code from a \code{SimInf_model} object
##'
##' @param model The \code{SimInf_model} object to extract the C code
##'     from.
##' @param pkg Character vector. If the C could should be used in a
##'     package named \code{pkg}, the function modifies the C code to
##'     facilitate adding the code to the package. Default is to not
##'     use this argument and return the C code unmodified.
##' @return Character vector with C code for the model.
##' @export
##' @examples
##' ## Use the model parser to create a 'SimInf_model' object that
##' ## expresses an SIR model, where 'b' is the transmission rate and
##' ## 'g' is the recovery rate.
##' model <- mparse(transitions = c("S -> b*S*I/(S+I+R) -> I", "I -> g*I -> R"),
##'                 compartments = c("S", "I", "R"),
##'                 gdata = c(b = 0.16, g = 0.077),
##'                 u0 = data.frame(S = 99, I = 1, R = 0),
##'                 tspan = 1:10)
##'
##' ## View the C code.
##' C_code(model)
##'
##' ## Modify the C code for a package named "XYZ"
##' C_code(model, "XYZ")
C_code <- function(model, pkg = NULL)
{
    ## Check model argument
    if (missing(model))
        stop("Missing 'model' argument")
    if (!is(model, "SimInf_model"))
        stop("'model' argument is not a 'SimInf_model' object")

    if (is.null(pkg))
        return(model@C_code)

    pkg <- as.character(pkg)
    stopifnot(identical(length(pkg), 1L), nchar(pkg[1]) > 0)

    lines <- model@C_code

    lines <- c(
        lines[1:2],
        "#include <R_ext/Visibility.h>",
        lines[-(1:2)],
        "static const R_CallMethodDef callMethods[] =",
        "{",
        "    {\"SimInf_model_run\", (DL_FUNC)&SimInf_model_run, 3},",
        "    {NULL, NULL, 0}",
        "};",
        "",
        paste0("void attribute_visible R_init_", pkg, "(DllInfo *info)"),
        "{",
        "    R_registerRoutines(info, NULL, callMethods, NULL, NULL);",
        "    R_useDynamicSymbols(info, FALSE);",
        "    R_forceSymbols(info, TRUE);",
        "}",
        "")

    lines
}
