% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/estimation.r
\name{inference}
\alias{inference}
\alias{inference.del_sropt}
\alias{inference.sropt}
\title{Inference on noncentrality parameter of F-like statistic}
\usage{
inference(z.s, type = c("KRS", "MLE", "unbiased"))

\method{inference}{sropt}(z.s, type = c("KRS", "MLE", "unbiased"))

\method{inference}{del_sropt}(z.s, type = c("KRS", "MLE", "unbiased"))
}
\arguments{
\item{z.s}{an object of type \code{sropt}, or \code{del_sropt}}

\item{type}{the estimator type. one of \code{c("KRS", "MLE", "unbiased")}}
}
\value{
an estimate of the non-centrality parameter, which is
the maximal population Sharpe ratio.
}
\description{
Estimates the non-centrality parameter associated with an observed
statistic following an optimal Sharpe Ratio distribution.
}
\details{
Let \eqn{F}{F} be an observed statistic distributed as a non-central F with
\eqn{\nu_1}{df1}, \eqn{\nu_2}{df2} degrees of freedom and non-centrality
parameter \eqn{\delta^2}{delta^2}. Three methods are presented to
estimate the non-centrality parameter from the statistic:

\itemize{
\item an unbiased estimator, which, unfortunately, may be negative.
\item the Maximum Likelihood Estimator, which may be zero, but not
negative.
\item the estimator of Kubokawa, Roberts, and Shaleh (KRS), which
is a shrinkage estimator.
}

The sropt distribution is equivalent to an F distribution up to a
square root and some rescalings.

The non-centrality parameter of the sropt distribution is
the square root of that of the Hotelling, \emph{i.e.} has
units 'per square root time'. As such, the \code{'unbiased'}
type can be problematic!
}
\examples{
# generate some sropts
nfac <- 3
nyr <- 5
ope <- 253
# simulations with no covariance structure.
# under the null:
set.seed(as.integer(charToRaw("determinstic")))
Returns <- matrix(rnorm(ope*nyr*nfac,mean=0,sd=0.0125),ncol=nfac)
asro <- as.sropt(Returns,drag=0,ope=ope)
est1 <- inference(asro,type='unbiased')
est2 <- inference(asro,type='KRS')
est3 <- inference(asro,type='MLE')

# under the alternative:
Returns <- matrix(rnorm(ope*nyr*nfac,mean=0.0005,sd=0.0125),ncol=nfac)
asro <- as.sropt(Returns,drag=0,ope=ope)
est1 <- inference(asro,type='unbiased')
est2 <- inference(asro,type='KRS')
est3 <- inference(asro,type='MLE')

# sample many under the alternative, look at the estimator.
df1 <- 3
df2 <- 512
ope <- 253
zeta.s <- 1.25
rvs <- rsropt(128, df1, df2, zeta.s, ope)
roll.own <- sropt(z.s=rvs,df1,df2,drag=0,ope=ope)
est1 <- inference(roll.own,type='unbiased')
est2 <- inference(roll.own,type='KRS')
est3 <- inference(roll.own,type='MLE')

# for del_sropt:
nfac <- 5
nyr <- 10
ope <- 253
set.seed(as.integer(charToRaw("fix seed")))
Returns <- matrix(rnorm(ope*nyr*nfac,mean=0.0005,sd=0.0125),ncol=nfac)
# hedge out the first one:
G <- matrix(diag(nfac)[1,],nrow=1)
asro <- as.del_sropt(Returns,G,drag=0,ope=ope)
est1 <- inference(asro,type='unbiased')
est2 <- inference(asro,type='KRS')
est3 <- inference(asro,type='MLE')
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
\references{
Kubokawa, T., C. P. Robert, and A. K. Saleh. "Estimation of noncentrality parameters."
Canadian Journal of Statistics 21, no. 1 (1993): 45-57. \url{http://www.jstor.org/stable/3315657}

Spruill, M. C. "Computation of the maximum likelihood estimate of a noncentrality parameter."
Journal of multivariate analysis 18, no. 2 (1986): 216-224.
\url{http://www.sciencedirect.com/science/article/pii/0047259X86900709}
}
\seealso{
F-distribution functions, \code{\link{df}}.
}
\keyword{htest}

