% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smle_select.R
\name{smle_select}
\alias{smle_select}
\alias{smle_select.smle}
\alias{smle_select.sdata}
\alias{smle_select.default}
\title{Elaborative post-screening selection with SMLE}
\usage{
smle_select(object, ...)

\method{smle_select}{smle}(object, ...)

\method{smle_select}{sdata}(
  object,
  k_min = 1,
  k_max = NULL,
  subset = NULL,
  gamma_ebic = 0.5,
  vote = FALSE,
  criterion = "ebic",
  codingtype = NULL,
  gamma_seq = c(seq(0, 1, 0.2)),
  vote_threshold = NULL,
  parallel = FALSE,
  num_clusters = NULL,
  ...
)

\method{smle_select}{default}(object, X = NULL, family = "gaussian", ...)
}
\arguments{
\item{object}{Object of class \code{'smle'} or \code{'sdata'}. Users can also
input a response vector and a feature matrix. See examples}

\item{...}{Further arguments passed to or from other methods.}

\item{k_min}{The lower bound of candidate model sparsity. Default is 1.}

\item{k_max}{The upper bound of candidate model sparsity. Default is 
the number of columns in feature matrix.}

\item{subset}{An index vector indicating which features (columns of the
feature matrix) are to be selected.  Not applicable if a \code{'smle'}
object is the input.}

\item{gamma_ebic}{The EBIC tuning parameter, in \eqn{[0 , 1]}. Default is 0.5.}

\item{vote}{The logical flag for whether to perform the voting procedure.Only available when \code{criterion ="ebic"}.}

\item{criterion}{Selection criterion. One of "\code{ebic}","\code{bic}","\code{aic}". Default is "\code{ebic}".}

\item{codingtype}{Coding types for categorical features; for more details see \code{\link{SMLE}()} documentation.}

\item{gamma_seq}{The sequence of values for \code{gamma_ebic} when \code{vote =TRUE}.}

\item{vote_threshold}{A relative voting threshold in percentage. A feature is
considered to be important when it receives votes passing the threshold. Default is 0.6.}

\item{parallel}{A logical flag to use parallel computing to do voting selection.
Default is \code{FALSE}. See Details.}

\item{num_clusters}{The number of compute clusters to use when 
\code{parallel = TRUE}. The default will be 2 times cores detected.}

\item{X}{Input features matrix (when feature matrix input by users).}

\item{family}{Model assumption; see \code{\link{SMLE}()} documentation. Default is Gaussian linear.

When input is a \code{'smle'} or \code{'sdata'} object, the same
model will be used in the selection.}
}
\value{
\item{call}{The call that produced this object.}
\item{ID_selected}{A list of selected features.}
\item{coef_selected}{Fitted model coefficients based on the selected
features.}
\item{intercept}{Fitted model intercept based on the selected features.}
\item{criterion_value}{Values of selection criterion for the candidate models
with various sparsity.}
\item{X,Y}{Original data input.}
\item{ctg}{A logical flag whether the input feature matrix includes categorical features}
\item{ID_pool}{A vector contains all features selected during voting. }
\item{ID_voted}{Vector containing the features selected when \code{vote = T}.}
\item{CI}{Indices of categorical features when \code{ctg = TRUE}.}
\item{family,gamma_ebic,gamma_seq,criterion,vote,codyingtype,vote_threshold}{Return of arguments passed in the function call.}
}
\description{
The features retained after screening are still likely to contain some that 
are not related to the response. The function \code{\link{smle_select}()} is designed to 
further identify the relevant features using \code{\link{SMLE}()}.
Given a response and a set of \eqn{K} features, this function
first runs \code{\link{SMLE}(fast = TRUE)} to generate a series of sub-models with
sparsity k varying from \code{k_min} to \code{k_max}.
It then selects the best model from the series based on a selection criterion.

When criterion EBIC is used, users can choose to repeat the selection with
different values of the tuning parameter \eqn{\gamma}, and
conduct importance voting for each feature. When \code{vote = T}, this function 
fits all the models with \eqn{\gamma} specified in \code{gamma_seq} and features 
with frequency higher than \code{vote_threshold} will be selected in \code{ID_voted}.
}
\details{
This function accepts three types of input objects; 
1) \code{'smle'} object, as the output from \code{\link{SMLE}()}; 
2) \code{'sdata'} object, as the output from \code{\link{Gen_Data}()}; 
3) other response and feature matrix input by users.

Note that this function is mainly design to conduct an elaborative selection
after feature screening. We do not recommend using it directly for
ultra-high-dimensional data without screening.
}
\examples{

set.seed(1)
Data<-Gen_Data(correlation = "MA", family = "gaussian")
fit<-SMLE(Y = Data$Y, X = Data$X, k = 20, family = "gaussian")

fit_bic<-smle_select(fit, criterion = "bic")
summary(fit_bic)

fit_ebic<-smle_select(fit, criterion = "ebic", vote = TRUE)
summary(fit_ebic)
plot(fit_ebic)


}
\references{
Chen. J. and Chen. Z. (2012). "Extended BIC for small-n-large-p sparse GLM."
\emph{Statistica Sinica}, \bold{22}(2), 555-574.
}
