% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/halton.lattice.r
\name{halton.lattice}
\alias{halton.lattice}
\title{Halton lattice inside a rectangle}
\usage{
halton.lattice(box = matrix(c(0, 0, 1, 1), 2), N = 10000, J = NULL,
  eta = rep(1, nrow(box)), triangular = FALSE, bases = NULL)
}
\arguments{
\item{box}{A DX2 matrix containing coordinates of the box. 
One row per dimension. Column 1 is the minimum, column 2 is the maximum. 
\code{box[1,]} contains \code{c(min,max)} coordinates of the box in dimension 1
(horizontal).  \code{box[2,]} contains \code{c(min,max)} coordinates of 
the box in dimension 2 (vertical). Etc for higher dimensions.  
Default is the 2D unit box.}

\item{N}{Approximate number of points to place in the whole box.  If \code{J} 
is specified, it takes precedence.  If \code{J} is NULL, the 
algorithm attempts to place \code{N} points in the bounding box 
using Halton boxes that are as close to square as possible.  
\code{N} is not exact, but is a target.}

\item{J}{A DX1 vector of base powers which determines the size and shape 
of the Halton boxes. Elements of \code{J} less than or equal 
to 1 are re-set to 1. See additional description in help for 
\code{\link{hal.polygon}} function.}

\item{eta}{A DX1 vector of the number of points to add inside each Halton box.
e.g., if \code{eta} = \code{c(3,2)}, a small grid of 3 by 2 points is 
added inside each Halton box. \code{eta[1]} is for the
horizontal dimension, \code{eta[2]} is for the vertical dimension, etc for 
higher dimensions.}

\item{triangular}{boolean, if TRUE, construct a triangular grid. 
If FALSE, construct rectangular grid.  See help for \code{\link{hal.polygon}}.}

\item{bases}{A DX1 vector of Halton bases.  These must be co-prime.}
}
\value{
A data frame containing coordinates in the Halton lattice. 
Names of the coordinates are \code{dimnames(box)[1]}.  If \code{box} does not 
have dimnames, names of the coordinates are \code{c("d1", "d2", ...)} (d1 is 
horizontal, d2 is vertical, etc).

In addition, return has following attributes:
\itemize{
   \item \code{J}: the \code{J} vector used to construct the lattice. 
     This is either the input \code{J} or the computed \code{J} when 
     only \code{N} is specified. 
   \item \code{eta}: the \code{eta} vector used in the lattice.
   \item \code{bases}: Bases of the van der Corput sequences used in the lattice, 
     one per dimension.
   \item \code{triangular}: Whether the lattice is triangular or square.
   \item \code{hl.bbox}: The input \code{box}.  If \code{box} does not 
   have dimnames, this attribute will be assigned dimnames of 
   \code{list(c("d1","d2"),c("min","max"))}. 
}
}
\description{
Constructs a lattice of Halton boxes (a Halton lattice) inside 
a rectangular box.
}
\details{
This is designed to be called with the bounding box of a spatial 
object.  See examples.

\bold{Definition of Halton lattice}: A Halton lattice has the same number 
of points in every Halton box.  Halton boxes are the \code{bases[1]^J[1]} X 
\code{bases[2]^J[2]} matrix of rectangles over a square. Each Halton box 
contains \code{prod(eta)} points.
}
\examples{

# Lattice of 2^3*3^2 = 72 points in unit box
hl <- halton.lattice( J=c(3,2) )

# Plot
hl.J <- attr(hl,"J")
hl.b <- attr(hl,"bases")
hl.bb <- attr(hl,"hl.bbox") 

plot( hl.bb[1,], hl.bb[2,], type="n", pty="s")
points( hl[,1], hl[,2], pch=16, cex=.75, col="red")

for(d in 1:ncol(hl)){
  tmp2 <- hl.bb[d,1] + (0:(hl.b[d]^hl.J[d]))*(diff(hl.bb[d,]))/(hl.b[d]^hl.J[d])
  if( d == 1){
      abline(v=tmp2)
  } else{
      abline(h=tmp2)
  }
}

# Lattice of approx 1000 points over bounding box of spatial object
hl <- halton.lattice( bbox(HI.coast), N=1000 )
}
\author{
Trent McDonald
}
\seealso{
\code{\link{halton.lattice}}, \code{\link{hal.polygon}}
}

